<?php
/**
 * ============================================
 * WEBHOOK - RECEBE NOTIFICAÇÕES DO GATEWAY
 * ============================================
 * Processa callbacks de pagamento e envia para UTMify
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/utmify.php';

// Headers
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

// Log de entrada
$rawInput = file_get_contents('php://input');
$headers = getallheaders();

logCheckout('webhook.log', 'Webhook recebido', [
    'method' => $_SERVER['REQUEST_METHOD'],
    'headers' => $headers,
    'body' => $rawInput,
    'get' => $_GET,
    'post' => $_POST
]);

// Aceita GET para testes
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    echo json_encode(['status' => 'ok', 'message' => 'Webhook ativo']);
    exit;
}

try {
    // Decodifica payload
    $data = json_decode($rawInput, true);
    
    if (!$data) {
        // Tenta form-urlencoded
        parse_str($rawInput, $data);
    }
    
    if (empty($data)) {
        throw new Exception('Payload vazio');
    }
    
    logCheckout('webhook.log', 'Dados decodificados', $data);
    
    // Extrai ID da transação (vários formatos possíveis)
    $transactionId = $data['id'] 
        ?? $data['transaction_id'] 
        ?? $data['txid'] 
        ?? $data['externalReference']
        ?? $data['external_reference']
        ?? $data['data']['id']
        ?? $data['data']['transaction_id']
        ?? null;
    
    // Extrai status (vários formatos possíveis)
    $statusRaw = $data['status'] 
        ?? $data['payment_status'] 
        ?? $data['paymentStatus']
        ?? $data['data']['status']
        ?? null;
    
    if (!$transactionId) {
        logCheckout('webhook.log', 'ID de transação não encontrado no payload');
        echo json_encode(['status' => 'error', 'message' => 'ID não encontrado']);
        exit;
    }
    
    logCheckout('webhook.log', "Processando: ID=$transactionId, Status=$statusRaw");
    
    // Mapeia status
    $gateway = getGatewayConfig();
    $statusInterno = mapStatus($statusRaw, $gateway);
    
    logCheckout('webhook.log', "Status mapeado: $statusRaw -> $statusInterno");
    
    // Carrega dados salvos da transação
    $storageDir = defined('UTM_STORAGE_DIR') ? UTM_STORAGE_DIR : __DIR__ . '/utm_storage/';
    $transacaoFile = $storageDir . $transactionId . '.json';
    
    $dadosTransacao = [];
    if (file_exists($transacaoFile)) {
        $dadosTransacao = json_decode(file_get_contents($transacaoFile), true) ?: [];
        logCheckout('webhook.log', 'Dados da transação carregados', $dadosTransacao);
    } else {
        logCheckout('webhook.log', 'Arquivo de transação não encontrado: ' . $transacaoFile);
    }
    
    // ========================================
    // SE PAGO - ENVIA PARA UTMIFY
    // ========================================
    if ($statusInterno === 'paid') {
        logCheckout('webhook.log', '💰 Pagamento confirmado! Enviando para UTMify...');
        
        // Mescla dados do webhook com dados salvos
        $dadosParaUtmify = array_merge($dadosTransacao, [
            'transaction_id' => $transactionId,
            'id' => $transactionId,
            'status' => 'paid',
            'paid_at' => date('c'),
        ]);
        
        // Se tiver dados adicionais no webhook, adiciona
        if (isset($data['amount'])) {
            $dadosParaUtmify['valor'] = floatval($data['amount']);
            $dadosParaUtmify['valor_centavos'] = intval(floatval($data['amount']) * 100);
        }
        
        try {
            $utmifyResult = utmifyPago($dadosParaUtmify);
            logCheckout('webhook.log', 'UTMify PAGO enviado: ' . ($utmifyResult ? '✅ OK' : '❌ ERRO'));
        } catch (Exception $e) {
            logCheckout('webhook.log', 'Erro ao enviar para UTMify: ' . $e->getMessage());
        }
        
        // Atualiza arquivo local
        $dadosTransacao['status'] = 'paid';
        $dadosTransacao['paid_at'] = date('c');
        @file_put_contents($transacaoFile, json_encode($dadosTransacao, JSON_PRETTY_PRINT));
        
        // Também pode enviar para Facebook CAPI aqui se necessário
        // enviarParaFacebookCAPI('Purchase', $dadosParaUtmify);
    }
    
    // ========================================
    // SE REEMBOLSADO/CANCELADO
    // ========================================
    if (in_array($statusInterno, ['refunded', 'cancelled', 'chargeback'])) {
        logCheckout('webhook.log', "⚠️ Transação $statusInterno");
        
        try {
            utmifyAtualizarStatus($transactionId, $statusInterno);
        } catch (Exception $e) {
            logCheckout('webhook.log', 'Erro ao atualizar UTMify: ' . $e->getMessage());
        }
        
        // Atualiza arquivo local
        $dadosTransacao['status'] = $statusInterno;
        $dadosTransacao['updated_at'] = date('c');
        @file_put_contents($transacaoFile, json_encode($dadosTransacao, JSON_PRETTY_PRINT));
    }
    
    // Resposta de sucesso
    echo json_encode([
        'status' => 'ok',
        'transaction_id' => $transactionId,
        'status_interno' => $statusInterno,
        'utmify_enviado' => ($statusInterno === 'paid')
    ]);
    
} catch (Exception $e) {
    logCheckout('webhook.log', '❌ Erro no webhook: ' . $e->getMessage());
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
}