<?php
/**
 * ============================================
 * API PIX - GERAR COBRANÇA
 * ============================================
 * Gera PIX via gateway configurado
 * Integração opcional com UTMify
 */

// Habilita exibição de erros para debug (remover em produção)
error_reporting(E_ALL);
ini_set('display_errors', 0); // Não exibe erros na tela (quebra JSON)
ini_set('log_errors', 1);

require_once __DIR__ . '/config.php';

// Carrega UTMify APENAS se existir (não quebra se não existir)
$utmifyDisponivel = false;
if (file_exists(__DIR__ . '/utmify.php')) {
    require_once __DIR__ . '/utmify.php';
    $utmifyDisponivel = true;
}

setCorsHeaders();

// Apenas POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    die(json_encode(['error' => 'Método não permitido', 'success' => false]));
}

// Recebe dados
$rawInput = file_get_contents('php://input');
$input = json_decode($rawInput, true);

if (!$input) {
    http_response_code(400);
    die(json_encode(['error' => 'Dados inválidos ou JSON malformado', 'success' => false]));
}

logCheckout('pix.log', 'Requisição recebida', $input);

try {
    // Extrai dados do cliente
    $cliente = $input['cliente'] ?? $input;
    $nome = trim($cliente['nome'] ?? $input['nome'] ?? '');
    $email = trim($cliente['email'] ?? $input['email'] ?? 'cliente@email.com');
    $cpf = preg_replace('/\D/', '', $cliente['cpf'] ?? $input['cpf'] ?? '');
    $telefone = preg_replace('/\D/', '', $cliente['celular'] ?? $cliente['telefone'] ?? $input['telefone'] ?? '');
    $cotas = intval($cliente['cotas'] ?? $input['cotas'] ?? 1);
    
    // Valor em centavos
    $valorCentavos = intval($input['valor'] ?? 0);
    if ($valorCentavos <= 0) {
        throw new Exception('Valor inválido');
    }
    
    // Validações
    if (empty($nome) || strlen($nome) < 3) {
        throw new Exception('Nome inválido');
    }
    if (strlen($cpf) !== 11) {
        throw new Exception('CPF inválido');
    }
    if (strlen($telefone) < 10) {
        throw new Exception('Telefone inválido');
    }
    
    // Email padrão se vazio
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $email = 'cliente@email.com';
    }
    
    // Captura UTMs e tracking
    $tracking = $input['tracking'] ?? [];
    $utmData = [
        'utm_source' => $input['utm_source'] ?? $tracking['utm_source'] ?? '',
        'utm_medium' => $input['utm_medium'] ?? $tracking['utm_medium'] ?? '',
        'utm_campaign' => $input['utm_campaign'] ?? $tracking['utm_campaign'] ?? '',
        'utm_content' => $input['utm_content'] ?? $tracking['utm_content'] ?? '',
        'utm_term' => $input['utm_term'] ?? $tracking['utm_term'] ?? '',
        'sck' => $input['sck'] ?? $tracking['sck'] ?? '',
        'src' => $input['src'] ?? $tracking['src'] ?? '',
        'fbclid' => $input['fbclid'] ?? $tracking['fbclid'] ?? '',
        'fbc' => $input['fbc'] ?? $tracking['fbc'] ?? '',
        'fbp' => $input['fbp'] ?? $tracking['fbp'] ?? '',
    ];
    
    logCheckout('pix.log', 'Dados extraídos', [
        'nome' => $nome,
        'cpf' => substr($cpf, 0, 3) . '***',
        'telefone' => substr($telefone, 0, 4) . '***',
        'valor_centavos' => $valorCentavos,
        'cotas' => $cotas
    ]);
    
    // Configuração do gateway
    $gateway = getGatewayConfig();
    $gatewayName = getGatewayName();
    
    logCheckout('pix.log', "Usando gateway: $gatewayName");
    
    // Monta payload para o gateway
    $valorReais = number_format($valorCentavos / 100, 2, '.', '');
    
    // Webhook URL
    $webhookUrl = defined('WEBHOOK_URL') ? WEBHOOK_URL : (defined('SITE_URL') ? SITE_URL . '/api/webhook.php' : '');
    
    // Referência externa única
    $externalRef = 'pix_' . time() . '_' . substr(md5(uniqid()), 0, 8);
    
    // Payload padrão (funciona para Umbrella e TechByNet)
    $payloadGateway = [
        'amount' => $valorReais,
        'paymentMethod' => 'PIX',
        'customer' => [
            'name' => $nome,
            'email' => $email,
            'phone' => formatPhone($telefone),
            'document' => [
                'type' => 'CPF',
                'number' => $cpf
            ]
        ],
        'pix' => [
            'expiresInDays' => 1
        ],
        'items' => [
            [
                'title' => $input['produto'] ?? (defined('PRODUTO_NOME') ? PRODUTO_NOME : 'Produto'),
                'quantity' => $cotas,
                'unitPrice' => $valorReais,
                'tangible' => false
            ]
        ],
        'externalReference' => $externalRef,
        'metadata' => array_merge($utmData, [
            'cotas' => $cotas,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
        ])
    ];
    
    // Adiciona webhook se configurado
    if (!empty($webhookUrl)) {
        $payloadGateway['webhookUrl'] = $webhookUrl;
    }
    
    logCheckout('pix.log', 'Payload para gateway', $payloadGateway);
    
    // URL do endpoint
    $apiUrl = $gateway['api_url'];
    if (!empty($gateway['endpoints']['pix_create'])) {
        $apiUrl .= $gateway['endpoints']['pix_create'];
    }
    
    // Headers
    $headers = [
        'Content-Type: application/json',
        'Accept: application/json',
        $gateway['auth_header'] . ': ' . $gateway['api_key']
    ];
    
    if (!empty($gateway['user_agent'])) {
        $headers[] = 'User-Agent: ' . $gateway['user_agent'];
    }
    
    logCheckout('pix.log', 'Chamando gateway', ['url' => $apiUrl]);
    
    // Faz requisição ao gateway
    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payloadGateway),
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 60,
        CURLOPT_SSL_VERIFYPEER => true,
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    logCheckout('pix.log', "Resposta gateway [HTTP $httpCode]", [
        'response' => substr($response, 0, 2000),
        'error' => $curlError
    ]);
    
    if ($curlError) {
        throw new Exception("Erro de conexão com gateway: $curlError");
    }
    
    $data = json_decode($response, true);
    
    if ($httpCode >= 400) {
        $errorMsg = $data['message'] ?? $data['error'] ?? $data['msg'] ?? "Erro HTTP $httpCode";
        logCheckout('pix.log', 'Erro do gateway', ['http_code' => $httpCode, 'response' => $data]);
        throw new Exception("Gateway retornou erro: $errorMsg");
    }
    
    // Extrai dados da resposta usando config
    $transactionId = extractField($data, 'id', $gateway);
    $qrcode = extractField($data, 'qrcode', $gateway);
    $qrcodeBase64 = extractField($data, 'qrcode_base64', $gateway);
    
    // Fallback: tenta encontrar em locais comuns
    if (empty($transactionId)) {
        $transactionId = $data['id'] ?? $data['txid'] ?? $data['transaction_id'] ?? $data['data']['id'] ?? $externalRef;
    }
    if (empty($qrcode)) {
        $qrcode = $data['qrcode'] ?? $data['pix_code'] ?? $data['emv'] ?? $data['copiaecola'] ?? 
                  $data['data']['qrcode'] ?? $data['data']['pix_code'] ?? $data['pix']['qrcode'] ?? '';
    }
    if (empty($qrcodeBase64)) {
        $qrcodeBase64 = $data['qrcodeBase64'] ?? $data['qrcode_base64'] ?? $data['data']['qrcodeBase64'] ?? '';
    }
    
    if (empty($qrcode) && empty($qrcodeBase64)) {
        logCheckout('pix.log', 'QR Code não encontrado na resposta', $data);
        throw new Exception('QR Code não retornado pelo gateway');
    }
    
    logCheckout('pix.log', 'PIX gerado com sucesso', [
        'transaction_id' => $transactionId,
        'qrcode_length' => strlen($qrcode),
        'has_base64' => !empty($qrcodeBase64)
    ]);
    
    // Prepara dados para salvar
    $dadosTransacao = [
        'transaction_id' => $transactionId,
        'id' => $transactionId,
        'external_reference' => $externalRef,
        'gateway' => $gatewayName,
        'valor_centavos' => $valorCentavos,
        'valor' => $valorCentavos / 100,
        'cotas' => $cotas,
        'produto' => $input['produto'] ?? (defined('PRODUTO_NOME') ? PRODUTO_NOME : 'Produto'),
        'status' => 'pending',
        'created_at' => date('c'),
        'ip' => $_SERVER['REMOTE_ADDR'] ?? '',
        
        // Cliente
        'cliente' => [
            'nome' => $nome,
            'email' => $email,
            'cpf' => $cpf,
            'celular' => $telefone,
        ],
        'nome' => $nome,
        'email' => $email,
        'cpf' => $cpf,
        'telefone' => $telefone,
        
        // UTMs e tracking
        'tracking' => $utmData,
        'utm_source' => $utmData['utm_source'],
        'utm_medium' => $utmData['utm_medium'],
        'utm_campaign' => $utmData['utm_campaign'],
        'utm_content' => $utmData['utm_content'],
        'utm_term' => $utmData['utm_term'],
        'sck' => $utmData['sck'],
        'src' => $utmData['src'],
        'fbclid' => $utmData['fbclid'],
        'fbc' => $utmData['fbc'],
        'fbp' => $utmData['fbp'],
    ];
    
    // Salva transação localmente (para consulta posterior e UTMify)
    $storageDir = defined('UTM_STORAGE_DIR') ? UTM_STORAGE_DIR : __DIR__ . '/utm_storage/';
    if (!is_dir($storageDir)) {
        @mkdir($storageDir, 0755, true);
    }
    $saveResult = @file_put_contents(
        $storageDir . $transactionId . '.json',
        json_encode($dadosTransacao, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)
    );
    
    logCheckout('pix.log', 'Transação salva', [
        'file' => $storageDir . $transactionId . '.json',
        'save_result' => $saveResult !== false
    ]);
    
    // ========================================
    // ENVIA PARA UTMIFY - STATUS PENDENTE
    // ========================================
    if ($utmifyDisponivel && function_exists('utmifyPendente')) {
        try {
            $utmifyResult = utmifyPendente($dadosTransacao);
            logCheckout('pix.log', 'UTMify pendente: ' . ($utmifyResult ? '✅ OK' : '❌ ERRO'));
        } catch (Exception $e) {
            logCheckout('pix.log', 'Erro ao enviar para UTMify: ' . $e->getMessage());
            // Não quebra o fluxo se UTMify falhar
        }
    } else {
        logCheckout('pix.log', 'UTMify não disponível (arquivo utmify.php não encontrado)');
    }
    
    // Resposta de sucesso
    $resposta = [
        'success' => true,
        'id' => $transactionId,
        'txid' => $transactionId,
        'transaction_id' => $transactionId,
        'qrcode' => $qrcode,
        'copiaecola' => $qrcode,
        'qrcodeBase64' => $qrcodeBase64,
        'valor' => $valorCentavos / 100,
        'gateway' => $gatewayName,
    ];
    
    logCheckout('pix.log', '✅ PIX gerado com sucesso', ['id' => $transactionId]);
    
    header('Content-Type: application/json');
    echo json_encode($resposta);
    
} catch (Exception $e) {
    logCheckout('pix.log', '❌ Erro: ' . $e->getMessage(), [
        'trace' => $e->getTraceAsString()
    ]);
    
    http_response_code(400);
    header('Content-Type: application/json');
    echo json_encode([
        'error' => $e->getMessage(),
        'success' => false
    ]);
}