<?php
/**
 * ============================================
 * UTMIFY - INTEGRAÇÃO DE CONVERSÕES
 * ============================================
 * Envia pedidos para o UTMify com tracking completo
 *
 * IMPORTANTE: Cada produto tem seu próprio token UTMify
 * para garantir que vendas vão para o painel correto
 */

require_once __DIR__ . '/config.php';

class Utmify {

    private $token;
    private $produtoSlug;
    private $apiUrl = 'https://api.utmify.com.br/api-credentials/orders';

    /**
     * Construtor - aceita token específico ou slug do produto
     *
     * @param string|null $tokenOrSlug Token UTMify direto OU slug do produto
     */
    public function __construct($tokenOrSlug = null) {
        if ($tokenOrSlug) {
            // Verifica se é um slug de produto
            $produtoConfig = getProdutoConfig($tokenOrSlug);
            if ($produtoConfig) {
                $this->token = $produtoConfig['utmify_token'];
                $this->produtoSlug = $tokenOrSlug;
            } else {
                // Assume que é um token direto
                $this->token = $tokenOrSlug;
                $this->produtoSlug = null;
            }
        } else {
            // Fallback para token padrão (compatibilidade)
            $this->token = UTMIFY_TOKEN;
            $this->produtoSlug = PRODUTO_SLUG;
        }
    }

    /**
     * Define o token UTMify manualmente
     */
    public function setToken($token) {
        $this->token = $token;
        return $this;
    }

    /**
     * Define o produto pelo slug e carrega o token correspondente
     */
    public function setProduto($slug) {
        $config = getProdutoConfig($slug);
        if ($config) {
            $this->token = $config['utmify_token'];
            $this->produtoSlug = $slug;
            return true;
        }
        return false;
    }

    /**
     * Retorna o token atual
     */
    public function getToken() {
        return $this->token;
    }

    /**
     * Retorna o slug do produto atual
     */
    public function getProdutoSlug() {
        return $this->produtoSlug;
    }

    /**
     * Valida se o token está configurado
     */
    public function isConfigured() {
        return !empty($this->token);
    }

    /**
     * Envia pedido para UTMify
     */
    public function sendOrder($orderId, $status, $amountCents, $customer, $product, $utms) {
        // Valida se tem token configurado
        if (!$this->isConfigured()) {
            logCheckout('utmify.log', "ERRO: Token UTMify não configurado", [
                'order_id' => $orderId,
                'produto_slug' => $this->produtoSlug
            ]);
            return [
                'success' => false,
                'error' => 'Token UTMify não configurado para este produto'
            ];
        }

        $payload = [
            'orderId' => (string) $orderId,
            'platform' => 'SmartCheckout-upp',
            'paymentMethod' => 'pix',
            'status' => $status,
            'createdAt' => date('Y-m-d H:i:s'),
            'approvedDate' => ($status === 'paid') ? date('Y-m-d H:i:s') : null,
            'refundedAt' => ($status === 'refunded') ? date('Y-m-d H:i:s') : null,
            'customer' => [
                'name' => $customer['name'] ?? 'Cliente',
                'email' => $customer['email'] ?? '',
                'phone' => '+55' . preg_replace('/\D/', '', $customer['phone'] ?? ''),
                'document' => preg_replace('/\D/', '', $customer['document'] ?? ''),
                'country' => 'BR',
                'ip' => $customer['ip'] ?? $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1'
            ],
            'products' => [[
                'id' => $product['id'] ?? PRODUTO_ID,
                'name' => $product['name'] ?? PRODUTO_NOME,
                'planId' => null,
                'planName' => null,
                'quantity' => (int) ($product['quantity'] ?? 1),
                'priceInCents' => (int) $amountCents
            ]],
            'trackingParameters' => [
                'src' => (string) ($utms['src'] ?? ''),
                'sck' => (string) ($utms['sck'] ?? ''),
                'utm_source' => (string) ($utms['utm_source'] ?? ''),
                'utm_medium' => (string) ($utms['utm_medium'] ?? ''),
                'utm_campaign' => (string) ($utms['utm_campaign'] ?? ''),
                'utm_content' => (string) ($utms['utm_content'] ?? ''),
                'utm_term' => (string) ($utms['utm_term'] ?? '')
            ],
            'commission' => [
                'totalPriceInCents' => (int) $amountCents,
                'gatewayFeeInCents' => 0,
                'userCommissionInCents' => (int) $amountCents
            ],
            'isTest' => false
        ];

        $ch = curl_init($this->apiUrl);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'x-api-token: ' . $this->token
            ],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        $success = $httpCode >= 200 && $httpCode < 300;

        // Log com identificação do produto e token (mascarado)
        $tokenMasked = substr($this->token, 0, 8) . '...' . substr($this->token, -4);

        logCheckout('utmify.log', "Order $orderId - Status: $status", [
            'success' => $success,
            'produto_slug' => $this->produtoSlug,
            'token_used' => $tokenMasked,
            'http_code' => $httpCode,
            'response' => json_decode($response, true),
            'error' => $error
        ]);

        return [
            'success' => $success,
            'http_code' => $httpCode,
            'response' => json_decode($response, true),
            'produto_slug' => $this->produtoSlug
        ];
    }

    /**
     * Envia pedido pendente (PIX gerado)
     */
    public function sendPending($orderId, $amountCents, $customer, $product, $utms) {
        return $this->sendOrder($orderId, 'waiting_payment', $amountCents, $customer, $product, $utms);
    }

    /**
     * Envia pedido pago
     */
    public function sendPaid($orderId, $amountCents, $customer, $product, $utms) {
        return $this->sendOrder($orderId, 'paid', $amountCents, $customer, $product, $utms);
    }

    /**
     * Envia reembolso
     */
    public function sendRefunded($orderId, $amountCents, $customer, $product, $utms) {
        return $this->sendOrder($orderId, 'refunded', $amountCents, $customer, $product, $utms);
    }
}

/**
 * Função auxiliar para criar instância Utmify com token do produto
 *
 * @param string $produtoSlug Slug do produto
 * @return Utmify|null Instância configurada ou null se produto inválido
 */
function createUtmifyForProduct($produtoSlug) {
    $config = getProdutoConfig($produtoSlug);
    if (!$config) {
        logCheckout('utmify.log', "ERRO: Produto não encontrado", ['slug' => $produtoSlug]);
        return null;
    }

    return new Utmify($produtoSlug);
}
