<?php
/**
 * ============================================
 * TRACKING API - ENDPOINT UNIFICADO
 * ============================================
 * Recebe eventos do frontend e envia para Facebook CAPI
 * Com deduplicação automática (mesmo event_id do browser)
 * 
 * EVENTOS DISPONÍVEIS:
 * - PageView
 * - ViewContent
 * - AddToCart
 * - InitiateCheckout
 * - AddPaymentInfo
 * - Purchase
 * - customEvent (qualquer outro)
 * 
 * REMOVIDO: Lead (não adequado para rifa)
 */
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/facebook_capi.php';

// Proteção anti-clone
validarOrigem();
setCorsHeaders();

$input = json_decode(file_get_contents('php://input'), true);

if (!$input || empty($input['event'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Evento não especificado']);
    exit;
}

$event = $input['event'];
$eventId = $input['event_id'] ?? null;
$sourceUrl = $input['source_url'] ?? '';
$value = floatval($input['value'] ?? 0);
$numItems = intval($input['num_items'] ?? 1);

// Captura fbc do cookie se não veio no request
$fbc = $input['fbc'] ?? null;
$fbp = $input['fbp'] ?? null;

// Tenta pegar dos cookies do servidor se não veio do frontend
if (empty($fbc) && isset($_COOKIE['_fbc'])) {
    $fbc = $_COOKIE['_fbc'];
}
if (empty($fbp) && isset($_COOKIE['_fbp'])) {
    $fbp = $_COOKIE['_fbp'];
}

// Valida formato do fbc (evita erro de creationTime)
$fbcCreationTimeSec = null;
if (!empty($fbc)) {
    $fbcParts = explode('.', $fbc);
    if (count($fbcParts) >= 4 && strlen($fbcParts[2]) > 0) {
        $fbcTimeRaw = $fbcParts[2];
        $nowMs = round(microtime(true) * 1000);
        $nowSec = time();
        
        if (strlen($fbcTimeRaw) <= 10) {
            $fbcTimeMs = intval($fbcTimeRaw) * 1000;
        } else {
            $fbcTimeMs = intval($fbcTimeRaw);
        }
        
        $fbcCreationTimeSec = intval($fbcTimeMs / 1000);
        
        $maxAgeSec = 7 * 24 * 60 * 60;
        $tolerance = 60;
        
        if ($fbcCreationTimeSec > ($nowSec + $tolerance) || $fbcCreationTimeSec < ($nowSec - $maxAgeSec)) {
            $fbc = null;
            $fbcCreationTimeSec = null;
        }
    } else {
        $fbc = null;
    }
}

// Gera fbp se ainda não existir
if (empty($fbp)) {
    $fbp = 'fb.1.' . round(microtime(true) * 1000) . '.' . mt_rand(1000000000, 9999999999);
}

// Captura IP corretamente
$clientIp = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['HTTP_X_REAL_IP'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
if (strpos($clientIp, ',') !== false) {
    $clientIp = trim(explode(',', $clientIp)[0]);
}

// Dados do usuário para o Facebook
$userData = [
    'email' => $input['email'] ?? null,
    'phone' => $input['phone'] ?? null,
    'name' => $input['name'] ?? null,
    'document' => $input['document'] ?? null,
    'cidade' => $input['cidade'] ?? null,
    'estado' => $input['estado'] ?? null,
    'cep' => $input['cep'] ?? null,
    'fbc' => $fbc,
    'fbp' => $fbp,
    'user_agent' => $input['user_agent'] ?? $_SERVER['HTTP_USER_AGENT'] ?? '',
    'ip' => $clientIp,
    'content_name' => $input['content_name'] ?? PRODUTO_NOME,
    'content_id' => $input['content_id'] ?? PRODUTO_ID
];

// Log detalhado para debug
logCheckout('tracking.log', "Evento: $event", [
    'event_id' => $eventId,
    'value' => $value,
    'fbc' => $fbc ? 'válido' : 'inválido/ausente',
    'fbc_creation_time' => $fbcCreationTimeSec ?? 'N/A',
    'event_time' => time(),
    'fbp' => $fbp ? 'presente' : 'ausente',
    'ip' => $clientIp ? 'presente' : 'ausente',
    'email' => !empty($input['email']) ? 'presente' : 'ausente',
    'phone' => !empty($input['phone']) ? 'presente' : 'ausente'
]);

$fb = new FacebookCAPI();
$result = ['success' => false];

switch ($event) {
    case 'PageView':
        $result = $fb->pageView($userData, $sourceUrl, $eventId);
        break;
    
    case 'ViewContent':
        $result = $fb->viewContent($userData, $value, $sourceUrl, $eventId);
        break;
    
    case 'AddToCart':
        $result = $fb->addToCart($userData, $value, $numItems, $sourceUrl, $eventId);
        break;
        
    case 'InitiateCheckout':
        $result = $fb->initiateCheckout($userData, $value, $numItems, $sourceUrl, $eventId);
        break;
        
    case 'AddPaymentInfo':
        $result = $fb->addPaymentInfo($userData, $value, $sourceUrl, $eventId);
        break;
        
    case 'Purchase':
        $orderId = $input['order_id'] ?? uniqid('order_');
        $result = $fb->purchase($userData, $value, $orderId, $numItems, $sourceUrl, $eventId);
        break;
        
    default:
        // Qualquer outro evento customizado
        $customData = [
            'currency' => 'BRL',
            'value' => $value,
            'content_name' => $userData['content_name']
        ];
        $result = $fb->customEvent($event, $userData, $customData, $sourceUrl, $eventId);
}

echo json_encode($result);