<?php
/**
 * ============================================
 * INTEGRAÇÃO UTMIFY - API
 * ============================================
 * Envia eventos de conversão para o UTMify
 */

require_once __DIR__ . '/config.php';

/**
 * Envia evento para UTMify
 * @param string $status - 'pending' ou 'paid'
 * @param array $data - Dados da transação
 */
function enviarParaUtmify($status, $data) {
    // Token do UTMify
    $token = defined('UTMIFY_TOKEN') ? UTMIFY_TOKEN : '';
    
    if (empty($token)) {
        logCheckout('utmify.log', 'Token UTMify não configurado');
        return false;
    }
    
    // URL da API UTMify
    $apiUrl = 'https://api.utmify.com.br/api/v1/orders';
    
    // Monta o payload no formato que o UTMify espera
    $payload = [
        // Identificação da transação
        'orderId' => $data['transaction_id'] ?? $data['id'] ?? uniqid('order_'),
        'platform' => 'custom', // ou 'hotmart', 'monetizze', etc
        
        // Status: 'waiting_payment', 'paid', 'refunded', 'canceled', 'chargeback'
        'paymentStatus' => ($status === 'paid') ? 'paid' : 'waiting_payment',
        
        // Valor em REAIS (não centavos)
        'orderValue' => isset($data['valor_centavos']) ? ($data['valor_centavos'] / 100) : ($data['valor'] ?? 0),
        
        // Dados do cliente
        'customer' => [
            'name' => $data['cliente']['nome'] ?? $data['nome'] ?? '',
            'email' => $data['cliente']['email'] ?? $data['email'] ?? '',
            'phone' => $data['cliente']['celular'] ?? $data['telefone'] ?? '',
            'document' => $data['cliente']['cpf'] ?? $data['cpf'] ?? '',
        ],
        
        // UTMs - IMPORTANTE: precisa vir do frontend
        'utm_source' => $data['utm_source'] ?? $data['tracking']['utm_source'] ?? '',
        'utm_medium' => $data['utm_medium'] ?? $data['tracking']['utm_medium'] ?? '',
        'utm_campaign' => $data['utm_campaign'] ?? $data['tracking']['utm_campaign'] ?? '',
        'utm_content' => $data['utm_content'] ?? $data['tracking']['utm_content'] ?? '',
        'utm_term' => $data['utm_term'] ?? $data['tracking']['utm_term'] ?? '',
        
        // Parâmetros do Facebook
        'fbc' => $data['fbc'] ?? $data['tracking']['fbc'] ?? '',
        'fbp' => $data['fbp'] ?? $data['tracking']['fbp'] ?? '',
        'fbclid' => $data['fbclid'] ?? $data['tracking']['fbclid'] ?? '',
        
        // Parâmetros extras
        'sck' => $data['sck'] ?? $data['tracking']['sck'] ?? '',
        'src' => $data['src'] ?? $data['tracking']['src'] ?? '',
        
        // IP do cliente
        'ip' => $data['ip'] ?? $_SERVER['REMOTE_ADDR'] ?? '',
        
        // Produto
        'products' => [
            [
                'name' => $data['produto'] ?? 'Produto',
                'quantity' => $data['cotas'] ?? 1,
                'price' => isset($data['valor_centavos']) ? ($data['valor_centavos'] / 100) : ($data['valor'] ?? 0),
            ]
        ],
        
        // Timestamp
        'createdAt' => $data['created_at'] ?? date('c'),
    ];
    
    // Remove campos vazios
    $payload = array_filter($payload, function($v) {
        if (is_array($v)) {
            return !empty(array_filter($v));
        }
        return $v !== '' && $v !== null;
    });
    
    // Log do envio
    logCheckout('utmify.log', "Enviando para UTMify [$status]", $payload);
    
    // Faz a requisição
    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'Accept: application/json',
            'x-api-token: ' . $token,
        ],
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => true,
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    // Log da resposta
    logCheckout('utmify.log', "Resposta UTMify [HTTP $httpCode]", [
        'response' => $response,
        'error' => $error
    ]);
    
    if ($httpCode >= 200 && $httpCode < 300) {
        logCheckout('utmify.log', "✅ Evento '$status' enviado com sucesso");
        return true;
    } else {
        logCheckout('utmify.log', "❌ Erro ao enviar evento '$status': HTTP $httpCode");
        return false;
    }
}

/**
 * Envia evento de pedido pendente (PIX gerado)
 */
function utmifyPendente($data) {
    return enviarParaUtmify('pending', $data);
}

/**
 * Envia evento de pedido pago
 */
function utmifyPago($data) {
    return enviarParaUtmify('paid', $data);
}

/**
 * Atualiza status de um pedido existente
 */
function utmifyAtualizarStatus($orderId, $novoStatus, $dadosAdicionais = []) {
    $token = defined('UTMIFY_TOKEN') ? UTMIFY_TOKEN : '';
    
    if (empty($token)) {
        logCheckout('utmify.log', 'Token UTMify não configurado');
        return false;
    }
    
    // URL para atualizar pedido
    $apiUrl = 'https://api.utmify.com.br/api/v1/orders/' . urlencode($orderId);
    
    $payload = array_merge([
        'paymentStatus' => $novoStatus, // 'paid', 'refunded', 'canceled', etc
    ], $dadosAdicionais);
    
    logCheckout('utmify.log', "Atualizando status no UTMify: $orderId -> $novoStatus", $payload);
    
    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_CUSTOMREQUEST => 'PUT',
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'Accept: application/json',
            'x-api-token: ' . $token,
        ],
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    logCheckout('utmify.log', "Resposta atualização UTMify [HTTP $httpCode]", ['response' => $response]);
    
    return $httpCode >= 200 && $httpCode < 300;
}