<?php
/**
 * ============================================
 * WEBHOOK - CONFIRMAÇÃO DE PAGAMENTO
 * ============================================
 * Recebe webhooks de múltiplos gateways (Umbrella, TechByNet)
 * e dispara eventos para UTMify
 *
 * IMPORTANTE: Cada produto usa seu próprio token UTMify
 * Vendas só são enviadas para o painel do produto correto
 */

require_once __DIR__ . '/config.php';
@include_once __DIR__ . '/utmify.php';

header('Content-Type: application/json');

logCheckout('webhook.log', '=== WEBHOOK RECEBIDO ===');

// Recebe payload
$payload = file_get_contents('php://input');
$data = json_decode($payload, true);

if (!$data) {
    logCheckout('webhook.log', 'Payload inválido', ['raw' => substr($payload, 0, 500)]);
    http_response_code(400);
    echo json_encode(['error' => 'Payload inválido']);
    exit;
}

logCheckout('webhook.log', 'Dados recebidos', $data);

// ============================================
// DETECTA O GATEWAY PELO PAYLOAD
// ============================================
$detectedGateway = detectGateway($data);
logCheckout('webhook.log', "Gateway detectado: $detectedGateway");

// Obtém configuração do gateway
global $GATEWAYS;
$gatewayConfig = $GATEWAYS[$detectedGateway] ?? $GATEWAYS[GATEWAY_ATIVO];

// ============================================
// EXTRAI DADOS CONFORME O GATEWAY
// ============================================
$webhookData = extractWebhookData($data, $detectedGateway);

$orderId = $webhookData['order_id'];
$status = $webhookData['status'];
$amountCents = $webhookData['amount'];
$amount = $amountCents / 100;
$customer = $webhookData['customer'];

// Mapeia status para interno
$statusNorm = mapStatus($status, $gatewayConfig);

logCheckout('webhook.log', 'Dados extraídos', [
    'gateway' => $detectedGateway,
    'order_id' => $orderId,
    'status_api' => $status,
    'status_norm' => $statusNorm,
    'amount' => $amount
]);

// ============================================
// IDENTIFICA O PRODUTO E SEU TOKEN UTMIFY
// ============================================
$produtoSlug = null;
$produtoConfig = null;
$savedData = null;
$jsonFile = UTM_STORAGE_DIR . $orderId . '.json';

// Tenta carregar dados salvos do pedido
if (file_exists($jsonFile)) {
    $savedData = json_decode(file_get_contents($jsonFile), true);

    // Identifica produto pelos dados salvos
    $produtoSlugSalvo = $savedData['produto_slug'] ?? '';
    $produtoNomeSalvo = $savedData['produto'] ?? '';

    // Primeiro tenta pelo slug salvo
    if (!empty($produtoSlugSalvo)) {
        $produtoSlug = identificarProduto($produtoNomeSalvo, $produtoSlugSalvo);
    }

    // Se não encontrou, tenta pelo nome
    if (!$produtoSlug && !empty($produtoNomeSalvo)) {
        $produtoSlug = identificarProduto($produtoNomeSalvo);
    }

    if ($produtoSlug) {
        $produtoConfig = getProdutoConfig($produtoSlug);
        logCheckout('webhook.log', 'Produto identificado pelo JSON salvo', [
            'produto_slug' => $produtoSlug,
            'produto_nome' => $produtoConfig['nome'] ?? '',
            'token_masked' => substr($produtoConfig['utmify_token'] ?? '', 0, 8) . '...'
        ]);
    }
}

// Se não encontrou pelo JSON, tenta identificar pelos produtos do webhook
if (!$produtoSlug && !empty($webhookData['products'])) {
    foreach ($webhookData['products'] as $product) {
        $productTitle = $product['title'] ?? $product['name'] ?? '';
        $productId = $product['id'] ?? '';

        $produtoSlug = identificarProduto($productTitle);
        if (!$produtoSlug && $productId) {
            $produtoSlug = identificarProduto($productId);
        }

        if ($produtoSlug) {
            $produtoConfig = getProdutoConfig($produtoSlug);
            logCheckout('webhook.log', 'Produto identificado pelo webhook', [
                'produto_slug' => $produtoSlug,
                'produto_nome' => $produtoConfig['nome'] ?? '',
                'product_title' => $productTitle
            ]);
            break;
        }
    }
}

// ============================================
// VALIDAÇÃO: PRODUTO DEVE ESTAR CONFIGURADO
// ============================================
if (!$produtoSlug || !$produtoConfig) {
    logCheckout('webhook.log', 'IGNORADO - Produto não identificado ou não configurado', [
        'order_id' => $orderId,
        'saved_slug' => $savedData['produto_slug'] ?? 'N/A',
        'saved_nome' => $savedData['produto'] ?? 'N/A',
        'webhook_products' => $webhookData['products'] ?? []
    ]);

    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Produto não configurado - webhook ignorado',
        'order_id' => $orderId
    ]);
    exit;
}

// ============================================
// VALIDAÇÃO: TOKEN UTMIFY DEVE EXISTIR
// ============================================
$utmifyToken = $produtoConfig['utmify_token'] ?? null;

if (empty($utmifyToken)) {
    logCheckout('webhook.log', 'IGNORADO - Token UTMify não configurado para o produto', [
        'order_id' => $orderId,
        'produto_slug' => $produtoSlug,
        'produto_nome' => $produtoConfig['nome'] ?? ''
    ]);

    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Token UTMify não configurado para este produto',
        'order_id' => $orderId,
        'produto' => $produtoSlug
    ]);
    exit;
}

logCheckout('webhook.log', 'Produto validado com sucesso', [
    'produto_slug' => $produtoSlug,
    'produto_nome' => $produtoConfig['nome'],
    'produto_id' => $produtoConfig['id'],
    'token_masked' => substr($utmifyToken, 0, 8) . '...' . substr($utmifyToken, -4)
]);

// ============================================
// PREPARA DADOS DO CLIENTE E UTMs
// ============================================
if ($savedData) {
    $customer = array_merge($customer, $savedData['customer'] ?? []);
    $utms = [
        'utm_source' => $savedData['utm_source'] ?? '',
        'utm_medium' => $savedData['utm_medium'] ?? '',
        'utm_campaign' => $savedData['utm_campaign'] ?? '',
        'utm_content' => $savedData['utm_content'] ?? '',
        'utm_term' => $savedData['utm_term'] ?? '',
        'sck' => $savedData['sck'] ?? '',
        'src' => $savedData['src'] ?? ''
    ];
    $clientIp = $savedData['customer_ip'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
    $cotas = $savedData['cotas'] ?? 1;
} else {
    $utms = [];
    $clientIp = $_SERVER['REMOTE_ADDR'] ?? '';
    $cotas = 1;
}

// ============================================
// PROCESSA CONFORME STATUS
// ============================================
$results = [
    'utmify' => false,
    'produto' => $produtoSlug
];

if ($statusNorm === 'paid') {
    logCheckout('webhook.log', "Processando PAGAMENTO CONFIRMADO - {$produtoConfig['nome']}");

    // Cria instância UTMify com token específico do produto
    if (class_exists('Utmify')) {
        $utmify = new Utmify($produtoSlug);

        // Verifica se o token está correto
        if ($utmify->getToken() !== $utmifyToken) {
            logCheckout('webhook.log', 'ERRO: Token UTMify não corresponde ao produto', [
                'expected' => substr($utmifyToken, 0, 8) . '...',
                'got' => substr($utmify->getToken(), 0, 8) . '...'
            ]);
        }

        $utmifyResult = $utmify->sendPaid(
            $orderId,
            $amountCents,
            [
                'name' => $customer['name'] ?? '',
                'email' => $customer['email'] ?? '',
                'phone' => $customer['phone'] ?? '',
                'document' => $customer['document'] ?? '',
                'ip' => $clientIp
            ],
            [
                'id' => $produtoConfig['id'],
                'name' => $produtoConfig['nome'],
                'quantity' => $cotas
            ],
            $utms
        );
        $results['utmify'] = $utmifyResult['success'] ?? false;

        logCheckout('webhook.log', 'UTMify Paid enviado', [
            'success' => $utmifyResult['success'] ?? false,
            'produto' => $produtoSlug,
            'http_code' => $utmifyResult['http_code'] ?? 0
        ]);
    }

} elseif ($statusNorm === 'refunded') {
    logCheckout('webhook.log', "Processando REEMBOLSO - {$produtoConfig['nome']}");

    if (class_exists('Utmify')) {
        $utmify = new Utmify($produtoSlug);

        $utmifyResult = $utmify->sendRefunded(
            $orderId,
            $amountCents,
            [
                'name' => $customer['name'] ?? '',
                'email' => $customer['email'] ?? '',
                'phone' => $customer['phone'] ?? '',
                'document' => $customer['document'] ?? '',
                'ip' => $clientIp
            ],
            [
                'id' => $produtoConfig['id'],
                'name' => $produtoConfig['nome'],
                'quantity' => $cotas
            ],
            $utms
        );
        $results['utmify'] = $utmifyResult['success'] ?? false;

        logCheckout('webhook.log', 'UTMify Refunded enviado', [
            'success' => $utmifyResult['success'] ?? false,
            'produto' => $produtoSlug,
            'http_code' => $utmifyResult['http_code'] ?? 0
        ]);
    }
} else {
    logCheckout('webhook.log', "Status não processável: $statusNorm", [
        'produto' => $produtoSlug,
        'order_id' => $orderId
    ]);
}

logCheckout('webhook.log', '=== WEBHOOK FINALIZADO ===', $results);

http_response_code(200);
echo json_encode([
    'success' => true,
    'gateway' => $detectedGateway,
    'order_id' => $orderId,
    'status' => $statusNorm,
    'amount' => $amount,
    'produto' => $produtoSlug,
    'produto_nome' => $produtoConfig['nome'],
    'results' => $results
]);


// ============================================
// FUNÇÕES AUXILIARES
// ============================================

/**
 * Detecta qual gateway enviou o webhook pelo formato do payload
 */
function detectGateway($data) {
    // TechByNet geralmente tem campos específicos
    if (isset($data['transaction_id']) || isset($data['pix_code'])) {
        return 'techbynet';
    }

    // Umbrella geralmente tem estrutura data.data ou objectId
    if (isset($data['objectId']) || isset($data['data']['data'])) {
        return 'umbrella';
    }

    // Verifica pelo header se disponível
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    if (stripos($userAgent, 'techbynet') !== false) {
        return 'techbynet';
    }
    if (stripos($userAgent, 'umbrella') !== false) {
        return 'umbrella';
    }

    // Default: usa gateway ativo
    return GATEWAY_ATIVO;
}

/**
 * Extrai dados do webhook conforme o gateway
 */
function extractWebhookData($data, $gateway) {
    $result = [
        'order_id' => '',
        'status' => '',
        'amount' => 0,
        'customer' => [
            'name' => '',
            'email' => '',
            'phone' => '',
            'document' => ''
        ],
        'products' => []
    ];

    if ($gateway === 'umbrella') {
        // Umbrella: dados podem estar em data ou data.data
        $webhookData = $data['data']['data'] ?? $data['data'] ?? $data;

        $result['order_id'] = $webhookData['id'] ?? $data['objectId'] ?? uniqid('order_');
        $result['status'] = $webhookData['status'] ?? '';
        $result['amount'] = $webhookData['amount'] ?? 0;

        $customer = $webhookData['customer'] ?? $webhookData['payer'] ?? [];
        $result['customer'] = [
            'name' => $customer['name'] ?? '',
            'email' => $customer['email'] ?? '',
            'phone' => $customer['phone'] ?? '',
            'document' => $customer['document']['number'] ?? $customer['document'] ?? ''
        ];

        $result['products'] = $webhookData['products'] ?? $webhookData['items'] ?? [];

    } elseif ($gateway === 'techbynet') {
        // TechByNet: estrutura mais plana
        $result['order_id'] = $data['id'] ?? $data['transaction_id'] ?? $data['external_reference'] ?? uniqid('order_');
        $result['status'] = $data['status'] ?? '';
        $result['amount'] = $data['amount'] ?? 0;

        $result['customer'] = [
            'name' => $data['payer_name'] ?? $data['customer_name'] ?? '',
            'email' => $data['payer_email'] ?? $data['customer_email'] ?? '',
            'phone' => $data['payer_phone'] ?? $data['customer_phone'] ?? '',
            'document' => $data['payer_document'] ?? $data['customer_document'] ?? ''
        ];

        $result['products'] = $data['products'] ?? $data['items'] ?? [];

    } else {
        // Fallback genérico
        $webhookData = $data['data'] ?? $data;

        $result['order_id'] = $webhookData['id'] ?? uniqid('order_');
        $result['status'] = $webhookData['status'] ?? '';
        $result['amount'] = $webhookData['amount'] ?? 0;

        $customer = $webhookData['customer'] ?? [];
        $result['customer'] = [
            'name' => $customer['name'] ?? '',
            'email' => $customer['email'] ?? '',
            'phone' => $customer['phone'] ?? '',
            'document' => $customer['document'] ?? ''
        ];

        $result['products'] = $webhookData['products'] ?? [];
    }

    return $result;
}
