<?php
/**
 * ============================================
 * STATUS API - VERIFICAÇÃO DE PAGAMENTO PIX
 * ============================================
 * Suporta múltiplos gateways (Umbrella, TechByNet)
 * Mantém verificação de comprovantes manuais
 */

require_once __DIR__ . '/config.php';
setCorsHeaders();

$id = $_GET['id'] ?? '';

if (empty($id)) {
    echo json_encode(['success' => false, 'error' => 'ID não informado']);
    exit;
}

// Arquivos de comprovantes (sistema centralizado na raiz)
$aprovadosFile = __DIR__ . '/../../comprovantes/aprovados.json';
$rejeitadosFile = __DIR__ . '/../../comprovantes/rejeitados.json';

// ========== VERIFICAÇÃO DE COMPROVANTE (check_aprovado) ==========
if (isset($_GET['check_aprovado'])) {
    // 1) Verifica se foi aprovado manualmente
    if (file_exists($aprovadosFile)) {
        $aprovados = json_decode(file_get_contents($aprovadosFile), true) ?: [];
        foreach ($aprovados as $a) {
            if (($a['pix_id'] ?? '') === $id) {
                echo json_encode([
                    'success' => true,
                    'isPaid' => true,
                    'aprovado' => true,
                    'rejeitado' => false
                ]);
                exit;
            }
        }
    }

    // 2) Verifica se foi rejeitado
    if (file_exists($rejeitadosFile)) {
        $rejeitados = json_decode(file_get_contents($rejeitadosFile), true) ?: [];
        foreach ($rejeitados as $r) {
            if (($r['pix_id'] ?? '') === $id) {
                echo json_encode([
                    'success' => true,
                    'isPaid' => false,
                    'aprovado' => false,
                    'rejeitado' => true,
                    'motivo' => $r['motivo'] ?? ''
                ]);
                exit;
            }
        }
    }

    // 3) Ainda pendente
    echo json_encode([
        'success' => true,
        'isPaid' => false,
        'aprovado' => false,
        'rejeitado' => false
    ]);
    exit;
}

// ========== CONSULTA GATEWAY ==========
try {
    $gateway = getGatewayConfig();
    $gatewayName = getGatewayName();
    
    // Monta URL e headers conforme gateway
    if ($gatewayName === 'umbrella') {
        // Umbrella: GET /api/user/transactions?id={id}
        $url = $gateway['api_url'] . '/api/user/transactions?id=' . urlencode($id);
        $headers = [
            'Content-Type: application/json',
            'User-Agent: UMBRELLAB2B/1.0',
            'x-api-key: ' . $gateway['api_key']
        ];
        
    } elseif ($gatewayName === 'techbynet') {
        // TechByNet: GET /api/user/transactions/{id}
        $url = $gateway['api_url'] . '/' . urlencode($id);
        $headers = [
            'Content-Type: application/json',
            'User-Agent: ' . ($gateway['user_agent'] ?: 'SmartCheckout/2.0'),
            'x-api-key: ' . $gateway['api_key']
        ];
        
    } else {
        throw new Exception("Gateway não suportado");
    }
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    // Log para debug
    logCheckout('status.log', 'Requisição gateway', [
        'gateway' => $gatewayName,
        'url' => $url,
        'http_code' => $httpCode,
        'error' => $error,
        'response' => substr($response, 0, 1000)
    ]);
    
    if ($error) {
        throw new Exception("Erro cURL: $error");
    }
    
    $data = json_decode($response, true);
    
    // Log dados parseados
    logCheckout('status.log', 'Dados parseados', [
        'data_keys' => $data ? array_keys($data) : 'null',
        'has_data_array' => isset($data['data']) ? (is_array($data['data']) ? 'array' : 'object') : 'no'
    ]);
    
    // Extrai status conforme gateway
    $status = null;
    
    if ($gatewayName === 'umbrella') {
        // Umbrella: estrutura é data.data.data[] (array aninhado)
        $transactions = $data['data']['data'] ?? $data['data'] ?? [];
        
        if (is_array($transactions) && !empty($transactions)) {
            // Se é array indexado (tem [0])
            if (isset($transactions[0])) {
                $status = $transactions[0]['status'] ?? null;
                logCheckout('status.log', 'Umbrella: pegou data.data.data[0]', ['status' => $status]);
            } else {
                // Se é objeto único
                $status = $transactions['status'] ?? null;
                logCheckout('status.log', 'Umbrella: pegou data.data.status', ['status' => $status]);
            }
        } else {
            logCheckout('status.log', 'Umbrella: transactions vazio', [
                'data_structure' => array_keys($data ?? [])
            ]);
        }
        
    } elseif ($gatewayName === 'techbynet') {
        // TechByNet: data.data.status ou data.status
        $status = $data['data']['data']['status'] ?? $data['data']['status'] ?? $data['status'] ?? null;
    }
    
    if ($httpCode >= 200 && $httpCode < 300 && $status) {
        $statusUpper = strtoupper($status);
        $isPaid = in_array($statusUpper, ['PAID', 'APPROVED', 'CONFIRMED', 'CONCLUIDA', 'COMPLETED']);
        
        echo json_encode([
            'success' => true,
            'gateway' => $gateway['name'],
            'status' => $status,
            'isPaid' => $isPaid
        ]);
        exit;
    }
    
} catch (Exception $e) {
    // Log do erro mas continua para fallback
    logCheckout('status.log', 'Erro gateway: ' . $e->getMessage(), ['id' => $id]);
}

// ========== FALLBACK: Verifica aprovação manual ==========
if (file_exists($aprovadosFile)) {
    $aprovados = json_decode(file_get_contents($aprovadosFile), true) ?: [];
    foreach ($aprovados as $a) {
        if (($a['pix_id'] ?? '') === $id) {
            echo json_encode([
                'success' => true,
                'status' => 'approved_manual',
                'isPaid' => true
            ]);
            exit;
        }
    }
}

// Nada encontrado
echo json_encode([
    'success' => false,
    'error' => 'Não foi possível verificar o status',
    'isPaid' => false
]);