<?php
/**
 * ============================================
 * STATUS API - VERIFICAÇÃO DE PAGAMENTO
 * ============================================
 * Suporta múltiplos gateways via gateways.json
 */

require_once __DIR__ . '/config.php';

setCorsHeaders();

$id = $_GET['id'] ?? $_GET['txid'] ?? null;
$checkAprovado = isset($_GET['check_aprovado']);

if (!$id) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'ID não informado']);
    exit;
}

// ============================================
// VERIFICAÇÃO DE APROVAÇÃO MANUAL (PRIORIDADE)
// ============================================
if ($checkAprovado) {
    // Tenta ambos os caminhos possíveis
    $caminhosPossiveis = [
        __DIR__ . '/../comprovantes/',  // /comprovantes/ (raiz)
        __DIR__ . '/comprovantes/'       // /api/comprovantes/
    ];
    
    foreach ($caminhosPossiveis as $comprovantesDir) {
        $aprovadosFile = $comprovantesDir . 'aprovados.json';
        $rejeitadosFile = $comprovantesDir . 'rejeitados.json';
        
        // Verifica aprovados
        if (file_exists($aprovadosFile)) {
            $aprovados = json_decode(file_get_contents($aprovadosFile), true) ?: [];
            foreach ($aprovados as $item) {
                if (($item['pix_id'] ?? '') === $id) {
                    echo json_encode([
                        'success' => true,
                        'isPaid' => true,
                        'aprovado' => true,
                        'status' => 'paid',
                        'method' => 'manual'
                    ]);
                    exit;
                }
            }
        }
        
        // Verifica rejeitados
        if (file_exists($rejeitadosFile)) {
            $rejeitados = json_decode(file_get_contents($rejeitadosFile), true) ?: [];
            foreach ($rejeitados as $item) {
                if (($item['pix_id'] ?? '') === $id) {
                    echo json_encode([
                        'success' => true,
                        'isPaid' => false,
                        'rejeitado' => true,
                        'motivo' => $item['motivo'] ?? 'Comprovante rejeitado',
                        'status' => 'rejected',
                        'method' => 'manual'
                    ]);
                    exit;
                }
            }
        }
    }
    
    // Não encontrado nas listas
    echo json_encode([
        'success' => true,
        'isPaid' => false,
        'aprovado' => false,
        'status' => 'pending',
        'method' => 'manual'
    ]);
    exit;
}

// ============================================
// VERIFICAÇÃO VIA GATEWAY
// ============================================
try {
    $gateway = getGatewayConfig();
    $gatewayName = getGatewayName();
    
    if (!$gateway) {
        throw new Exception("Gateway não configurado");
    }
    
    // Monta URL conforme gateway
    $endpoint = $gateway['endpoints']['pix_status'] ?? '';
    $baseUrl = rtrim($gateway['api_url'], '/');
    
    if ($gatewayName === 'umbrella') {
        // Umbrella: GET /api/user/transactions?id={id}
        $url = $baseUrl . '/api/user/transactions?id=' . urlencode($id);
    } elseif ($gatewayName === 'allowpay') {
        // AllowPay: GET /functions/v1/transactions/{id}
        $url = $baseUrl . '/' . urlencode($id);
    } elseif ($gatewayName === 'genesys') {
        // Genesys: GET /v1/transactions/{id}
        $url = $baseUrl . '/v1/transactions/' . urlencode($id);
    } else {
        // Padrão: GET {base_url}/{endpoint}/{id}
        $url = $baseUrl . rtrim($endpoint, '/') . '/' . urlencode($id);
    }
    
    // Monta header de autenticação
    $authType = $gateway['auth_type'] ?? 'apikey';
    
    if ($authType === 'basic') {
        // Basic Auth: base64(api_key:company_id)
        $credentials = $gateway['api_key'] . ':' . ($gateway['company_id'] ?? '');
        $authValue = 'Basic ' . base64_encode($credentials);
        $authHeaderLine = 'Authorization: ' . $authValue;
    } else {
        // API Key padrão
        $authHeader = $gateway['auth_header'] ?? 'x-api-key';
        $authPrefix = $gateway['auth_prefix'] ?? '';
        $authValue = $authPrefix . $gateway['api_key'];
        $authHeaderLine = $authHeader . ': ' . $authValue;
    }
    
    $headers = [
        'Content-Type: application/json',
        'Accept: application/json',
        'User-Agent: ' . ($gateway['user_agent'] ?: 'SmartCheckout/2.0'),
        $authHeaderLine
    ];
    
    logCheckout('status.log', 'Consultando status', [
        'gateway' => $gatewayName,
        'url' => $url,
        'id' => $id
    ]);
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    logCheckout('status.log', 'Resposta do gateway', [
        'http_code' => $httpCode,
        'response' => substr($response, 0, 500)
    ]);
    
    if ($error) {
        throw new Exception("Erro de conexão: $error");
    }
    
    $data = json_decode($response, true);
    
    if (!$data) {
        throw new Exception("Resposta inválida do gateway");
    }
    
    // ============================================
    // EXTRAI STATUS DA RESPOSTA
    // ============================================
    $status = null;
    
    // Tenta usando o path configurado
    $statusField = $gateway['status_field'] ?? 'data.status';
    $status = extractByPath($data, $statusField);
    
    // Fallbacks comuns
    if (!$status) {
        $statusPaths = [
            'data.data[0].status',  // Umbrella
            'data.data.status',     // TechByNet
            'data.status',
            'status'
        ];
        
        foreach ($statusPaths as $path) {
            $status = extractByPath($data, $path);
            if ($status) {
                logCheckout('status.log', "Status encontrado em: $path", ['status' => $status]);
                break;
            }
        }
    }
    
    if ($httpCode >= 200 && $httpCode < 300 && $status) {
        $isPaid = isPaidStatus($status);
        $mappedStatus = mapStatus($status);
        
        // ============================================
        // SE GATEWAY DIZ PENDENTE, VERIFICA APROVAÇÃO MANUAL
        // ============================================
        if (!$isPaid) {
            // Tenta ambos os caminhos possíveis
            $caminhos = [
                __DIR__ . '/../comprovantes/aprovados.json',  // /comprovantes/ (raiz)
                __DIR__ . '/comprovantes/aprovados.json'       // /api/comprovantes/
            ];
            
            foreach ($caminhos as $aprovadosFile) {
                if (file_exists($aprovadosFile)) {
                    $aprovados = json_decode(file_get_contents($aprovadosFile), true) ?: [];
                    foreach ($aprovados as $item) {
                        if (($item['pix_id'] ?? '') === $id || ($item['txid'] ?? '') === $id) {
                            logCheckout('status.log', 'Aprovado manualmente (comprovante)', [
                                'id' => $id,
                                'arquivo' => $aprovadosFile
                            ]);
                            echo json_encode([
                                'success' => true,
                                'gateway' => $gateway['name'],
                                'id' => $id,
                                'status' => 'paid',
                                'apiStatus' => $status,
                                'isPaid' => true,
                                'method' => 'manual_approved'
                            ]);
                            exit;
                        }
                    }
                }
            }
        }
        
        logCheckout('status.log', 'Status extraído', [
            'api_status' => $status,
            'mapped_status' => $mappedStatus,
            'is_paid' => $isPaid
        ]);
        
        echo json_encode([
            'success' => true,
            'gateway' => $gateway['name'],
            'id' => $id,
            'status' => $mappedStatus,
            'apiStatus' => $status,
            'isPaid' => $isPaid,
            'method' => 'gateway'
        ]);
        exit;
    }
    
    // Se não conseguiu extrair status, tenta aprovação manual como fallback
    logCheckout('status.log', 'Fallback para verificação manual', [
        'http_code' => $httpCode,
        'status_found' => $status
    ]);
    
    // Fallback: verifica arquivos de aprovação
    $caminhos = [
        __DIR__ . '/../comprovantes/aprovados.json',
        __DIR__ . '/comprovantes/aprovados.json'
    ];
    
    foreach ($caminhos as $aprovadosFile) {
        if (file_exists($aprovadosFile)) {
            $aprovados = json_decode(file_get_contents($aprovadosFile), true) ?: [];
            foreach ($aprovados as $item) {
                if (($item['pix_id'] ?? '') === $id) {
                    echo json_encode([
                        'success' => true,
                        'isPaid' => true,
                        'status' => 'paid',
                        'method' => 'manual_fallback'
                    ]);
                    exit;
                }
            }
        }
    }
    
    echo json_encode([
        'success' => false,
        'error' => 'Não foi possível verificar o status',
        'isPaid' => false,
        'status' => 'unknown'
    ]);
    
} catch (Exception $e) {
    logCheckout('status.log', 'ERRO: ' . $e->getMessage());
    
    // Em caso de erro, tenta verificação manual
    $caminhos = [
        __DIR__ . '/../comprovantes/aprovados.json',
        __DIR__ . '/comprovantes/aprovados.json'
    ];
    
    foreach ($caminhos as $aprovadosFile) {
        if (file_exists($aprovadosFile)) {
            $aprovados = json_decode(file_get_contents($aprovadosFile), true) ?: [];
            foreach ($aprovados as $item) {
                if (($item['pix_id'] ?? '') === $id) {
                    echo json_encode([
                        'success' => true,
                        'isPaid' => true,
                        'status' => 'paid',
                        'method' => 'manual_fallback'
                    ]);
                    exit;
                }
            }
        }
    }
    
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'isPaid' => false
    ]);
}