<?php
/**
 * API de Consulta de CPF
 * Consulta dados do cliente via API externa e retorna JSON
 */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');

// Verifica se o CPF foi fornecido
if (!isset($_GET['cpf']) || empty($_GET['cpf'])) {
    echo json_encode([
        'success' => false,
        'error' => 'CPF não fornecido'
    ]);
    exit;
}

// Limpa o CPF (apenas números)
$cpf = preg_replace('/[^0-9]/', '', $_GET['cpf']);

// Valida se tem 11 dígitos
if (strlen($cpf) !== 11) {
    echo json_encode([
        'success' => false,
        'error' => 'CPF inválido. Deve conter 11 dígitos.'
    ]);
    exit;
}

// URL da API externa (com porta 80)
//$api_url = "https://base1.sistemafull.site:80/api/cpf1?CPF={$cpf}";

// Configuração do cURL
$ch = curl_init();
curl_setopt_array($ch, [
    CURLOPT_URL => $api_url,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_HTTPHEADER => [
        'Accept: application/json',
        'User-Agent: PHP/ConsultaCPF'
    ],
    CURLOPT_SSL_VERIFYPEER => false,
    CURLOPT_SSL_VERIFYHOST => false,
]);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
curl_close($ch);

// Verifica erros de conexão
if ($response === false || $error) {
    echo json_encode([
        'success' => false,
        'error' => 'Erro ao consultar a API: ' . $error
    ]);
    exit;
}

// Verifica código HTTP
if ($httpCode !== 200) {
    echo json_encode([
        'success' => false,
        'error' => 'API retornou HTTP ' . $httpCode
    ]);
    exit;
}

// Decodifica JSON
$json = json_decode($response, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    echo json_encode([
        'success' => false,
        'error' => 'Erro ao processar resposta: ' . json_last_error_msg()
    ]);
    exit;
}

// Verifica se tem dados
if (empty($json) || !isset($json['NOME'])) {
    echo json_encode([
        'success' => false,
        'error' => 'CPF não encontrado'
    ]);
    exit;
}

// Extrai dados
$nome = $json['NOME'] ?? '';
$nascimento = $json['NASC'] ?? '';
$sexo = $json['SEXO'] ?? '';
$renda = $json['RENDA'] ?? '';

// Formata data de nascimento (de "1985-01-07 00:00:00" para "07/01/1985")
$nascimentoFormatado = '';
$idade = 0;
if (!empty($nascimento)) {
    $dataLimpa = explode(' ', $nascimento)[0];
    $dataNascimento = DateTime::createFromFormat('Y-m-d', $dataLimpa);
    
    if ($dataNascimento !== false) {
        $hoje = new DateTime();
        $idade = $hoje->diff($dataNascimento)->y;
        $nascimentoFormatado = $dataNascimento->format('d/m/Y');
    }
}

// Obtém primeiro nome
$primeiroNome = '';
if (!empty($nome)) {
    $primeiroNome = explode(' ', trim($nome))[0];
}

// Retorna JSON com os dados
echo json_encode([
    'success' => true,
    'dados' => [
        'cpf' => $cpf,
        'nome' => $nome,
        'primeiro_nome' => $primeiroNome,
        'nascimento' => $nascimentoFormatado,
        'idade' => $idade,
        'sexo' => $sexo,
        'telefone' => '',
        'email' => '',
        'score' => '',
        'renda' => $renda
    ]
]);