<?php
/**
 * API DE CHAT AO VIVO - PRÊMIOS DO CARLINHOS
 * Com proteção anti-clone e upload de comprovante
 * Local: /backup/api/chat.php OU /api/chat.php
 */

// ========== FUSO HORÁRIO BRASIL ==========
date_default_timezone_set('America/Sao_Paulo');

// ==================== PROTEÇÃO ANTI-CLONE ====================

// Domínios autorizados (CONFIGURE SEUS DOMÍNIOS AQUI)
$DOMINIOS_AUTORIZADOS = [
    'checkout-carlinhosmaia.com',
    'www.checkout-carlinhosmaia.com',
    'localhost',
    '127.0.0.1'
];

// Chave secreta (MUDE PARA UMA CHAVE ÚNICA)
define('CHAVE_SECRETA_CHAT', 'SuaChaveSecreta2024_MudeME!');

// Verifica origem da requisição
function verificarOrigem($dominiosAutorizados) {
    $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
    $referer = $_SERVER['HTTP_REFERER'] ?? '';
    $host = $_SERVER['HTTP_HOST'] ?? '';
    
    if (strpos($host, 'localhost') !== false || strpos($host, '127.0.0.1') !== false) {
        return true;
    }
    
    foreach ($dominiosAutorizados as $dominio) {
        if (
            strpos($origin, $dominio) !== false ||
            strpos($referer, $dominio) !== false ||
            strpos($host, $dominio) !== false
        ) {
            return true;
        }
    }
    
    $logDir = __DIR__ . '/../data/logs';
    if (!is_dir($logDir)) mkdir($logDir, 0755, true);
    
    $log = [
        'time' => date('Y-m-d H:i:s'),
        'ip' => $_SERVER['REMOTE_ADDR'] ?? '',
        'origin' => $origin,
        'referer' => $referer,
        'ua' => $_SERVER['HTTP_USER_AGENT'] ?? ''
    ];
    file_put_contents($logDir . '/blocked_' . date('Y-m-d') . '.log', json_encode($log) . "\n", FILE_APPEND);
    
    return false;
}

function verificarRateLimit($limite = 100, $janela = 60) {
    $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    $arquivo = sys_get_temp_dir() . '/chat_rate_' . md5($ip) . '.json';
    
    $dados = file_exists($arquivo) ? json_decode(file_get_contents($arquivo), true) : [];
    $agora = time();
    
    $dados = array_filter($dados ?? [], function($t) use ($agora, $janela) {
        return ($agora - $t) < $janela;
    });
    
    if (count($dados) >= $limite) {
        return false;
    }
    
    $dados[] = $agora;
    file_put_contents($arquivo, json_encode(array_values($dados)));
    
    return true;
}

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-Chat-Token');
header('X-Frame-Options: SAMEORIGIN');
header('X-Content-Type-Options: nosniff');
header('X-XSS-Protection: 1; mode=block');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

define('DATA_DIR', __DIR__ . '/../data');
define('CHATS_DIR', DATA_DIR . '/chats');
define('AGENT_FILE', DATA_DIR . '/agent.json');

function getUtmDir() {
    $possiblePaths = [
        __DIR__ . '/utm_data',
        __DIR__ . '/../api/utm_data',
        $_SERVER['DOCUMENT_ROOT'] . '/api/utm_data',
    ];
    
    foreach ($possiblePaths as $path) {
        if (is_dir($path)) {
            return $path;
        }
    }
    
    return $possiblePaths[0];
}

define('UTM_DATA_DIR', getUtmDir());

if (!file_exists(DATA_DIR)) mkdir(DATA_DIR, 0755, true);
if (!file_exists(CHATS_DIR)) mkdir(CHATS_DIR, 0755, true);

function response($data, $code = 200) {
    http_response_code($code);
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

function getChat($chatId) {
    $file = CHATS_DIR . '/' . preg_replace('/[^a-zA-Z0-9_-]/', '', $chatId) . '.json';
    if (!file_exists($file)) return null;
    return json_decode(file_get_contents($file), true);
}

function saveChat($chatId, $data) {
    $file = CHATS_DIR . '/' . preg_replace('/[^a-zA-Z0-9_-]/', '', $chatId) . '.json';
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function getAllChats() {
    $chats = [];
    $files = glob(CHATS_DIR . '/*.json');
    foreach ($files as $file) {
        $chat = json_decode(file_get_contents($file), true);
        if ($chat && ($chat['status'] ?? 'active') === 'active') {
            $chats[] = $chat;
        }
    }
    usort($chats, function($a, $b) {
        return strtotime($b['timestamp'] ?? 0) - strtotime($a['timestamp'] ?? 0);
    });
    return $chats;
}

function getAgentStatus() {
    if (!file_exists(AGENT_FILE)) {
        return ['online' => false, 'last_seen' => null];
    }
    $data = json_decode(file_get_contents(AGENT_FILE), true);
    
    if (isset($data['last_seen']) && (time() - $data['last_seen']) > 15) {
        $data['online'] = false;
    }
    
    return $data;
}

function setAgentOnline($online = true) {
    $data = ['online' => $online, 'last_seen' => time()];
    file_put_contents(AGENT_FILE, json_encode($data));
    return $data;
}

// ========== ROTEAMENTO ==========

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? null;

// Verifica FormData PRIMEIRO (para upload)
if ($method === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
}

// Se não for FormData, tenta JSON
$input = null;
if ($method === 'POST' && !$action) {
    $input = json_decode(file_get_contents('php://input'), true);
    if ($input) $action = $input['action'] ?? $action;
}

// ========== UPLOAD DE COMPROVANTE ==========
if ($method === 'POST' && $action === 'upload_comprovante') {
    $chatId = $_POST['chat_id'] ?? '';
    $pixId = $_POST['pix_id'] ?? '';
    $clienteNome = $_POST['cliente_nome'] ?? '';
    $clienteCpf = $_POST['cliente_cpf'] ?? '';
    $clienteEmail = $_POST['cliente_email'] ?? '';
    $clienteCelular = $_POST['cliente_celular'] ?? '';
    $clienteValor = $_POST['cliente_valor'] ?? 0;
    $clienteCotas = $_POST['cliente_cotas'] ?? 0;
    $clienteJson = $_POST['cliente_json'] ?? '{}';
    
    // Tenta pegar dados do JSON completo se disponível
    $clienteData = json_decode($clienteJson, true) ?: [];
    
    if (!$chatId) {
        response(['error' => 'chat_id obrigatório'], 400);
    }
    
    if (!isset($_FILES['comprovante']) || $_FILES['comprovante']['error'] !== UPLOAD_ERR_OK) {
        response(['error' => 'Arquivo não enviado ou erro no upload'], 400);
    }
    
    $file = $_FILES['comprovante'];
    
    // Valida extensão (igual upload.php)
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $allowedExt = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'pdf'];
    if (!in_array($ext, $allowedExt)) {
        response(['error' => 'Tipo de arquivo não permitido. Use: ' . implode(', ', $allowedExt)], 400);
    }
    
    if ($file['size'] > 10 * 1024 * 1024) {
        response(['error' => 'Arquivo muito grande (máx 10MB)'], 400);
    }
    
    // Usa mesma pasta do upload.php: /comprovantes/
    $uploadDir = __DIR__ . '/../comprovantes/';
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    // Remove do rejeitados.json se existir (permite reenvio)
    $rejeitadosFile = $uploadDir . 'rejeitados.json';
    if (file_exists($rejeitadosFile) && $pixId) {
        $rejeitados = json_decode(file_get_contents($rejeitadosFile), true) ?: [];
        $totalAntes = count($rejeitados);
        $rejeitados = array_filter($rejeitados, function($r) use ($pixId) {
            return ($r['pix_id'] ?? '') !== $pixId;
        });
        if (count($rejeitados) < $totalAntes) {
            file_put_contents($rejeitadosFile, json_encode(array_values($rejeitados), JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        }
    }
    
    // Gera nome único (formato igual upload.php)
    $timestamp = date('Y-m-d_H-i-s');
    $pixIdClean = preg_replace('/[^a-zA-Z0-9]/', '', $pixId ?: $chatId);
    $filename = $timestamp . '_' . $pixIdClean . '.' . $ext;
    
    $destPath = $uploadDir . $filename;
    
    if (!move_uploaded_file($file['tmp_name'], $destPath)) {
        response(['error' => 'Erro ao salvar arquivo'], 500);
    }
    
    // Adiciona ao pendentes.json (igual upload.php)
    $pendentesFile = $uploadDir . 'pendentes.json';
    $pendentes = [];
    if (file_exists($pendentesFile)) {
        $conteudo = file_get_contents($pendentesFile);
        if ($conteudo) {
            $pendentes = json_decode($conteudo, true) ?: [];
        }
    }
    
    // Monta dados do cliente com fallbacks
    $pendentes[] = [
        'id' => uniqid('comp_'),
        'pix_id' => $pixId,
        'chat_id' => $chatId,
        'arquivo' => $filename,
        'cliente' => [
            'nome' => $clienteNome ?: ($clienteData['nome'] ?? $clienteData['name'] ?? ''),
            'cpf' => $clienteCpf ?: ($clienteData['cpf'] ?? $clienteData['document'] ?? ''),
            'email' => $clienteEmail ?: ($clienteData['email'] ?? ''),
            'celular' => $clienteCelular ?: ($clienteData['celular'] ?? $clienteData['telefone'] ?? $clienteData['phone'] ?? ''),
            'valor' => floatval($clienteValor ?: ($clienteData['valor'] ?? 0)),
            'cotas' => intval($clienteCotas ?: ($clienteData['cotas'] ?? 0))
        ],
        'data' => date('Y-m-d H:i:s'),
        'status' => 'pendente',
        'ip' => $_SERVER['REMOTE_ADDR'] ?? '',
        'origem' => 'chat'
    ];
    
    file_put_contents($pendentesFile, json_encode($pendentes, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    // Atualiza chat
    $chat = getChat($chatId);
    if ($chat) {
        $chat['messages'][] = [
            'from' => 'user',
            'message' => '📎 Comprovante enviado: ' . $file['name'],
            'attachment' => $filename,
            'time' => date('H:i'),
            'timestamp' => date('c')
        ];
        $chat['last_message'] = '📎 Comprovante enviado';
        $chat['timestamp'] = date('c');
        $chat['unread'] = true;
        $chat['has_comprovante'] = true;
        $chat['comprovante_file'] = $filename;
        saveChat($chatId, $chat);
    }
    
    response([
        'success' => true,
        'filename' => $filename,
        'message' => 'Comprovante enviado com sucesso'
    ]);
}

// ========== GET ==========

if ($method === 'GET') {
    switch ($action) {
        case 'get_chats':
            response(['success' => true, 'chats' => getAllChats(), 'agent_online' => getAgentStatus()['online']]);
            break;
            
        case 'get_messages':
            $chatId = $_GET['chat_id'] ?? '';
            $lastCheck = intval($_GET['last_check'] ?? 0);
            
            if (!$chatId) response(['error' => 'chat_id obrigatório'], 400);
            
            $chat = getChat($chatId);
            if (!$chat) response(['error' => 'Chat não encontrado'], 404);
            
            $newMessages = [];
            foreach ($chat['messages'] ?? [] as $msg) {
                if ($msg['from'] === 'agent') {
                    $msgTime = strtotime($msg['timestamp'] ?? 'now') * 1000;
                    if ($msgTime > $lastCheck) $newMessages[] = $msg;
                }
            }
            
            response([
                'success' => true,
                'messages' => $newMessages,
                'typing' => $chat['agent_typing'] ?? false,
                'agent_online' => getAgentStatus()['online']
            ]);
            break;
            
        case 'agent_status':
            response(['success' => true, 'online' => getAgentStatus()['online']]);
            break;
        
        case 'get_proactive':
            $visitorId = $_GET['visitor_id'] ?? '';
            $cpf = preg_replace('/[^0-9]/', '', $_GET['cpf'] ?? '');
            $telefone = preg_replace('/[^0-9]/', '', $_GET['telefone'] ?? '');
            $pixId = $_GET['pix_id'] ?? '';
            
            $messages = [];
            $foundChatId = null;
            
            $files = glob(CHATS_DIR . '/*.json');
            foreach ($files as $file) {
                $chat = json_decode(file_get_contents($file), true);
                if (!$chat) continue;
                
                if (!($chat['proativo'] ?? false)) continue;
                if ($chat['status'] === 'closed') continue;
                
                $chatCliente = $chat['cliente'] ?? [];
                $chatCpf = preg_replace('/[^0-9]/', '', $chatCliente['cpf'] ?? $chatCliente['document'] ?? '');
                $chatTel = preg_replace('/[^0-9]/', '', $chatCliente['telefone'] ?? $chatCliente['phone'] ?? '');
                $chatPixId = $chat['pix_id'] ?? '';
                $chatVisitorId = $chat['visitor_id'] ?? '';
                
                $match = false;
                if ($visitorId && $chatVisitorId && $visitorId === $chatVisitorId) $match = true;
                if ($cpf && $chatCpf && strpos($chatCpf, $cpf) !== false) $match = true;
                if ($telefone && $chatTel && strpos($chatTel, $telefone) !== false) $match = true;
                if ($pixId && $chatPixId && stripos($chatPixId, $pixId) !== false) $match = true;
                
                if ($match) {
                    $foundChatId = $chat['id'];
                    
                    foreach ($chat['messages'] ?? [] as $msg) {
                        if ($msg['from'] === 'agent' && !($msg['delivered'] ?? false)) {
                            $messages[] = [
                                'message' => $msg['message'],
                                'time' => $msg['time'] ?? '',
                                'chat_id' => $chat['id']
                            ];
                        }
                    }
                    
                    if (count($messages) > 0) {
                        foreach ($chat['messages'] as &$msg) {
                            if ($msg['from'] === 'agent') {
                                $msg['delivered'] = true;
                            }
                        }
                        saveChat($chat['id'], $chat);
                    }
                    
                    break;
                }
            }
            
            response([
                'success' => true, 
                'messages' => $messages,
                'chat_id' => $foundChatId
            ]);
            break;
        
        case 'get_all_messages':
            $chatId = $_GET['chat_id'] ?? '';
            if (!$chatId) response(['error' => 'chat_id obrigatório'], 400);
            
            $chat = getChat($chatId);
            if (!$chat) response(['error' => 'Chat não encontrado'], 404);
            
            response([
                'success' => true,
                'messages' => $chat['messages'] ?? [],
                'chat' => [
                    'id' => $chat['id'],
                    'user_name' => $chat['user_name'] ?? '',
                    'subject' => $chat['subject'] ?? ''
                ]
            ]);
            break;
        
        case 'buscar_cliente':
            $tipo = $_GET['tipo'] ?? 'cpf';
            $valor = $_GET['valor'] ?? '';
            
            if (!$valor) {
                response(['success' => false, 'error' => 'Valor não informado']);
            }
            
            $cliente = null;
            
            $files = glob(CHATS_DIR . '/*.json');
            foreach ($files as $file) {
                $chat = json_decode(file_get_contents($file), true);
                if (!$chat) continue;
                
                $chatCliente = $chat['cliente'] ?? [];
                
                if ($tipo === 'pix') {
                    $pixIdChat = $chat['pix_id'] ?? $chatCliente['pix_id'] ?? '';
                    if ($pixIdChat && stripos($pixIdChat, $valor) !== false) {
                        $cliente = array_merge($chatCliente, [
                            'nome' => $chatCliente['nome'] ?? $chat['user_name'] ?? 'Cliente',
                            'pix_id' => $pixIdChat,
                            'valor' => $chat['valor'] ?? $chatCliente['valor'] ?? null,
                            'cotas' => $chat['cotas'] ?? $chatCliente['cotas'] ?? null,
                            'chat_id' => $chat['id'] ?? null
                        ]);
                        break;
                    }
                } else if ($tipo === 'cpf') {
                    $cpfBusca = preg_replace('/[^0-9]/', '', $valor);
                    $cpfChat = preg_replace('/[^0-9]/', '', $chatCliente['cpf'] ?? $chatCliente['document'] ?? '');
                    
                    if ($cpfBusca && $cpfChat && strpos($cpfChat, $cpfBusca) !== false) {
                        $cliente = array_merge($chatCliente, [
                            'nome' => $chatCliente['nome'] ?? $chat['user_name'] ?? 'Cliente',
                            'pix_id' => $chat['pix_id'] ?? $chatCliente['pix_id'] ?? null,
                            'valor' => $chat['valor'] ?? $chatCliente['valor'] ?? null,
                            'cotas' => $chat['cotas'] ?? $chatCliente['cotas'] ?? null,
                            'chat_id' => $chat['id'] ?? null
                        ]);
                        break;
                    }
                }
            }
            
            if (!$cliente) {
                $utmDirs = [
                    UTM_DATA_DIR,
                    __DIR__ . '/utm_data',
                    __DIR__ . '/../api/utm_data',
                    $_SERVER['DOCUMENT_ROOT'] . '/api/utm_data'
                ];
                
                foreach ($utmDirs as $utmDir) {
                    if (!is_dir($utmDir)) continue;
                    
                    $utmFiles = glob($utmDir . '/*.json');
                    
                    foreach ($utmFiles as $file) {
                        $lead = json_decode(file_get_contents($file), true);
                        if (!$lead) continue;
                        
                        $customer = $lead['customer'] ?? [];
                        $found = false;
                        
                        if ($tipo === 'pix' || $tipo === 'transaction') {
                            $txId = $lead['transaction_id'] ?? '';
                            if ($txId && stripos($txId, $valor) !== false) {
                                $found = true;
                            }
                        } else if ($tipo === 'cpf') {
                            $cpfBusca = preg_replace('/[^0-9]/', '', $valor);
                            $cpfLead = preg_replace('/[^0-9]/', '', $customer['document'] ?? '');
                            if ($cpfBusca && $cpfLead && strpos($cpfLead, $cpfBusca) !== false) {
                                $found = true;
                            }
                        } else if ($tipo === 'telefone' || $tipo === 'whatsapp') {
                            $telBusca = preg_replace('/[^0-9]/', '', $valor);
                            $telLead = preg_replace('/[^0-9]/', '', $customer['phone'] ?? '');
                            if ($telBusca && $telLead && strpos($telLead, $telBusca) !== false) {
                                $found = true;
                            }
                        } else if ($tipo === 'email') {
                            $emailLead = strtolower($customer['email'] ?? '');
                            if ($emailLead && stripos($emailLead, strtolower($valor)) !== false) {
                                $found = true;
                            }
                        } else if ($tipo === 'nome') {
                            $nomeLead = strtolower($customer['name'] ?? '');
                            if ($nomeLead && stripos($nomeLead, strtolower($valor)) !== false) {
                                $found = true;
                            }
                        }
                        
                        if ($found) {
                            $valorReais = isset($lead['valor_centavos']) ? $lead['valor_centavos'] / 100 : null;
                            
                            $cliente = [
                                'nome' => $customer['name'] ?? 'Cliente',
                                'cpf' => $customer['document'] ?? null,
                                'telefone' => $customer['phone'] ?? null,
                                'email' => $customer['email'] ?? null,
                                'pix_id' => $lead['transaction_id'] ?? null,
                                'valor' => $valorReais,
                                'cotas' => $lead['cotas'] ?? null,
                                'produto' => $lead['produto'] ?? null,
                                'gateway' => $lead['gateway'] ?? null,
                                'status' => $lead['status'] ?? null,
                                'ip' => $lead['customer_ip'] ?? null,
                                'created_at' => $lead['created_at'] ?? null,
                                'utm' => [
                                    'source' => $lead['utm_source'] ?? null,
                                    'medium' => $lead['utm_medium'] ?? null,
                                    'campaign' => $lead['utm_campaign'] ?? null,
                                    'content' => $lead['utm_content'] ?? null,
                                    'term' => $lead['utm_term'] ?? null,
                                    'sck' => $lead['sck'] ?? null,
                                    'src' => $lead['src'] ?? null
                                ],
                                'facebook' => [
                                    'fbc' => $lead['fbc'] ?? null,
                                    'fbp' => $lead['fbp'] ?? null
                                ],
                                'source_file' => basename($file),
                                'source_dir' => $utmDir
                            ];
                            break 2;
                        }
                    }
                }
            }
            
            if ($cliente) {
                response(['success' => true, 'cliente' => $cliente]);
            } else {
                response(['success' => false, 'error' => 'Cliente não encontrado']);
            }
            break;
        
        case 'debug_paths':
            $paths = [
                'UTM_DATA_DIR' => UTM_DATA_DIR,
                'UTM_DATA_DIR_exists' => is_dir(UTM_DATA_DIR),
                '__DIR__' => __DIR__,
                'DOCUMENT_ROOT' => $_SERVER['DOCUMENT_ROOT'],
            ];
            response(['success' => true, 'debug' => $paths]);
            break;
            
        default:
            response(['error' => 'Ação inválida'], 400);
    }
}

// ========== POST ==========

if ($method === 'POST') {
    $input = $input ?? [];
    
    switch ($action) {
        case 'start_chat':
            $chatId = $input['chat_id'] ?? 'chat_' . time() . '_' . bin2hex(random_bytes(4));
            
            $chat = [
                'id' => $chatId,
                'user_name' => $input['user_name'] ?? 'Visitante',
                'subject' => $input['subject'] ?? 'outro',
                'status' => 'active',
                'unread' => true,
                'last_message' => $input['initial_message'] ?: 'Chat iniciado',
                'timestamp' => date('c'),
                'created_at' => date('c'),
                'cliente' => $input['cliente'] ?? [],
                'pix_id' => $input['pix_id'] ?? ($input['cliente']['pix_id'] ?? null),
                'valor' => $input['cliente']['valor'] ?? null,
                'cotas' => $input['cliente']['cotas'] ?? null,
                'page_url' => $input['page_url'] ?? '',
                'messages' => [],
                'agent_typing' => false
            ];
            
            if ($input['initial_message'] ?? '') {
                $chat['messages'][] = [
                    'from' => 'user',
                    'message' => $input['initial_message'],
                    'time' => date('H:i'),
                    'timestamp' => date('c')
                ];
            }
            
            saveChat($chatId, $chat);
            response(['success' => true, 'chat_id' => $chatId, 'agent_online' => getAgentStatus()['online']]);
            break;
        
        case 'start_chat_proativo':
            $chatId = $input['chat_id'] ?? 'chat_' . time() . '_' . bin2hex(random_bytes(4));
            $cliente = $input['cliente'] ?? [];
            $agente = $input['agente'] ?? 'Atendente';
            
            $chat = [
                'id' => $chatId,
                'user_name' => $cliente['nome'] ?? 'Cliente',
                'subject' => 'proativo',
                'status' => 'active',
                'unread' => false,
                'last_message' => 'Conversa iniciada por ' . $agente,
                'timestamp' => date('c'),
                'created_at' => date('c'),
                'cliente' => $cliente,
                'pix_id' => $cliente['pix_id'] ?? null,
                'valor' => $cliente['valor'] ?? null,
                'cotas' => $cliente['cotas'] ?? null,
                'proativo' => true,
                'agente' => $agente,
                'messages' => [
                    [
                        'from' => 'system',
                        'message' => '📞 Conversa iniciada proativamente por ' . $agente,
                        'time' => date('H:i'),
                        'timestamp' => date('c')
                    ]
                ],
                'agent_typing' => false
            ];
            
            saveChat($chatId, $chat);
            response(['success' => true, 'chat_id' => $chatId]);
            break;
            
        case 'message':
            $chatId = $input['chat_id'] ?? '';
            $message = $input['message'] ?? '';
            $from = $input['from'] ?? 'user';
            
            if (!$chatId || !$message) response(['error' => 'chat_id e message obrigatórios'], 400);
            
            $chat = getChat($chatId);
            if (!$chat) response(['error' => 'Chat não encontrado'], 404);
            
            $chat['messages'][] = [
                'from' => $from,
                'message' => $message,
                'time' => date('H:i'),
                'timestamp' => date('c')
            ];
            
            $chat['last_message'] = $message;
            $chat['timestamp'] = date('c');
            if ($from === 'user') $chat['unread'] = true;
            $chat['agent_typing'] = false;
            
            saveChat($chatId, $chat);
            response(['success' => true]);
            break;
            
        case 'agent_message':
            $chatId = $input['chat_id'] ?? '';
            $message = $input['message'] ?? '';
            
            if (!$chatId || !$message) response(['error' => 'Dados obrigatórios'], 400);
            
            $chat = getChat($chatId);
            if (!$chat) response(['error' => 'Chat não encontrado'], 404);
            
            $chat['messages'][] = [
                'from' => 'agent',
                'message' => $message,
                'time' => date('H:i'),
                'timestamp' => date('c')
            ];
            
            $chat['last_message'] = $message;
            $chat['timestamp'] = date('c');
            $chat['unread'] = false;
            $chat['agent_typing'] = false;
            
            saveChat($chatId, $chat);
            setAgentOnline(true);
            response(['success' => true]);
            break;
            
        case 'mark_read':
            $chat = getChat($input['chat_id'] ?? '');
            if ($chat) {
                $chat['unread'] = false;
                saveChat($input['chat_id'], $chat);
            }
            response(['success' => true]);
            break;
            
        case 'typing':
            $chat = getChat($input['chat_id'] ?? '');
            if ($chat) {
                $chat['agent_typing'] = $input['typing'] ?? false;
                saveChat($input['chat_id'], $chat);
            }
            setAgentOnline(true);
            response(['success' => true]);
            break;
            
        case 'close_chat':
            $chat = getChat($input['chat_id'] ?? '');
            if ($chat) {
                $chat['status'] = 'closed';
                $chat['closed_at'] = date('c');
                saveChat($input['chat_id'], $chat);
            }
            response(['success' => true]);
            break;
            
        case 'agent_heartbeat':
            setAgentOnline(true);
            $chats = getAllChats();
            $unread = count(array_filter($chats, fn($c) => $c['unread'] ?? false));
            response(['success' => true, 'online' => true, 'chats' => $chats, 'unread_count' => $unread]);
            break;
            
        case 'agent_offline':
            setAgentOnline(false);
            response(['success' => true, 'online' => false]);
            break;
            
        default:
            response(['error' => 'Ação inválida: ' . $action], 400);
    }
}

response(['error' => 'Método não permitido'], 405);