<?php
/**
 * API para buscar transações por CPF
 * Busca nos arquivos JSON da pasta utm_data
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');

// Pasta onde estão os arquivos JSON
$UTM_DATA_DIR = __DIR__ . '/utm_data/';

// Pega o CPF da query string
$cpf = isset($_GET['cpf']) ? preg_replace('/\D/', '', $_GET['cpf']) : '';

if (strlen($cpf) !== 11) {
    echo json_encode([
        'success' => false,
        'error' => 'CPF inválido. Deve conter 11 dígitos.'
    ]);
    exit;
}

$transacoesEncontradas = [];

// Verifica se a pasta existe
if (!is_dir($UTM_DATA_DIR)) {
    echo json_encode([
        'success' => true,
        'transacoes' => [],
        'total' => 0,
        'message' => 'Pasta de dados não encontrada'
    ]);
    exit;
}

// Lista todos os arquivos JSON na pasta
$arquivos = glob($UTM_DATA_DIR . '*.json');

foreach ($arquivos as $arquivo) {
    try {
        $conteudo = file_get_contents($arquivo);
        if (!$conteudo) continue;
        
        $dados = json_decode($conteudo, true);
        if (!$dados || !is_array($dados)) continue;
        
        // Extrai CPF do customer.document
        $cpfTransacao = '';
        
        if (isset($dados['customer']['document'])) {
            $cpfTransacao = preg_replace('/\D/', '', $dados['customer']['document']);
        } elseif (isset($dados['customer']['cpf'])) {
            $cpfTransacao = preg_replace('/\D/', '', $dados['customer']['cpf']);
        } elseif (isset($dados['cpf'])) {
            $cpfTransacao = preg_replace('/\D/', '', $dados['cpf']);
        } elseif (isset($dados['document'])) {
            $cpfTransacao = preg_replace('/\D/', '', $dados['document']);
        }
        
        // Verifica se é o CPF buscado
        if ($cpfTransacao === $cpf) {
            // Verifica se está pago
            $statusPago = ['paid', 'pago', 'approved', 'aprovado', 'confirmed', 'confirmado', 'completed'];
            $status = strtolower($dados['webhook_status'] ?? $dados['status'] ?? '');
            
            if (in_array($status, $statusPago)) {
                $transacoesEncontradas[] = [
                    'transaction_id' => $dados['transaction_id'] ?? basename($arquivo, '.json'),
                    'gateway' => $dados['gateway'] ?? 'unknown',
                    'created_at' => $dados['created_at'] ?? null,
                    'paid_at' => $dados['paid_at'] ?? null,
                    'valor_centavos' => $dados['valor_centavos'] ?? 0,
                    'cotas' => $dados['cotas'] ?? 0,
                    'produto' => $dados['produto'] ?? 'Cotas',
                    'produto_id' => $dados['produto_id'] ?? null,
                    'webhook_status' => $dados['webhook_status'] ?? $dados['status'] ?? 'paid',
                    'customer' => [
                        'name' => $dados['customer']['name'] ?? 'Cliente',
                        'email' => $dados['customer']['email'] ?? '',
                        'phone' => $dados['customer']['phone'] ?? '',
                        'document' => $cpfTransacao
                    ]
                ];
            }
        }
    } catch (Exception $e) {
        // Ignora erros em arquivos individuais
        continue;
    }
}

// Ordena por data (mais recente primeiro)
usort($transacoesEncontradas, function($a, $b) {
    $dataA = strtotime($a['paid_at'] ?? $a['created_at'] ?? '1970-01-01');
    $dataB = strtotime($b['paid_at'] ?? $b['created_at'] ?? '1970-01-01');
    return $dataB - $dataA;
});

echo json_encode([
    'success' => true,
    'cpf' => substr($cpf, 0, 3) . '.***.***-' . substr($cpf, -2),
    'transacoes' => $transacoesEncontradas,
    'total' => count($transacoesEncontradas)
], JSON_UNESCAPED_UNICODE);