<?php
/**
 * ============================================
 * CLOAKER - APENAS BRASIL + MOBILE
 * ============================================
 * Inclua este arquivo no início das páginas PHP:
 * require_once __DIR__ . '/cloaker.php';
 * 
 * Ou use .htaccess para páginas HTML
 */

// Configurações
define('CLOAKER_ATIVO', true);                    // true = ativo, false = desativado
define('APENAS_BRASIL', true);                    // Bloquear acessos de fora do Brasil
define('APENAS_MOBILE', true);                    // Bloquear acessos de desktop
define('PAGINA_BLOQUEIO', '/rifas.html');         // Página para redirecionamento
define('PERMITIR_BOTS', false);                   // Permitir bots do Google, Facebook, etc
define('LOG_ACESSOS', true);                      // Salvar log de acessos bloqueados

// IPs que sempre passam (whitelist)
$whitelist = [
    '127.0.0.1',
    '::1',
    // Adicione IPs específicos aqui se precisar
];

// User agents de bots permitidos
$bots_permitidos = [
    'googlebot',
    'bingbot',
    'facebookexternalhit',
    'facebot',
    'twitterbot',
    'linkedinbot',
    'whatsapp',
    'telegrambot',
    'applebot',
    'slackbot'
];

/**
 * Obtém o IP real do visitante
 */
function getClientIP() {
    $ip = '';
    
    if (!empty($_SERVER['HTTP_CF_CONNECTING_IP'])) {
        // Cloudflare
        $ip = $_SERVER['HTTP_CF_CONNECTING_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        // Proxy
        $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
        $ip = trim($ips[0]);
    } elseif (!empty($_SERVER['HTTP_X_REAL_IP'])) {
        $ip = $_SERVER['HTTP_X_REAL_IP'];
    } elseif (!empty($_SERVER['REMOTE_ADDR'])) {
        $ip = $_SERVER['REMOTE_ADDR'];
    }
    
    return filter_var($ip, FILTER_VALIDATE_IP) ? $ip : '';
}

/**
 * Verifica se é um bot permitido
 */
function isAllowedBot() {
    global $bots_permitidos;
    
    if (!PERMITIR_BOTS) return false;
    
    $userAgent = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
    
    foreach ($bots_permitidos as $bot) {
        if (strpos($userAgent, $bot) !== false) {
            return true;
        }
    }
    
    return false;
}

/**
 * Verifica se é dispositivo móvel
 */
function isMobile() {
    $userAgent = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
    
    // Palavras-chave de dispositivos móveis
    $mobile_keywords = [
        'mobile',
        'android',
        'iphone',
        'ipod',
        'ipad',
        'windows phone',
        'blackberry',
        'bb10',
        'opera mini',
        'opera mobi',
        'webos',
        'palm',
        'symbian',
        'samsung',
        'nokia',
        'lg-',
        'htc',
        'mot-',
        'sony',
        'huawei',
        'xiaomi',
        'redmi',
        'oppo',
        'vivo',
        'realme',
        'oneplus'
    ];
    
    foreach ($mobile_keywords as $keyword) {
        if (strpos($userAgent, $keyword) !== false) {
            return true;
        }
    }
    
    // Verifica header específico de mobile
    if (isset($_SERVER['HTTP_X_WAP_PROFILE']) || isset($_SERVER['HTTP_PROFILE'])) {
        return true;
    }
    
    // Verifica Accept header para WAP
    $accept = strtolower($_SERVER['HTTP_ACCEPT'] ?? '');
    if (strpos($accept, 'application/vnd.wap') !== false) {
        return true;
    }
    
    return false;
}

/**
 * Verifica o país do IP usando múltiplas APIs
 */
function getCountry($ip) {
    // API 1: ip-api.com (gratuita, 45 req/min)
    $response = @file_get_contents("http://ip-api.com/json/{$ip}?fields=countryCode,status");
    if ($response) {
        $data = json_decode($response, true);
        if ($data && $data['status'] === 'success') {
            return $data['countryCode'] ?? '';
        }
    }
    
    // API 2: ipapi.co (gratuita, 1000 req/dia)
    $response = @file_get_contents("https://ipapi.co/{$ip}/country/");
    if ($response && strlen($response) === 2) {
        return $response;
    }
    
    // API 3: ipwho.is (gratuita, ilimitada)
    $response = @file_get_contents("https://ipwho.is/{$ip}");
    if ($response) {
        $data = json_decode($response, true);
        if ($data && $data['success']) {
            return $data['country_code'] ?? '';
        }
    }
    
    // Se todas falharem, permite acesso (fail-open)
    return 'BR';
}

/**
 * Salva log de acesso bloqueado
 */
function logBloqueio($ip, $pais, $motivo) {
    if (!LOG_ACESSOS) return;
    
    $logFile = __DIR__ . '/logs/cloaker_' . date('Y-m-d') . '.log';
    $logDir = dirname($logFile);
    
    if (!is_dir($logDir)) {
        @mkdir($logDir, 0755, true);
    }
    
    $log = sprintf(
        "[%s] IP: %s | País: %s | Motivo: %s | UA: %s | URL: %s\n",
        date('Y-m-d H:i:s'),
        $ip,
        $pais,
        $motivo,
        $_SERVER['HTTP_USER_AGENT'] ?? 'N/A',
        $_SERVER['REQUEST_URI'] ?? 'N/A'
    );
    
    @file_put_contents($logFile, $log, FILE_APPEND | LOCK_EX);
}

/**
 * Bloqueia o acesso
 */
function bloquearAcesso($ip, $pais, $motivo = 'Bloqueado') {
    logBloqueio($ip, $pais, $motivo);
    
    // Opção 1: Redirecionar para página de bloqueio
    if (file_exists($_SERVER['DOCUMENT_ROOT'] . PAGINA_BLOQUEIO)) {
        header('Location: ' . PAGINA_BLOQUEIO);
        exit;
    }
    
    // Opção 2: Mostrar página em branco (simula site offline)
    http_response_code(403);
    header('Content-Type: text/html; charset=utf-8');
    echo '<!DOCTYPE html><html><head><title>403</title></head><body></body></html>';
    exit;
}

// ========== EXECUÇÃO DO CLOAKER ==========

if (CLOAKER_ATIVO) {
    $ip = getClientIP();
    
    // Verifica whitelist
    if (in_array($ip, $whitelist)) {
        return; // Permite acesso
    }
    
    // Verifica se é bot permitido
    if (isAllowedBot()) {
        return; // Permite acesso
    }
    
    // Verifica dispositivo móvel
    if (APENAS_MOBILE && !isMobile()) {
        bloquearAcesso($ip, 'N/A', 'Acesso via desktop bloqueado');
    }
    
    // Verifica país
    if (APENAS_BRASIL) {
        $pais = getCountry($ip);
        
        if ($pais !== 'BR') {
            bloquearAcesso($ip, $pais, 'Acesso de fora do Brasil');
        }
    }
}