<?php
/**
 * ============================================
 * TRACKING.PHP - ANTI-DUPLICAÇÃO
 * ============================================
 * Recebe eventos do frontend e envia para Facebook CAPI
 * Com cache de event_id para evitar duplicação
 */

date_default_timezone_set('America/Sao_Paulo');

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/facebook_capi.php';

// ============================================
// LOG SIMPLES
// ============================================
function logTracking($message, $data = []) {
    $logDir = __DIR__ . '/logs';
    if (!is_dir($logDir)) @mkdir($logDir, 0755, true);
    
    $timestamp = date('Y-m-d H:i:s');
    $logLine = "[$timestamp] $message";
    if (!empty($data)) {
        $logLine .= ' | ' . json_encode($data, JSON_UNESCAPED_UNICODE);
    }
    @file_put_contents($logDir . '/tracking.log', $logLine . "\n", FILE_APPEND);
}

// ============================================
// ANTI-DUPLICAÇÃO
// ============================================
function isEventDuplicate($eventId) {
    if (empty($eventId)) return false;
    
    $cacheDir = __DIR__ . '/cache';
    if (!is_dir($cacheDir)) @mkdir($cacheDir, 0755, true);
    
    $cacheFile = $cacheDir . '/events_' . date('Y-m-d-H') . '.json';
    
    // Carrega cache
    $cache = [];
    if (file_exists($cacheFile)) {
        $cache = json_decode(file_get_contents($cacheFile), true) ?: [];
    }
    
    // Verifica se já existe
    if (isset($cache[$eventId])) {
        return true;
    }
    
    // Adiciona ao cache
    $cache[$eventId] = time();
    
    // Limpa eventos antigos (mais de 10 minutos)
    $now = time();
    foreach ($cache as $id => $time) {
        if ($now - $time > 600) {
            unset($cache[$id]);
        }
    }
    
    // Salva cache
    @file_put_contents($cacheFile, json_encode($cache));
    
    // Limpa arquivos de cache antigos
    $oldFiles = glob($cacheDir . '/events_*.json');
    $currentFile = basename($cacheFile);
    foreach ($oldFiles as $file) {
        if (basename($file) !== $currentFile) {
            @unlink($file);
        }
    }
    
    return false;
}

// ============================================
// RECEBE DADOS
// ============================================
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || empty($input['event'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Evento não especificado']);
    exit;
}

$event = $input['event'];
$eventId = $input['event_id'] ?? null;

// ANTI-DUPLICAÇÃO: Se já processou este event_id, ignora
if (isEventDuplicate($eventId)) {
    logTracking("DUPLICADO ignorado: $event", ['event_id' => $eventId]);
    echo json_encode(['success' => true, 'skipped' => true, 'reason' => 'duplicate']);
    exit;
}

$sourceUrl = $input['source_url'] ?? '';
$value = floatval($input['value'] ?? 0);
$numItems = intval($input['num_items'] ?? 1);

// Cookies do Facebook
$fbc = $input['fbc'] ?? $_COOKIE['_fbc'] ?? null;
$fbp = $input['fbp'] ?? $_COOKIE['_fbp'] ?? null;

// Valida fbc
if (!empty($fbc)) {
    $fbcParts = explode('.', $fbc);
    if (count($fbcParts) < 4) {
        $fbc = null;
    }
}

// Gera fbp se não existir
if (empty($fbp)) {
    $fbp = 'fb.1.' . round(microtime(true) * 1000) . '.' . mt_rand(1000000000, 9999999999);
}

// IP do cliente
$clientIp = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['HTTP_X_REAL_IP'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
if (strpos($clientIp, ',') !== false) {
    $clientIp = trim(explode(',', $clientIp)[0]);
}

// Dados do usuário
$userData = [
    'email' => $input['email'] ?? null,
    'phone' => $input['phone'] ?? null,
    'name' => $input['name'] ?? null,
    'document' => $input['document'] ?? null,
    'fbc' => $fbc,
    'fbp' => $fbp,
    'user_agent' => $input['user_agent'] ?? $_SERVER['HTTP_USER_AGENT'] ?? '',
    'ip' => $clientIp,
    'content_name' => $input['content_name'] ?? (defined('PRODUTO_NOME') ? PRODUTO_NOME : 'Produto'),
    'content_id' => $input['content_id'] ?? (defined('PRODUTO_ID') ? PRODUTO_ID : 'produto-001')
];

logTracking("Recebido: $event", [
    'event_id' => $eventId,
    'has_fbc' => !empty($fbc),
    'has_fbp' => !empty($fbp),
    'has_email' => !empty($input['email'])
]);

// ============================================
// ENVIA PARA FACEBOOK CAPI
// ============================================
$fb = new FacebookCAPI();
$result = ['success' => false];

switch ($event) {
    case 'PageView':
        $result = $fb->pageView($userData, $sourceUrl, $eventId);
        break;
    
    case 'ViewContent':
        $result = $fb->viewContent($userData, $value, $sourceUrl, $eventId);
        break;
    
    case 'AddToCart':
        $result = $fb->addToCart($userData, $value, $numItems, $sourceUrl, $eventId);
        break;
        
    case 'InitiateCheckout':
        $result = $fb->initiateCheckout($userData, $value, $numItems, $sourceUrl, $eventId);
        break;
        
    case 'AddPaymentInfo':
        $result = $fb->addPaymentInfo($userData, $value, $sourceUrl, $eventId);
        break;
        
    case 'Purchase':
        $orderId = $input['order_id'] ?? uniqid('order_');
        $result = $fb->purchase($userData, $value, $orderId, $numItems, $sourceUrl, $eventId);
        break;
        
    default:
        $customData = [
            'currency' => 'BRL',
            'value' => $value,
            'content_name' => $userData['content_name']
        ];
        $result = $fb->customEvent($event, $userData, $customData, $sourceUrl, $eventId);
}

logTracking("$event enviado", [
    'success' => $result['success'] ?? false,
    'event_id' => $eventId,
    'http_code' => $result['http_code'] ?? null
]);

echo json_encode($result);