/**
 * ============================================
 * SMART TRACKING v5 - SEM DUPLICAÇÃO
 * ============================================
 * 
 * CORREÇÕES:
 * - Removida duplicação de eventos
 * - Anti-duplicação por event_id
 * - Melhor sincronização Pixel + CAPI
 * 
 * INSTALAÇÃO:
 * <script>
 *   !function(f,b,e,v,n,t,s){...}  // Código do Pixel
 *   fbq('init', '3061241000727171');
 * </script>
 * <script src="/api/smart-tracking.js"></script>
 * <script>
 *   fbq('track', 'PageView');
 * </script>
 */

(function() {
    'use strict';
    
    // ============================================
    // CONFIGURAÇÃO
    // ============================================
    var CONFIG = {
        trackingEndpoint: '/api/tracking.php',  // ✅ Caminho absoluto corrigido
        debug: false
    };
    
    // Cache de eventos já enviados (anti-duplicação)
    var sentEvents = {};
    
    // ============================================
    // UTILITÁRIOS
    // ============================================
    
    function log(msg, data) {
        if (CONFIG.debug) {
            console.log('[SmartTracking]', msg, data || '');
        }
    }
    
    function generateEventId(eventName) {
        var timestamp = Date.now();
        var random = Math.random().toString(36).substring(2, 11);
        return eventName.toLowerCase() + '_' + random + '_' + timestamp;
    }
    
    function getFbCookies() {
        var cookies = {};
        document.cookie.split(';').forEach(function(c) {
            var parts = c.trim().split('=');
            if (parts.length === 2) {
                cookies[parts[0]] = parts[1];
            }
        });
        
        var fbc = cookies._fbc || null;
        var fbp = cookies._fbp || null;
        
        // Cria fbc a partir do fbclid se não existir
        if (!fbc) {
            var params = new URLSearchParams(window.location.search);
            var fbclid = params.get('fbclid');
            if (fbclid) {
                fbc = 'fb.1.' + Date.now() + '.' + fbclid;
                document.cookie = '_fbc=' + fbc + '; max-age=7776000; path=/; SameSite=Lax';
            }
        }
        
        // Cria fbp se não existir
        if (!fbp) {
            fbp = 'fb.1.' + Date.now() + '.' + Math.floor(Math.random() * 10000000000);
            document.cookie = '_fbp=' + fbp + '; max-age=7776000; path=/; SameSite=Lax';
        }
        
        return { fbc: fbc, fbp: fbp };
    }
    
    function getUtms() {
        var params = new URLSearchParams(window.location.search);
        
        var utms = {
            utm_source: params.get('utm_source') || sessionStorage.getItem('utm_source') || '',
            utm_medium: params.get('utm_medium') || sessionStorage.getItem('utm_medium') || '',
            utm_campaign: params.get('utm_campaign') || sessionStorage.getItem('utm_campaign') || '',
            utm_content: params.get('utm_content') || sessionStorage.getItem('utm_content') || '',
            utm_term: params.get('utm_term') || sessionStorage.getItem('utm_term') || ''
        };
        
        // Salva no sessionStorage para persistir na sessão
        Object.keys(utms).forEach(function(key) {
            if (utms[key]) {
                sessionStorage.setItem(key, utms[key]);
            }
        });
        
        return utms;
    }
    
    function getClientData() {
        try {
            var stored = JSON.parse(localStorage.getItem('cliente') || '{}');
            
            var emailField = document.querySelector('input[name="email"], input[type="email"], #email');
            var phoneField = document.querySelector('input[name="celular"], input[name="telefone"], input[name="phone"], #celular, #telefone');
            var nameField = document.querySelector('input[name="nome"], input[name="name"], #nome');
            var cpfField = document.querySelector('input[name="cpf"], input[name="document"], #cpf');
            
            return {
                email: (emailField && emailField.value) || stored.email || null,
                phone: (phoneField && phoneField.value) || stored.celular || stored.telefone || null,
                name: (nameField && nameField.value) || stored.nome || null,
                document: (cpfField && cpfField.value) || stored.cpf || null
            };
        } catch(e) {
            return {};
        }
    }
    
    function getProductValue() {
        try {
            // 1. localStorage (em centavos)
            var valorCentavos = localStorage.getItem('valor');
            if (valorCentavos && parseInt(valorCentavos) > 0) {
                return parseInt(valorCentavos) / 100;
            }
            
            // 2. Elemento de preço na página
            var valorElement = document.querySelector('#txtValor, #txtValorMob, .precoNew, .mobValor');
            if (valorElement) {
                var texto = valorElement.textContent || '';
                var numero = texto.replace(/[^\d,]/g, '').replace(',', '.');
                if (numero && parseFloat(numero) > 0) {
                    return parseFloat(numero);
                }
            }
            
            // 3. URL param
            var params = new URLSearchParams(window.location.search);
            var valorUrl = params.get('valor');
            if (valorUrl && parseInt(valorUrl) > 0) {
                return parseInt(valorUrl) / 100;
            }
            
            return 0;
        } catch(e) {
            return 0;
        }
    }
    
    function getQuantity() {
        try {
            var cotas = localStorage.getItem('cotas');
            if (cotas && parseInt(cotas) > 0) {
                return parseInt(cotas);
            }
            
            var qtdElement = document.querySelector('#inpQtd, #inpQtdMob');
            if (qtdElement && qtdElement.value) {
                return parseInt(qtdElement.value) || 1;
            }
            
            return 1;
        } catch(e) {
            return 1;
        }
    }
    
    // ============================================
    // ENVIO PARA O CAPI (COM ANTI-DUPLICAÇÃO)
    // ============================================
    
    function sendToCAPI(eventName, eventId, customData) {
        // ANTI-DUPLICAÇÃO: Se já enviou este event_id, ignora
        if (sentEvents[eventId]) {
            log('⚠️ Evento já enviado, ignorando:', eventId);
            return Promise.resolve(eventId);
        }
        
        // Marca como enviado
        sentEvents[eventId] = true;
        
        // Limpa cache antigo (mais de 5 minutos)
        var now = Date.now();
        Object.keys(sentEvents).forEach(function(id) {
            var timestamp = parseInt(id.split('_').pop());
            if (now - timestamp > 300000) {
                delete sentEvents[id];
            }
        });
        
        customData = customData || {};
        
        var fbCookies = getFbCookies();
        var utms = getUtms();
        var clientData = getClientData();
        
        // Valor automático se não fornecido
        var value = customData.value;
        if (value === undefined || value === null || value === 0) {
            value = getProductValue();
        }
        
        var numItems = customData.num_items || customData.quantity || getQuantity();
        
        var payload = {
            event: eventName,
            event_id: eventId,
            source_url: window.location.href,
            user_agent: navigator.userAgent,
            fbc: fbCookies.fbc,
            fbp: fbCookies.fbp,
            email: clientData.email,
            phone: clientData.phone,
            name: clientData.name,
            document: clientData.document,
            utm_source: utms.utm_source,
            utm_medium: utms.utm_medium,
            utm_campaign: utms.utm_campaign,
            utm_content: utms.utm_content,
            utm_term: utms.utm_term,
            value: parseFloat(value) || 0,
            currency: 'BRL',
            content_name: customData.content_name || 'Viva Sorte+',
            content_id: (customData.content_ids && customData.content_ids[0]) || customData.content_id || 'vivasorte-001',
            num_items: parseInt(numItems) || 1
        };
        
        // Remove campos vazios
        Object.keys(payload).forEach(function(key) {
            if (payload[key] === null || payload[key] === undefined || payload[key] === '') {
                delete payload[key];
            }
        });
        
        log('📤 CAPI:', { event: eventName, event_id: eventId, value: payload.value });
        
        return new Promise(function(resolve) {
            fetch(CONFIG.trackingEndpoint, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload),
                keepalive: true
            }).then(function(response) {
                if (response.ok) {
                    log('✅ CAPI OK:', eventName);
                }
                resolve(eventId);
            }).catch(function(err) {
                log('❌ CAPI erro:', err);
                resolve(eventId);
            });
        });
    }
    
    // ============================================
    // INTERCEPTA O PIXEL DO FACEBOOK
    // ============================================
    
    var originalFbq = window.fbq;
    
    function newFbq() {
        var args = Array.prototype.slice.call(arguments);
        var action = args[0];
        var eventName = args[1];
        var eventData = args[2] || {};
        var options = args[3] || {};
        
        // Se for track ou trackCustom, intercepta
        if ((action === 'track' || action === 'trackCustom') && eventName) {
            // Gera ou usa eventID existente
            var eventId = options.eventID || generateEventId(eventName);
            
            // Garante eventID nas options para o Pixel
            if (!options.eventID) {
                options.eventID = eventId;
                args[3] = options;
            }
            
            // Adiciona value/currency se necessário
            var needsValue = ['ViewContent', 'AddToCart', 'InitiateCheckout', 'AddPaymentInfo', 'Purchase'].indexOf(eventName) !== -1;
            
            if (needsValue) {
                if (!eventData.value || eventData.value === 0) {
                    eventData.value = getProductValue();
                }
                if (!eventData.currency) {
                    eventData.currency = 'BRL';
                }
                if (!eventData.num_items) {
                    eventData.num_items = getQuantity();
                }
                args[2] = eventData;
            }
            
            log('📡 Pixel:', { event: eventName, event_id: eventId });
            
            // Envia para CAPI (com anti-duplicação interno)
            sendToCAPI(eventName, eventId, eventData);
        }
        
        // Chama função original do Pixel
        if (originalFbq) {
            return originalFbq.apply(window, args);
        }
    }
    
    // Copia propriedades do fbq original
    if (originalFbq) {
        for (var key in originalFbq) {
            if (originalFbq.hasOwnProperty(key)) {
                newFbq[key] = originalFbq[key];
            }
        }
        if (originalFbq.queue) newFbq.queue = originalFbq.queue;
        if (originalFbq.loaded !== undefined) newFbq.loaded = originalFbq.loaded;
        if (originalFbq.version) newFbq.version = originalFbq.version;
        if (originalFbq.push) newFbq.push = newFbq;
    }
    
    // Substitui fbq global
    window.fbq = newFbq;
    
    // ============================================
    // API PÚBLICA
    // ============================================
    
    window.SmartTracking = {
        /**
         * Envia evento para Pixel + CAPI
         * O mesmo event_id é usado nos dois para deduplicação correta
         */
        sendEvent: function(eventName, data) {
            data = data || {};
            var eventId = generateEventId(eventName);
            
            // Prepara dados
            if (!data.value || data.value === 0) {
                data.value = getProductValue();
            }
            if (!data.currency) {
                data.currency = 'BRL';
            }
            if (!data.num_items) {
                data.num_items = getQuantity();
            }
            
            // Envia para Pixel (que vai interceptar e enviar para CAPI também)
            if (typeof window.fbq !== 'undefined') {
                // Usa a função ORIGINAL para não duplicar
                // O interceptor vai pegar e enviar para CAPI
                window.fbq('track', eventName, data, { eventID: eventId });
            } else {
                // Se não tem Pixel, envia só para CAPI
                sendToCAPI(eventName, eventId, data);
            }
            
            return Promise.resolve(eventId);
        },
        
        // Aliases
        track: function(eventName, data) {
            return this.sendEvent(eventName, data);
        },
        
        // Só CAPI (sem Pixel)
        trackCAPI: function(eventName, data) {
            data = data || {};
            var eventId = generateEventId(eventName);
            return sendToCAPI(eventName, eventId, data);
        },
        
        // Utilitários
        getFbCookies: getFbCookies,
        getUtms: getUtms,
        getClientData: getClientData,
        getProductValue: getProductValue,
        getQuantity: getQuantity,
        
        init: function() {
            getFbCookies();
            log('✅ Inicializado');
        }
    };
    
    // Inicializa
    SmartTracking.init();
    
    log('✅ Smart Tracking v5 carregado');
    
})();