<?php
/**
 * ============================================
 * CONFIGURAÇÃO CENTRAL - MULTI-GATEWAY
 * ============================================
 * Lê configurações do gateways.json (editável pelo painel)
 */

// Configurações do produto
define('PRODUTO_NOME', 'Cotas');
define('PRODUTO_ID', 'premios-001');
define('SITE_URL', 'https://vivasorteaoficial.com/');

// URLs
define('WEBHOOK_URL', 'https://vivasorteaoficial.com/api/webhook.php');
define('UTM_STORAGE_DIR', __DIR__ . '/utm_data/');

// Arquivo de configuração dos gateways
define('GATEWAYS_CONFIG_FILE', __DIR__ . '/gateways.json');

// ============================================
// CARREGA CONFIGURAÇÃO DOS GATEWAYS
// ============================================

function loadGatewaysConfig() {
    static $config = null;
    
    if ($config !== null) {
        return $config;
    }
    
    if (file_exists(GATEWAYS_CONFIG_FILE)) {
        $json = file_get_contents(GATEWAYS_CONFIG_FILE);
        $config = json_decode($json, true);
        
        if ($config) {
            return $config;
        }
    }
    
    // Fallback para configuração padrão se o arquivo não existir
    $config = [
        'gateway_ativo' => 'umbrella',
        'gateways' => [
            'umbrella' => [
                'enabled' => true,
                'name' => 'Umbrella',
                'api_url' => 'https://api-gateway.umbrellapag.com',
                'api_key' => 'd46fa92f-9484-409f-a82a-e2734029979e',
                'webhook_secret' => '',
                'user_agent' => 'UMBRELLAB2B/1.0',
                'auth_header' => 'x-api-key',
                'endpoints' => [
                    'pix_create' => '/api/user/transactions',
                    'pix_status' => '/api/user/transactions'
                ],
                'status_field' => 'data.data[0].status',
                'qrcode_field' => 'data.qrCode',
                'txid_field' => 'data.id'
            ]
        ],
        'status_map' => [
            'PAID' => 'paid',
            'AUTHORIZED' => 'paid',
            'CONCLUIDA' => 'paid',
            'APPROVED' => 'paid',
            'COMPLETED' => 'paid',
            'CONFIRMED' => 'paid',
            'PENDING' => 'pending',
            'WAITING' => 'pending',
            'WAITING_PAYMENT' => 'pending',
            'PROCESSING' => 'pending',
            'ATIVA' => 'pending',
            'EXPIRED' => 'expired',
            'CANCELLED' => 'cancelled',
            'CANCELED' => 'cancelled',
            'REFUSED' => 'cancelled',
            'CHARGEDBACK' => 'refunded',
            'REFUNDED' => 'refunded',
            'IN_PROTEST' => 'pending'
        ]
    ];
    
    return $config;
}

/**
 * Retorna o nome do gateway ativo
 */
function getGatewayName() {
    $config = loadGatewaysConfig();
    return $config['gateway_ativo'] ?? 'umbrella';
}

/**
 * Retorna a configuração do gateway ativo
 */
function getGatewayConfig() {
    $config = loadGatewaysConfig();
    $gatewayName = $config['gateway_ativo'] ?? 'umbrella';
    
    return $config['gateways'][$gatewayName] ?? null;
}

/**
 * Retorna todos os gateways configurados
 */
function getAllGateways() {
    $config = loadGatewaysConfig();
    return $config['gateways'] ?? [];
}

/**
 * Retorna o mapeamento de status
 */
function getStatusMap() {
    $config = loadGatewaysConfig();
    return $config['status_map'] ?? [];
}

/**
 * Mapeia status da API para status interno
 */
function mapStatus($apiStatus) {
    $statusMap = getStatusMap();
    $statusUpper = strtoupper($apiStatus);
    
    return $statusMap[$statusUpper] ?? 'unknown';
}

/**
 * Verifica se o status indica pagamento confirmado
 */
function isPaidStatus($status) {
    $paidStatuses = ['paid', 'PAID', 'APPROVED', 'AUTHORIZED', 'COMPLETED', 'CONFIRMED', 'CONCLUIDA'];
    return in_array(strtoupper($status), array_map('strtoupper', $paidStatuses));
}

/**
 * Extrai valor de resposta usando path como "data.data[0].status"
 */
function extractByPath($data, $path) {
    if (!$path || !$data) return null;
    
    $parts = preg_split('/\./', $path);
    $current = $data;
    
    foreach ($parts as $part) {
        // Verifica se é acesso a array tipo "data[0]"
        if (preg_match('/^(.+)\[(\d+)\]$/', $part, $matches)) {
            $key = $matches[1];
            $index = (int)$matches[2];
            
            if (!isset($current[$key]) || !is_array($current[$key])) {
                return null;
            }
            $current = $current[$key][$index] ?? null;
        } else {
            if (!isset($current[$part])) {
                return null;
            }
            $current = $current[$part];
        }
        
        if ($current === null) return null;
    }
    
    return $current;
}

// ============================================
// FUNÇÕES AUXILIARES
// ============================================

/**
 * Validação de origem (anti-clone)
 */
function validarOrigem() {
    // Lista de domínios permitidos
    $allowedOrigins = [
        'vivasorteaoficial.com/',
        'www.vivasorteaoficial.com/',
        'localhost',
        '127.0.0.1'
    ];
    
    $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
    $referer = $_SERVER['HTTP_REFERER'] ?? '';
    $host = parse_url($origin ?: $referer, PHP_URL_HOST) ?? '';
    
    // Em produção, descomente para ativar proteção anti-clone
    /*
    if (!empty($host) && !in_array($host, $allowedOrigins)) {
        http_response_code(403);
        echo json_encode(['success' => false, 'error' => 'Acesso negado']);
        exit;
    }
    */
}

/**
 * Headers CORS
 */
function setCorsHeaders() {
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, Authorization');
    header('Content-Type: application/json; charset=utf-8');
    
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        http_response_code(200);
        exit;
    }
}

/**
 * Log de checkout
 */
function logCheckout($file, $message, $data = []) {
    $logDir = __DIR__ . '/logs';
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $logFile = $logDir . '/' . $file;
    $timestamp = date('Y-m-d H:i:s');
    
    $logLine = "[$timestamp] $message";
    if (!empty($data)) {
        $logLine .= ' | ' . json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }
    $logLine .= "\n";
    
    file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);
}

/**
 * Converte string vazia para null
 */
function emptyToNull($value) {
    return ($value === '' || $value === null) ? null : $value;
}

/**
 * Gera referência única para transação
 */
function generateTxRef($prefix = 'TX') {
    return $prefix . time() . rand(1000, 9999);
}

/**
 * Formata valor para centavos
 */
function toCents($value) {
    return intval(floatval($value) * 100);
}

/**
 * Formata centavos para reais
 */
function toReais($cents) {
    return number_format($cents / 100, 2, ',', '.');
}

// Cria diretório de UTM se não existir
if (!is_dir(UTM_STORAGE_DIR)) {
    @mkdir(UTM_STORAGE_DIR, 0755, true);
}