<?php
/**
 * ============================================
 * WEBHOOK - NOTIFICAÇÕES DE PAGAMENTO
 * ============================================
 * Recebe notificações de todos os gateways
 */

require_once __DIR__ . '/config.php';

// Carrega Facebook CAPI
$fbCapiFile = __DIR__ . '/facebook_capi.php';
if (file_exists($fbCapiFile)) {
    require_once $fbCapiFile;
    logCheckout('webhook.log', 'FacebookCAPI carregado');
} else {
    logCheckout('webhook.log', 'AVISO: facebook_capi.php não encontrado', ['path' => $fbCapiFile]);
}

// Carrega UTMify
$utmifyFile = __DIR__ . '/utmify.php';
if (file_exists($utmifyFile)) {
    require_once $utmifyFile;
}

// Log da requisição
$rawInput = file_get_contents('php://input');
$data = json_decode($rawInput, true);

logCheckout('webhook.log', 'Webhook recebido', [
    'method' => $_SERVER['REQUEST_METHOD'],
    'content_type' => $_SERVER['CONTENT_TYPE'] ?? '',
    'raw_length' => strlen($rawInput),
    'data_preview' => substr($rawInput, 0, 500)
]);

if (!$data) {
    http_response_code(400);
    echo json_encode(['error' => 'Payload inválido']);
    exit;
}

// ============================================
// DETECTA O GATEWAY AUTOMATICAMENTE
// ============================================
function detectGateway($data) {
    // Genesys: tem external_id e total_amount
    if (isset($data['external_id']) && isset($data['total_amount'])) {
        return 'genesys';
    }
    
    // AllowPay: tem objectId e type
    if (isset($data['objectId']) && isset($data['type'])) {
        return 'allowpay';
    }
    
    // Umbrella: tem objectId mas estrutura diferente
    if (isset($data['objectId']) && isset($data['data']['data'])) {
        return 'umbrella';
    }
    
    // TechByNet: tem transaction_id
    if (isset($data['transaction_id']) || isset($data['pix_code'])) {
        return 'techbynet';
    }
    
    // Fallback: usa gateway ativo
    return getGatewayName();
}

$detectedGateway = detectGateway($data);

logCheckout('webhook.log', 'Gateway detectado', ['gateway' => $detectedGateway]);

// ============================================
// EXTRAI DADOS DO WEBHOOK
// ============================================
$txId = null;
$status = null;
$amount = null;

if ($detectedGateway === 'genesys') {
    // Genesys Finance - formato simples
    $txId = $data['id'] ?? null;
    $status = $data['status'] ?? null;
    // total_amount está em REAIS, converter para centavos
    $amount = isset($data['total_amount']) ? intval($data['total_amount'] * 100) : null;
    
} elseif ($detectedGateway === 'allowpay') {
    // AllowPay
    $txId = $data['data']['id'] ?? $data['objectId'] ?? null;
    $status = $data['data']['status'] ?? null;
    $amount = $data['data']['amount'] ?? null;
    
} elseif ($detectedGateway === 'umbrella') {
    // Umbrella
    $txData = $data['data']['data'][0] ?? $data['data'] ?? [];
    $txId = $txData['id'] ?? $data['objectId'] ?? null;
    $status = $txData['status'] ?? null;
    $amount = $txData['amount'] ?? null;
    
} elseif ($detectedGateway === 'techbynet') {
    // TechByNet
    $txData = $data['data']['data'] ?? $data['data'] ?? $data;
    $txId = $txData['id'] ?? $data['transaction_id'] ?? null;
    $status = $txData['status'] ?? null;
    $amount = $txData['amount'] ?? null;
    
} else {
    // Genérico
    $txId = $data['data']['id'] ?? $data['id'] ?? $data['objectId'] ?? null;
    $status = $data['data']['status'] ?? $data['status'] ?? null;
    $amount = $data['data']['amount'] ?? $data['amount'] ?? null;
}

logCheckout('webhook.log', 'Dados extraídos', [
    'txId' => $txId,
    'status' => $status,
    'amount' => $amount
]);

if (!$txId) {
    logCheckout('webhook.log', 'ERRO: txId não encontrado');
    http_response_code(400);
    echo json_encode(['error' => 'Transaction ID não encontrado']);
    exit;
}

// ============================================
// VERIFICA SE É PAGAMENTO CONFIRMADO
// ============================================
$isPaid = isPaidStatus($status);

if ($isPaid) {
    logCheckout('webhook.log', 'PAGAMENTO CONFIRMADO', [
        'txId' => $txId,
        'status' => $status,
        'amount' => $amount
    ]);
    
    // ============================================
    // CARREGA DADOS DO UTM (tenta múltiplos IDs)
    // ============================================
    $utmData = [];
    $utmFile = null;
    
    // IDs possíveis para buscar o arquivo
    $possibleIds = array_filter([
        $txId,                                    // ID principal
        $data['external_id'] ?? null,             // External ID do Genesys
        $data['data']['id'] ?? null,              // ID em data.id
        $data['objectId'] ?? null,                // AllowPay/Umbrella
        $data['transaction_id'] ?? null           // TechByNet
    ]);
    
    logCheckout('webhook.log', 'Buscando arquivo UTM', ['possible_ids' => $possibleIds]);
    
    foreach ($possibleIds as $checkId) {
        $checkFile = UTM_STORAGE_DIR . $checkId . '.json';
        if (file_exists($checkFile)) {
            $utmFile = $checkFile;
            $utmData = json_decode(file_get_contents($checkFile), true) ?: [];
            logCheckout('webhook.log', 'UTM data carregado', [
                'file' => $checkFile,
                'found_with_id' => $checkId
            ]);
            break;
        }
    }
    
    // Se não encontrou por ID, tenta buscar pelo transaction_id salvo no arquivo
    if (empty($utmData)) {
        $allFiles = glob(UTM_STORAGE_DIR . '*.json');
        foreach ($allFiles as $file) {
            $fileData = json_decode(file_get_contents($file), true);
            if ($fileData && isset($fileData['transaction_id'])) {
                if (in_array($fileData['transaction_id'], $possibleIds)) {
                    $utmFile = $file;
                    $utmData = $fileData;
                    logCheckout('webhook.log', 'UTM data encontrado por transaction_id interno', [
                        'file' => $file
                    ]);
                    break;
                }
            }
        }
    }
    
    if (empty($utmData)) {
        logCheckout('webhook.log', 'AVISO: Arquivo UTM não encontrado', [
            'tried_ids' => $possibleIds,
            'utm_dir' => UTM_STORAGE_DIR,
            'dir_exists' => is_dir(UTM_STORAGE_DIR)
        ]);
    }
    
    // ============================================
    // FACEBOOK CAPI - PURCHASE
    // ============================================
    if (class_exists('FacebookCAPI')) {
        try {
            $fb = new FacebookCAPI();
            
            // Se não temos dados UTM, tenta extrair do webhook do Genesys
            if (empty($utmData) && $detectedGateway === 'genesys') {
                $customer = $data['customer'] ?? [];
                $utmData = [
                    'customer' => [
                        'name' => $customer['name'] ?? '',
                        'email' => $customer['email'] ?? '',
                        'phone' => $customer['phone'] ?? '',
                        'document' => $customer['document'] ?? ''
                    ],
                    'valor_centavos' => $amount,
                    'produto' => 'Cotas',
                    'cotas' => 1
                ];
                logCheckout('webhook.log', 'Usando dados do webhook Genesys para Facebook');
            }
            
            $valorReais = ($amount ?: $utmData['valor_centavos'] ?? 0) / 100;
            
            logCheckout('webhook.log', 'Enviando Facebook CAPI Purchase', [
                'txId' => $txId,
                'valorReais' => $valorReais,
                'has_customer' => !empty($utmData['customer']),
                'has_fbc' => !empty($utmData['fbc']),
                'has_fbp' => !empty($utmData['fbp'])
            ]);
            
            $result = $fb->sendPurchase($txId, $valorReais, $utmData);
            
            logCheckout('webhook.log', 'Facebook CAPI Purchase resultado', [
                'success' => $result['success'] ?? false,
                'event_id' => $result['event_id'] ?? null,
                'events_received' => $result['events_received'] ?? 0
            ]);
        } catch (Exception $e) {
            logCheckout('webhook.log', 'Erro Facebook CAPI: ' . $e->getMessage());
        }
    } else {
        logCheckout('webhook.log', 'AVISO: Classe FacebookCAPI não encontrada');
    }
    
    // ============================================
    // UTMIFY - PAID
    // ============================================
    if (class_exists('Utmify')) {
        try {
            $utmify = new Utmify();
            
            // Prepara dados do cliente
            $customer = [
                'name' => $utmData['customer']['name'] ?? $utmData['nome'] ?? '',
                'email' => $utmData['customer']['email'] ?? $utmData['email'] ?? '',
                'phone' => $utmData['customer']['phone'] ?? $utmData['telefone'] ?? '',
                'document' => $utmData['customer']['document'] ?? $utmData['cpf'] ?? '',
                'ip' => $utmData['customer_ip'] ?? $_SERVER['REMOTE_ADDR'] ?? ''
            ];
            
            // Prepara dados do produto
            $product = [
                'id' => $utmData['produto_id'] ?? (defined('PRODUTO_ID') ? PRODUTO_ID : 'produto-001'),
                'name' => $utmData['produto'] ?? (defined('PRODUTO_NOME') ? PRODUTO_NOME : 'Produto'),
                'quantity' => $utmData['cotas'] ?? 1
            ];
            
            // UTMs
            $utms = [
                'utm_source' => $utmData['utm_source'] ?? '',
                'utm_medium' => $utmData['utm_medium'] ?? '',
                'utm_campaign' => $utmData['utm_campaign'] ?? '',
                'utm_content' => $utmData['utm_content'] ?? '',
                'utm_term' => $utmData['utm_term'] ?? '',
                'sck' => $utmData['sck'] ?? '',
                'src' => $utmData['src'] ?? ''
            ];
            
            $utmifyResult = $utmify->sendPaid(
                $txId,
                $amount ?: $utmData['valor_centavos'] ?? 0,
                $customer,
                $product,
                $utms
            );
            
            logCheckout('webhook.log', 'UTMify Paid enviado', $utmifyResult);
        } catch (Exception $e) {
            logCheckout('webhook.log', 'Erro UTMify: ' . $e->getMessage());
        }
    }
    
    // ============================================
    // MARCA COMO PROCESSADO
    // ============================================
    $utmData['status'] = 'paid';
    $utmData['paid_at'] = date('Y-m-d H:i:s');
    $utmData['webhook_status'] = $status;
    $utmData['gateway_response'] = [
        'txId' => $txId,
        'amount' => $amount,
        'gateway' => $detectedGateway
    ];
    
    // Salva no arquivo original ou cria um novo
    if ($utmFile) {
        @file_put_contents($utmFile, json_encode($utmData, JSON_PRETTY_PRINT));
        logCheckout('webhook.log', 'Arquivo UTM atualizado', ['file' => $utmFile]);
    } else {
        // Cria novo arquivo com o txId
        $newFile = UTM_STORAGE_DIR . $txId . '.json';
        @file_put_contents($newFile, json_encode($utmData, JSON_PRETTY_PRINT));
        logCheckout('webhook.log', 'Novo arquivo UTM criado', ['file' => $newFile]);
    }
}

// Resposta de sucesso
http_response_code(200);
echo json_encode([
    'success' => true,
    'txId' => $txId,
    'status' => $status,
    'isPaid' => $isPaid,
    'gateway' => $detectedGateway
]);
