<?php
/**
 * ============================================
 * PIX API - GERAÇÃO DE PAGAMENTO
 * ============================================
 * Suporta múltiplos gateways via gateways.json
 */

require_once __DIR__ . '/config.php';
@include_once __DIR__ . '/facebook_capi.php';
@include_once __DIR__ . '/utmify.php';

setCorsHeaders();

$input = json_decode(file_get_contents('php://input'), true);

if (!$input) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Dados inválidos']);
    exit;
}

// Validação - aceita formato antigo (cliente) ou novo (campos diretos)
$cliente = $input['cliente'] ?? null;

if ($cliente) {
    // Formato antigo: { cliente: {...}, tracking: {...} }
    $nome = $cliente['nome'] ?? '';
    $cpf = preg_replace('/\D/', '', $cliente['cpf'] ?? '');
    $email = $cliente['email'] ?? '';
    $telefone = preg_replace('/\D/', '', $cliente['celular'] ?? '');
    $cotas = $cliente['cotas'] ?? 1;
    $valorCentavos = intval($input['valor'] ?? 0);
    $tracking = $input['tracking'] ?? [];
} else {
    // Formato novo: campos diretos
    $nome = trim($input['nome'] ?? '');
    $cpf = preg_replace('/\D/', '', $input['cpf'] ?? '');
    $email = trim($input['email'] ?? '');
    $telefone = preg_replace('/\D/', '', $input['telefone'] ?? '');
    $cotas = $input['cotas'] ?? 1;
    $valorCentavos = intval($input['valor'] ?? 990);
    $tracking = $input['tracking'] ?? [];
}

// ============================================
// PRODUTO DINÂMICO (para upsells)
// ============================================
// Aceita produto via payload ou usa padrão do config
$produto = $input['produto'] ?? [];
$produtoNome = $produto['nome'] ?? $input['produto_nome'] ?? PRODUTO_NOME;
$produtoId = $produto['id'] ?? $input['produto_id'] ?? PRODUTO_ID;
$produtoDescricao = $produto['descricao'] ?? $input['produto_descricao'] ?? null;

// Validações
if (strlen($nome) < 3) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Nome inválido']);
    exit;
}
if (strlen($cpf) !== 11) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'CPF inválido']);
    exit;
}
if ($valorCentavos <= 0) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Valor inválido']);
    exit;
}

$valorReais = $valorCentavos / 100;

logCheckout('pix.log', 'Criando PIX', [
    'valor' => $valorReais,
    'cliente' => $nome,
    'cotas' => $cotas,
    'produto' => $produtoNome,
    'produto_id' => $produtoId
]);

try {
    // ============================================
    // OBTÉM CONFIGURAÇÃO DO GATEWAY ATIVO
    // ============================================
    $gateway = getGatewayConfig();
    $gatewayName = getGatewayName();
    
    if (!$gateway) {
        throw new Exception("Gateway '$gatewayName' não configurado");
    }
    
    logCheckout('pix.log', "Usando gateway: {$gateway['name']}", ['gateway' => $gatewayName]);
    
    // ============================================
    // MONTA PAYLOAD CONFORME O GATEWAY
    // ============================================
    $txRef = generateTxRef();
    $payload = [];
    
    if ($gatewayName === 'allowpay') {
        // ========== ALLOWPAY ==========
        // Formato específico do AllowPay conforme documentação
        $descricaoItem = $produtoDescricao ?: ($produtoNome . ' - ' . $cotas . ' números');
        
        $payload = [
            'amount' => $valorCentavos,
            'description' => $descricaoItem,
            'paymentMethod' => 'PIX',
            'postbackUrl' => WEBHOOK_URL,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1',
            
            // Customer com document como objeto
            'customer' => [
                'name' => $nome,
                'email' => $email ?: 'cliente@email.com',
                'phone' => $telefone ?: '11999999999',
                'document' => [
                    'type' => strlen($cpf) > 11 ? 'CNPJ' : 'CPF',
                    'number' => $cpf
                ]
            ],
            
            // Shipping com address aninhado
            'shipping' => [
                'address' => [
                    'street' => 'Avenida Paulista',
                    'streetNumber' => '1000',
                    'complement' => '',
                    'neighborhood' => 'Bela Vista',
                    'city' => 'Sao Paulo',
                    'state' => 'SP',
                    'zipCode' => '01310100'
                ]
            ],
            
            // PIX com expiresInDays
            'pix' => [
                'expiresInDays' => 1
            ],
            
            // Items
            'items' => [[
                'title' => $descricaoItem,
                'unitPrice' => $valorCentavos,
                'quantity' => 1,
                'externalRef' => $txRef
            ]]
        ];
        
        // Log do payload para debug
        logCheckout('pix.log', 'AllowPay Payload', [
            'payload' => json_encode($payload, JSON_UNESCAPED_UNICODE)
        ]);
        
    } else if ($gatewayName === 'genesys') {
        // ========== GENESYS FINANCE ==========
        // Formato específico do Genesys - VALOR EM REAIS (não centavos)
        $valorReais = $valorCentavos / 100;
        $descricaoItem = $produtoDescricao ?: ($produtoNome . ' - ' . $cotas . ' números');
        
        $payload = [
            'external_id' => $txRef,
            'total_amount' => $valorReais,
            'payment_method' => 'PIX',
            'webhook_url' => WEBHOOK_URL,
            'ip' => $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1',
            
            'items' => [[
                'id' => $txRef,
                'title' => $produtoNome,
                'description' => $descricaoItem,
                'price' => $valorReais,
                'quantity' => 1,
                'is_physical' => false
            ]],
            
            'customer' => [
                'name' => $nome,
                'email' => $email ?: 'cliente@email.com',
                'phone' => $telefone ?: '11999999999',
                'document_type' => strlen($cpf) > 11 ? 'CNPJ' : 'CPF',
                'document' => $cpf,
                // UTMs nativos do Genesys
                'utm_source' => $tracking['utm_source'] ?? '',
                'utm_medium' => $tracking['utm_medium'] ?? '',
                'utm_campaign' => $tracking['utm_campaign'] ?? '',
                'utm_content' => $tracking['utm_content'] ?? '',
                'utm_term' => $tracking['utm_term'] ?? ''
            ]
        ];
        
        // Log do payload para debug
        logCheckout('pix.log', 'Genesys Payload', [
            'payload' => json_encode($payload, JSON_UNESCAPED_UNICODE)
        ]);
        
    } else {
        // ========== PADRÃO (Umbrella, TechByNet, etc) ==========
        $descricaoItem = $produtoDescricao ?: ($produtoNome . ' - ' . $cotas . ' números');
        
        $payload = [
            'amount' => $valorCentavos,
            'currency' => 'BRL',
            'paymentMethod' => 'PIX',
            'installments' => 1,
            'customer' => [
                'name' => $nome,
                'email' => $email ?: 'cliente@email.com',
                'document' => [
                    'number' => $cpf,
                    'type' => strlen($cpf) > 11 ? 'CNPJ' : 'CPF'
                ],
                'phone' => $telefone,
                'externalRef' => $txRef,
                'address' => [
                    'street' => 'Avenida Paulista',
                    'streetNumber' => '1000',
                    'complement' => '',
                    'zipCode' => '01310100',
                    'neighborhood' => 'Bela Vista',
                    'city' => 'São Paulo',
                    'state' => 'SP',
                    'country' => 'BR'
                ]
            ],
            'items' => [[
                'title' => $descricaoItem,
                'unitPrice' => $valorCentavos,
                'quantity' => 1,
                'tangible' => false,
                'externalRef' => 'item-' . $txRef
            ]],
            'pix' => [
                'expiresInDays' => 1
            ],
            'shipping' => [
                'fee' => 0,
                'address' => [
                    'street' => 'Avenida Paulista',
                    'streetNumber' => '1000',
                    'complement' => '',
                    'zipCode' => '01310100',
                    'neighborhood' => 'Bela Vista',
                    'city' => 'São Paulo',
                    'state' => 'SP',
                    'country' => 'BR'
                ]
            ],
            'postbackUrl' => WEBHOOK_URL,
            'metadata' => json_encode(['pedido' => $txRef, 'cotas' => $cotas, 'produto' => $produtoNome]),
            'traceable' => true,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1'
        ];
    }
    
    // ============================================
    // MONTA URL E HEADERS
    // ============================================
    $endpoint = $gateway['endpoints']['pix_create'] ?? '';
    $url = rtrim($gateway['api_url'], '/');
    if ($endpoint) {
        $url .= $endpoint;
    }
    
    // Monta header de autenticação
    $authType = $gateway['auth_type'] ?? 'apikey';
    
    if ($authType === 'basic') {
        // Basic Auth: base64(api_key:company_id)
        $credentials = $gateway['api_key'] . ':' . ($gateway['company_id'] ?? '');
        $authValue = 'Basic ' . base64_encode($credentials);
        $authHeaderLine = 'Authorization: ' . $authValue;
        
        logCheckout('pix.log', 'Usando Basic Auth', [
            'credentials_preview' => substr($credentials, 0, 20) . '...'
        ]);
    } else {
        // API Key padrão
        $authHeader = $gateway['auth_header'] ?? 'x-api-key';
        $authPrefix = $gateway['auth_prefix'] ?? '';
        $authValue = $authPrefix . $gateway['api_key'];
        $authHeaderLine = $authHeader . ': ' . $authValue;
    }
    
    $headers = [
        'Content-Type: application/json',
        'Accept: application/json',
        'User-Agent: ' . ($gateway['user_agent'] ?: 'SmartCheckout/2.0'),
        $authHeaderLine
    ];
    
    logCheckout('pix.log', 'Request', [
        'url' => $url,
        'txRef' => $txRef
    ]);
    
    // ============================================
    // FAZ REQUISIÇÃO PARA O GATEWAY
    // ============================================
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => true
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    logCheckout('pix.log', 'Resposta do gateway', [
        'gateway' => $gatewayName,
        'http_code' => $httpCode,
        'response_length' => strlen($response),
        'response' => substr($response, 0, 1500)
    ]);
    
    if ($error) {
        throw new Exception("Erro cURL: $error");
    }
    
    // Se resposta vazia
    if (empty($response)) {
        logCheckout('pix.log', 'ERRO: Resposta vazia do gateway');
        throw new Exception("Gateway retornou resposta vazia (HTTP $httpCode)");
    }
    
    $responseData = json_decode($response, true);
    
    if (!$responseData) {
        logCheckout('pix.log', 'ERRO: JSON inválido', [
            'json_error' => json_last_error_msg(),
            'raw_response' => $response
        ]);
        throw new Exception("Resposta inválida do gateway: " . substr($response, 0, 200));
    }
    
    // Verifica se há erro na resposta
    // IMPORTANTE: Umbrella usa "message" para sucesso também, então verificamos se há "data"
    $hasData = isset($responseData['data']) || isset($responseData['id']) || isset($responseData['pix']);
    $hasError = isset($responseData['error']);
    $httpSuccess = $httpCode >= 200 && $httpCode < 300;
    
    // Só é erro se: tem campo error OU (não tem dados E não é HTTP success)
    if ($hasError || (!$hasData && !$httpSuccess)) {
        $errorMsg = $responseData['error'] ?? $responseData['message'] ?? 'Erro desconhecido';
        logCheckout('pix.log', 'ERRO do gateway', ['error' => $errorMsg]);
        throw new Exception("Gateway: $errorMsg");
    }
    
    // ============================================
    // EXTRAI DADOS DA RESPOSTA
    // ============================================
    $txId = null;
    $pixCode = null;
    $pixBase64 = null;
    
    // Tenta extrair usando o path configurado
    $txidField = $gateway['txid_field'] ?? 'data.id';
    $qrcodeField = $gateway['qrcode_field'] ?? 'data.qrCode';
    
    $txId = extractByPath($responseData, $txidField);
    $pixCode = extractByPath($responseData, $qrcodeField);
    
    // Fallbacks para QR Code (tenta vários caminhos comuns)
    if (!$pixCode) {
        $qrcodePaths = [
            'pix.payload',       // Genesys Finance
            'pix.qrcode',        // AllowPay
            'pix.qrCode',        // AllowPay variação
            'data.qrCode',
            'data.qrcode',
            'data.data.qrCode',
            'data.data.qrcode',
            'data.pix.qrCode',
            'data.pix.qrcode',
            'data.pix.payload',  // Genesys variação
            'data.brCode',
            'data.emv',
            'qrCode',
            'qrcode',
            'brCode',
            'emv',
            'payload'            // Genérico
        ];
        
        foreach ($qrcodePaths as $path) {
            $pixCode = extractByPath($responseData, $path);
            if ($pixCode) {
                logCheckout('pix.log', "QR Code encontrado em: $path");
                break;
            }
        }
    }
    
    // Fallbacks para ID
    if (!$txId) {
        $idPaths = ['id', 'data.id', 'data.data.id', 'objectId'];
        foreach ($idPaths as $path) {
            $txId = extractByPath($responseData, $path);
            if ($txId) {
                logCheckout('pix.log', "ID encontrado em: $path");
                break;
            }
        }
    }
    
    // QR Code Base64
    $base64Paths = ['data.qrCodeBase64', 'data.data.qrCodeBase64', 'qrCodeBase64'];
    foreach ($base64Paths as $path) {
        $pixBase64 = extractByPath($responseData, $path);
        if ($pixBase64) break;
    }
    
    // Se não encontrou txId, gera um local
    if (!$txId) {
        $txId = $txRef;
        logCheckout('pix.log', 'Usando txRef como txId', ['txId' => $txId]);
    }
    
    logCheckout('pix.log', 'Dados extraídos', [
        'txId' => $txId,
        'pixCode_length' => $pixCode ? strlen($pixCode) : 0,
        'pixCode_preview' => $pixCode ? substr($pixCode, 0, 50) . '...' : null
    ]);
    
    if (!$pixCode) {
        logCheckout('pix.log', 'ERRO: QR Code não encontrado', [
            'response_keys' => array_keys($responseData)
        ]);
        throw new Exception("QR Code PIX não encontrado na resposta do gateway");
    }
    
    // ============================================
    // SALVA DADOS PARA WEBHOOK/TRACKING
    // ============================================
    $utmData = [
        'transaction_id' => $txId,
        'gateway' => $gatewayName,
        'created_at' => date('Y-m-d H:i:s'),
        'valor_centavos' => $valorCentavos,
        'cotas' => $cotas,
        'produto' => $produtoNome,
        'produto_id' => $produtoId,
        
        // UTMs
        'utm_source' => emptyToNull($tracking['utm_source'] ?? null),
        'utm_medium' => emptyToNull($tracking['utm_medium'] ?? null),
        'utm_campaign' => emptyToNull($tracking['utm_campaign'] ?? null),
        'utm_term' => emptyToNull($tracking['utm_term'] ?? null),
        'utm_content' => emptyToNull($tracking['utm_content'] ?? null),
        'sck' => emptyToNull($tracking['sck'] ?? null),
        'src' => emptyToNull($tracking['src'] ?? null),
        'fbc' => emptyToNull($tracking['fbc'] ?? null),
        'fbp' => emptyToNull($tracking['fbp'] ?? null),
        
        // IP
        'customer_ip' => $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '',
        
        // Cliente
        'customer' => [
            'name' => $nome,
            'email' => $email,
            'phone' => $telefone,
            'document' => $cpf
        ]
    ];
    
    @file_put_contents(
        UTM_STORAGE_DIR . $txId . '.json',
        json_encode($utmData, JSON_PRETTY_PRINT)
    );
    
    logCheckout('pix.log', 'PIX criado com sucesso', [
        'tx_id' => $txId,
        'gateway' => $gatewayName,
        'valor' => $valorReais
    ]);
    
    // ============================================
    // UTMIFY (STATUS PENDENTE)
    // ============================================
    if (class_exists('Utmify')) {
        try {
            $utmify = new Utmify();
            $utmify->sendPending(
                $txId,
                $valorCentavos,
                [
                    'name' => $nome,
                    'email' => $email,
                    'phone' => $telefone,
                    'document' => $cpf,
                    'ip' => $utmData['customer_ip']
                ],
                [
                    'id' => $produtoId,
                    'name' => $produtoNome,
                    'quantity' => $cotas
                ],
                $tracking
            );
        } catch (Exception $e) {
            logCheckout('pix.log', 'Erro UTMify: ' . $e->getMessage());
        }
    }
    
    // ============================================
    // RESPOSTA DE SUCESSO
    // ============================================
    echo json_encode([
        'success' => true,
        'gateway' => $gateway['name'],
        'id' => $txId,
        'txid' => $txId,
        'pix' => [
            'qrcode' => $pixCode,
            'qrcodeBase64' => $pixBase64
        ],
        'qrcode' => $pixCode,
        'qrcodeBase64' => $pixBase64,
        'amount' => $valorCentavos,
        'valor' => $valorReais,
        'status' => 'pending'
    ]);
    
} catch (Exception $e) {
    logCheckout('pix.log', 'ERRO: ' . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
