<?php
/**
 * Sistema de Roteamento - Compatível com PHP 5.6+
 */

require __DIR__ . '/info/autoload.php';

use GeoIp2\Database\Reader;

// Carrega configurações
$config = require __DIR__ . '/config.php';

// ============================================
// CLASSE PRINCIPAL
// ============================================

class TrafficRouter
{
    private $config;
    private $params;
    private $clientIP;
    private $userAgent;
    
    public function __construct($config)
    {
        $this->config = $config;
        $this->params = $_GET;
        $this->clientIP = $this->detectClientIP();
        $this->userAgent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
    }
    
    /**
     * Executa todas as validações e redireciona
     */
    public function process()
    {
        // 1. Verificar se é bot suspeito
        if ($this->config['bot_detection']['enabled'] && $this->isSuspiciousBot()) {
            $this->log('Bot suspeito detectado');
            $this->redirectToSafe();
        }
        
        // 2. Verificar geolocalização
        if ($this->config['geo']['enabled'] && !$this->isAllowedCountry()) {
            $this->redirectToSafe();
        }
        
        // 3. Verificar parâmetros obrigatórios
        if (!$this->validateRequiredParams()) {
            $this->redirectToSafe();
        }
        
        // 4. Definir cookie
        $this->setSessionCookie();
        
        // 5. Determinar destino e redirecionar
        $destination = $this->determineDestination();
        $finalUrl = $this->appendParams($destination);
        
        $this->log("Acesso válido -> " . $finalUrl);
        $this->redirect($finalUrl);
    }
    
    /**
     * Detecta IP real do cliente
     */
    private function detectClientIP()
    {
        $headers = array(
            'HTTP_CF_CONNECTING_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'REMOTE_ADDR'
        );
        
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                if (strpos($ip, ',') !== false) {
                    $parts = explode(',', $ip);
                    $ip = trim($parts[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return '0.0.0.0';
    }
    
    /**
     * Verifica se o User-Agent parece ser de um bot
     */
    private function isSuspiciousBot()
    {
        $botConfig = $this->config['bot_detection'];
        
        // Verifica UA vazio
        if ($botConfig['block_empty_ua'] && empty($this->userAgent)) {
            return true;
        }
        
        // Verifica keywords suspeitas
        $uaLower = strtolower($this->userAgent);
        foreach ($botConfig['suspicious_keywords'] as $keyword) {
            if (strpos($uaLower, $keyword) !== false) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Verifica se o país é permitido
     */
    private function isAllowedCountry()
    {
        $geoConfig = $this->config['geo'];
        
        try {
            $reader = new Reader($geoConfig['database_path']);
            $record = $reader->country($this->clientIP);
            $countryCode = $record->country->isoCode;
            
            if (!in_array($countryCode, $geoConfig['allowed_countries'])) {
                $this->log("País bloqueado: " . $countryCode);
                return false;
            }
            
            return true;
            
        } catch (Exception $e) {
            $this->log("Erro GeoIP: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Valida parâmetros obrigatórios
     */
    private function validateRequiredParams()
    {
        foreach ($this->config['required_params'] as $param => $allowedValues) {
            $value = isset($this->params[$param]) ? $this->params[$param] : '';
            
            if (empty($value)) {
                $this->log("Param ausente: " . $param);
                return false;
            }
            
            if ($allowedValues !== null && !in_array($value, $allowedValues, true)) {
                $this->log("Valor inválido " . $param . ": " . $value);
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Determina URL de destino baseado nos parâmetros
     */
    private function determineDestination()
    {
        // 1. Primeiro verifica parâmetros especiais
        if (isset($this->config['special_params']) && is_array($this->config['special_params'])) {
            foreach ($this->config['special_params'] as $param => $routes) {
                $value = isset($this->params[$param]) ? strtolower($this->params[$param]) : '';
                if (!empty($value) && isset($routes[$value])) {
                    return $routes[$value];
                }
            }
        }
        
        // 2. Verifica campanha
        $campaign = isset($this->params['utm_campaign']) ? strtolower($this->params['utm_campaign']) : '';
        $campaigns = $this->config['campaigns'];
        
        if (!empty($campaign) && isset($campaigns[$campaign])) {
            return $campaigns[$campaign];
        }
        
        // 3. Retorna destino padrão
        return isset($campaigns['default']) ? $campaigns['default'] : '/';
    }
    
    /**
     * Adiciona parâmetros preservados à URL
     */
    private function appendParams($url)
    {
        $preserveParams = array();
        
        foreach ($this->config['preserve_params'] as $param) {
            if (isset($this->params[$param]) && !empty($this->params[$param])) {
                $preserveParams[$param] = $this->params[$param];
            }
        }
        
        if (empty($preserveParams)) {
            return $url;
        }
        
        $separator = (strpos($url, '?') !== false) ? '&' : '?';
        return $url . $separator . http_build_query($preserveParams);
    }
    
    /**
     * Define cookie de sessão
     */
    private function setSessionCookie()
    {
        $cookie = $this->config['cookie'];
        
        if (!isset($_COOKIE[$cookie['name']])) {
            setcookie(
                $cookie['name'],
                $cookie['value'],
                time() + $cookie['duration'],
                $cookie['path'],
                '',
                $cookie['secure'],
                $cookie['httponly']
            );
        }
    }
    
    /**
     * Redireciona para página segura
     */
    private function redirectToSafe()
    {
        $this->redirect($this->config['safe_page']);
    }
    
    /**
     * Executa redirecionamento
     */
    private function redirect($url)
    {
        header("Location: " . $url, true, 302);
        exit();
    }
    
    /**
     * Registra log
     */
    private function log($message)
    {
        $logConfig = $this->config['logging'];
        
        if (!$logConfig['enabled']) {
            return;
        }
        
        $logDir = dirname($logConfig['file']);
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        // Rotação de log se necessário
        if (file_exists($logConfig['file']) && filesize($logConfig['file']) > $logConfig['max_size']) {
            rename($logConfig['file'], $logConfig['file'] . '.' . date('Y-m-d-His'));
        }
        
        $paramsJson = json_encode($this->params);
        $entry = sprintf(
            "[%s] %s | IP: %s | UA: %s | Params: %s\n",
            date('Y-m-d H:i:s'),
            $message,
            $this->clientIP,
            substr($this->userAgent, 0, 100),
            $paramsJson
        );
        
        file_put_contents($logConfig['file'], $entry, FILE_APPEND | LOCK_EX);
    }
}

// ============================================
// EXECUÇÃO
// ============================================

$router = new TrafficRouter($config);
$router->process();