<?php
/**
 * Backend - API Umbrellapag
 * Versão com UTC forçado e debug de timezone
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// ============================================
// CONFIGURAÇÃO DA API
// ============================================
define('API_BASE_URL', 'https://api-gateway.umbrellapag.com');
define('API_TOKEN', 'd46fa92f-9484-409f-a82a-e2734029979e');
define('USER_AGENT', 'AtivoB2B/1.0');

// ============================================
// CONFIGURAÇÃO UTMIFY
// ============================================
define('UTMIFY_TOKEN', '3uwgd9W2qWRgPv6LcfQjfv0n1hkMitcgilZH');
define('UTMIFY_URL', 'https://api.utmify.com.br/api-credentials/orders');

// Diretório para armazenar UTMs
define('UTM_STORAGE_DIR', __DIR__ . '/utm_storage/');

if (!is_dir(UTM_STORAGE_DIR)) {
    mkdir(UTM_STORAGE_DIR, 0755, true);
}

// ============================================
// FUNÇÃO PARA REGISTRAR LOGS
// ============================================
function logMessage($file, $message) {
    $logDir = __DIR__ . '/logs/';
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    // Log sempre em UTC
    $utcTime = gmdate("Y-m-d H:i:s");
    file_put_contents($logDir . $file, "[$utcTime UTC] " . $message . "\n", FILE_APPEND);
}

// ============================================
// FUNÇÃO PARA OBTER DATA EM UTC
// Formato Utmify: "YYYY-MM-DD HH:MM:SS"
// ============================================
function getUtcDateTime() {
    return gmdate('Y-m-d H:i:s');
}

// ============================================
// FUNÇÃO PARA CONVERTER VALOR VAZIO PARA NULL
// ============================================
function emptyToNull($value) {
    if ($value === null || $value === '' || (is_string($value) && trim($value) === '')) {
        return null;
    }
    return $value;
}

// ============================================
// FUNÇÃO PARA SALVAR UTMs
// ============================================
function saveUtmData($transactionId, $utmData) {
    if (empty($transactionId)) return false;
    
    $filePath = UTM_STORAGE_DIR . $transactionId . '.json';
    $data = [
        'transaction_id' => $transactionId,
        'created_at_utc' => getUtcDateTime(),
        'utm_source' => emptyToNull($utmData['utm_source'] ?? null),
        'utm_medium' => emptyToNull($utmData['utm_medium'] ?? null),
        'utm_campaign' => emptyToNull($utmData['utm_campaign'] ?? null),
        'utm_term' => emptyToNull($utmData['utm_term'] ?? null),
        'utm_content' => emptyToNull($utmData['utm_content'] ?? null),
        'sck' => emptyToNull($utmData['sck'] ?? null),
        'src' => emptyToNull($utmData['src'] ?? null),
        'customer_ip' => $_SERVER['REMOTE_ADDR'] ?? null
    ];
    
    $result = file_put_contents($filePath, json_encode($data, JSON_PRETTY_PRINT));
    
    if ($result) {
        logMessage('utm_save.log', "UTMs salvas para transação $transactionId: " . json_encode($data));
    }
    
    return $result !== false;
}

// ============================================
// FUNÇÃO PARA ENVIAR PARA UTMIFY
// ============================================
function sendToUtmify($transactionData, $customerData, $utmData, $productData, $status = 'waiting_payment') {
    
    // Formata telefone
    $formattedPhone = null;
    if (!empty($customerData['phone'])) {
        $cleanPhone = preg_replace('/\D/', '', $customerData['phone']);
        $formattedPhone = '+55' . $cleanPhone;
    }
    
    // Data em UTC
    $createdAtUtc = getUtcDateTime();
    
    // Log de debug do timezone
    logMessage('timezone_debug.log', "=== DEBUG TIMEZONE ===");
    logMessage('timezone_debug.log', "PHP Timezone: " . date_default_timezone_get());
    logMessage('timezone_debug.log', "Server time(): " . time());
    logMessage('timezone_debug.log', "date() local: " . date('Y-m-d H:i:s'));
    logMessage('timezone_debug.log', "gmdate() UTC: " . gmdate('Y-m-d H:i:s'));
    logMessage('timezone_debug.log', "createdAtUtc gerado: " . $createdAtUtc);
    
    $infoVenda = [
        'orderId' => (string) $transactionData['id'],
        'platform' => 'Umbrellapag',
        'paymentMethod' => 'pix',
        'status' => $status,
        'createdAt' => $createdAtUtc,
        'approvedDate' => null,
        'refundedAt' => null,
        'customer' => [
            'name' => $customerData['name'],
            'email' => $customerData['email'],
            'phone' => $formattedPhone,
            'document' => $customerData['document'],
            'country' => 'BR',
            'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
        ],
        'products' => [
            [
                'id' => 'PROD01',
                'name' => $productData['title'] ?? 'Produto Digital',
                'planId' => 'PLAN01',
                'planName' => 'Plano Único',
                'quantity' => 1,
                'priceInCents' => (int) $transactionData['amount'],
            ]
        ],
        'trackingParameters' => [
            'src' => emptyToNull($utmData['src'] ?? null),
            'sck' => emptyToNull($utmData['sck'] ?? null),
            'utm_source' => emptyToNull($utmData['utm_source'] ?? null),
            'utm_medium' => emptyToNull($utmData['utm_medium'] ?? null),
            'utm_campaign' => emptyToNull($utmData['utm_campaign'] ?? null),
            'utm_term' => emptyToNull($utmData['utm_term'] ?? null),
            'utm_content' => emptyToNull($utmData['utm_content'] ?? null),
        ],
        'commission' => [
            'totalPriceInCents' => (int) $transactionData['amount'],
            'gatewayFeeInCents' => 0,
            'userCommissionInCents' => (int) $transactionData['amount'],
            'currency' => 'BRL',
        ],
        'isTest' => false,
    ];
    
    $arrayInfo = json_encode($infoVenda, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    
    logMessage('utmify_create_request.log', "Enviando para UTMify | Transação: {$transactionData['id']} | Status: $status | createdAt: $createdAtUtc");
    logMessage('utmify_create_request.log', "JSON: $arrayInfo");
    
    // Envia para UTMify
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => UTMIFY_URL,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $arrayInfo,
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'x-api-token: ' . UTMIFY_TOKEN,
        ],
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_SSL_VERIFYHOST => 2,
        CURLOPT_SSL_VERIFYPEER => true,
    ]);
    
    $response = curl_exec($curl);
    $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    $error = curl_error($curl);
    curl_close($curl);
    
    if ($error) {
        logMessage('utmify_create_error.log', "Erro cURL (Transação {$transactionData['id']}): $error");
        return false;
    }
    
    logMessage('utmify_create_response.log', "Transação: {$transactionData['id']} | HTTP: $httpCode | Response: $response");
    
    return ($httpCode >= 200 && $httpCode < 300);
}

// ============================================
// FUNÇÃO PARA FAZER REQUISIÇÕES À API
// ============================================
function apiRequest($endpoint, $method = 'POST', $data = null) {
    $url = API_BASE_URL . $endpoint;
    
    $headers = [
        'Content-Type: application/json',
        'User-Agent: ' . USER_AGENT,
        'x-api-key: ' . API_TOKEN
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    if ($method === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($data) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        logMessage('api_error.log', "Erro cURL: $error | URL: $url");
        return [
            'success' => false,
            'error' => $error,
            'httpCode' => $httpCode
        ];
    }
    
    $decoded = json_decode($response, true);
    
    return [
        'success' => ($httpCode >= 200 && $httpCode < 300),
        'httpCode' => $httpCode,
        'data' => $decoded
    ];
}

// ============================================
// CRIAR TRANSAÇÃO PIX
// ============================================
function createPixTransaction($params) {
    $valorFixo = 47.90;
    
    $defaults = [
        'name' => 'Pedro Silva',
        'email' => 'cliente@gmail.com',
        'document' => '00895889919',
        'phone' => '11999999999',
        'productTitle' => 'Produto DigitalZ'
    ];
    
    $name = !empty($params['name']) ? $params['name'] : $defaults['name'];
    $email = !empty($params['email']) ? $params['email'] : $defaults['email'];
    $document = !empty($params['document']) ? $params['document'] : $defaults['document'];
    $phone = !empty($params['phone']) ? $params['phone'] : $defaults['phone'];
    $productTitle = !empty($params['productTitle']) ? $params['productTitle'] : $defaults['productTitle'];
    
    $amountInCents = intval($valorFixo * 100);
    
    // Log das UTMs recebidas
    logMessage('utm_received.log', "UTMs recebidas: utm_source=" . ($params['utm_source'] ?? 'VAZIO') . 
        " | utm_campaign=" . ($params['utm_campaign'] ?? 'VAZIO') .
        " | utm_medium=" . ($params['utm_medium'] ?? 'VAZIO'));
    
    $data = [
        'amount' => $amountInCents,
        'currency' => 'BRL',
        'paymentMethod' => 'pix',
        'customer' => [
            'name' => $name,
            'email' => $email,
            'document' => [
                'number' => preg_replace('/\D/', '', $document),
                'type' => strlen(preg_replace('/\D/', '', $document)) > 11 ? 'CNPJ' : 'CPF'
            ],
            'phone' => preg_replace('/\D/', '', $phone)
        ],
        'items' => [
            [
                'title' => $productTitle,
                'unitPrice' => $amountInCents,
                'quantity' => 1,
                'tangible' => false
            ]
        ],
        'pix' => [
            'expiresInDays' => 1
        ]
    ];
    
    $result = apiRequest('/api/user/transactions', 'POST', $data);
    
    if ($result['success'] && isset($result['data']['data']['id'])) {
        $transactionId = $result['data']['data']['id'];
        $txData = $result['data']['data'];
        
        $utmData = [
            'utm_source' => emptyToNull($params['utm_source'] ?? null),
            'utm_medium' => emptyToNull($params['utm_medium'] ?? null),
            'utm_campaign' => emptyToNull($params['utm_campaign'] ?? null),
            'utm_term' => emptyToNull($params['utm_term'] ?? null),
            'utm_content' => emptyToNull($params['utm_content'] ?? null),
            'sck' => emptyToNull($params['sck'] ?? null),
            'src' => emptyToNull($params['src'] ?? null)
        ];
        
        saveUtmData($transactionId, $utmData);
        
        $customerData = [
            'name' => $name,
            'email' => $email,
            'document' => preg_replace('/\D/', '', $document),
            'phone' => $phone
        ];
        
        $productData = ['title' => $productTitle];
        
        $transactionDataForUtmify = [
            'id' => $transactionId,
            'amount' => $amountInCents
        ];
        
        $utmifyResult = sendToUtmify($transactionDataForUtmify, $customerData, $utmData, $productData, 'waiting_payment');
        
        logMessage('transactions.log', ($utmifyResult ? "✅" : "❌") . " Transação: $transactionId | UTMify: " . ($utmifyResult ? "OK" : "FALHOU"));
    }
    
    return $result;
}

// ============================================
// VERIFICAR STATUS DO PAGAMENTO
// ============================================
function checkPaymentStatus($transactionId) {
    $result = apiRequest('/api/user/transactions/' . $transactionId, 'GET');
    return $result;
}

// ============================================
// ROTEAMENTO
// ============================================
$action = $_GET['action'] ?? $_POST['action'] ?? '';

$rawInput = file_get_contents('php://input');
$jsonInput = json_decode($rawInput, true);

$input = [];
if (!empty($_GET)) $input = array_merge($input, $_GET);
if (!empty($_POST)) $input = array_merge($input, $_POST);
if (!empty($jsonInput) && is_array($jsonInput)) $input = array_merge($input, $jsonInput);

switch ($action) {
    case 'create-pix':
        $required = ['name', 'email', 'document', 'phone', 'amount'];
        $missing = [];
        
        foreach ($required as $field) {
            if (empty($input[$field])) {
                $missing[] = $field;
            }
        }
        
        if (!empty($missing)) {
            echo json_encode([
                'success' => false,
                'error' => 'Campos obrigatórios faltando: ' . implode(', ', $missing)
            ]);
            exit;
        }
        
        $result = createPixTransaction($input);
        
        if ($result['success'] && isset($result['data']['data'])) {
            $txData = $result['data']['data'];
            
            $pixCode = $txData['qrCode'] 
                ?? $txData['pix']['qrCode'] 
                ?? $txData['pix']['brCode'] 
                ?? $txData['pix']['emv'] 
                ?? $txData['brCode'] 
                ?? $txData['emv'] 
                ?? null;
            
            echo json_encode([
                'success' => true,
                'transactionId' => $txData['id'],
                'pixCode' => $pixCode,
                'amount' => $txData['amount'] ?? $input['amount'],
                'status' => $txData['status'] ?? 'pending'
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'error' => $result['data']['message'] ?? $result['data']['error'] ?? 'Erro ao criar transação',
                'httpCode' => $result['httpCode']
            ]);
        }
        break;
        
    case 'check-status':
        $transactionId = $input['transactionId'] ?? $_GET['transactionId'] ?? '';
        
        if (empty($transactionId)) {
            echo json_encode(['success' => false, 'error' => 'ID da transação não informado']);
            exit;
        }
        
        $result = checkPaymentStatus($transactionId);
        
        if ($result['success'] && isset($result['data']['data'])) {
            echo json_encode([
                'success' => true,
                'status' => $result['data']['data']['status'] ?? 'pending',
                'raw' => $result['data']['data']
            ]);
        } else {
            echo json_encode(['success' => false, 'error' => 'Erro ao verificar status']);
        }
        break;
        
    default:
        echo json_encode(['success' => false, 'error' => 'Ação não especificada']);
        break;
}