// ============================================
// CONFIGURAÇÃO
// ============================================
const API_BACKEND = 'api/umbrela.php'; // Caminho para o backend PHP
const STATUS_API = 'api/status.php'; // Arquivo de status que criamos

// ⚠️ CONFIGURE AQUI A PÁGINA DE REDIRECIONAMENTO APÓS PAGAMENTO
const REDIRECT_URL_SUCCESS = 'https://seusite.com/obrigado.html'; // Altere para sua página de sucesso
const REDIRECT_URL_ERROR = 'https://seusite.com/erro.html'; // Altere para sua página de erro (opcional)

// ============================================
// CAPTURAR DADOS DA URL
// ============================================
function getUrlParams() {
    const params = new URLSearchParams(window.location.search);
    return {
        name: params.get('name') || params.get('nome') || params.get('fullName') || '',
        email: params.get('email') || '',
        document: params.get('document') || params.get('cpf') || params.get('documento') || '',
        phone: params.get('phone') || params.get('telefone') || params.get('cel') || '',
        amount: parseFloat(params.get('amount') || params.get('valor') || '3283'),
        productTitle: params.get('title') || params.get('titulo') || params.get('produto') || 'Produto',
        // Captura parâmetros extras para passar no redirecionamento
        sck: params.get('sck') || '',
        src: params.get('src') || '',
        utm_campaign: params.get('utm_campaign') || '',
        utm_content: params.get('utm_content') || '',
        utm_medium: params.get('utm_medium') || '',
        utm_source: params.get('utm_source') || '',
        utm_term: params.get('utm_term') || ''
    };
}

const URL_PARAMS = getUrlParams();
console.log('📋 Dados capturados da URL:', URL_PARAMS);

// Estado global
let currentTransactionId = null;
let currentPixCode = null;
let pollingInterval = null;
let pollingAttempts = 0;
const MAX_POLLING_ATTEMPTS = 360; // 30 minutos (5 segundos * 360)

// ============================================
// CRIAR PAGAMENTO PIX (via Backend PHP)
// ============================================
async function createPixPayment() {
    updateLoadingText('Gerando pagamento PIX...');
    
    try {
        const response = await fetch(`${API_BACKEND}?action=create-pix`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                name: URL_PARAMS.name,
                email: URL_PARAMS.email,
                document: URL_PARAMS.document,
                phone: URL_PARAMS.phone,
                amount: URL_PARAMS.amount,
                productTitle: URL_PARAMS.productTitle
            })
        });

        const result = await response.json();
        console.log('📦 Resposta do Backend:', result);

        if (result.success) {
            return {
                transactionId: result.transactionId,
                pixCode: result.pixCode,
                amount: result.amount,
                status: result.status
            };
        } else {
            console.error('❌ Erro:', result);
            throw new Error(result.error || 'Erro ao criar pagamento PIX');
        }

    } catch (error) {
        console.error('❌ Erro ao criar pagamento:', error);
        throw error;
    }
}

// ============================================
// VERIFICAR STATUS (via Backend PHP)
// ============================================
async function checkPaymentStatus(transactionId) {
    try {
        // Usando o novo arquivo de status
        const formData = new FormData();
        formData.append('id_transacao_post', transactionId);
        formData.append('sck_post', URL_PARAMS.sck);
        formData.append('src_post', URL_PARAMS.src);
        formData.append('utm_campaign_post', URL_PARAMS.utm_campaign);
        formData.append('utm_content_post', URL_PARAMS.utm_content);
        formData.append('utm_medium_post', URL_PARAMS.utm_medium);
        formData.append('utm_source_post', URL_PARAMS.utm_source);
        formData.append('utm_term_post', URL_PARAMS.utm_term);

        const response = await fetch(STATUS_API, {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        console.log('📊 Resposta Status API:', result);
        
        if (result.success) {
            return {
                status: result.status,
                isPaid: result.isPaid,
                raw: result.raw
            };
        }
        
        return { status: 'pending', isPaid: false };
        
    } catch (error) {
        console.error('Erro ao verificar status:', error);
        return { status: 'pending', isPaid: false };
    }
}

// ============================================
// REDIRECIONAMENTO APÓS PAGAMENTO
// ============================================
function redirectToSuccessPage() {
    // Monta a URL com parâmetros opcionais
    let redirectUrl = REDIRECT_URL_SUCCESS;
    
    // Adiciona parâmetros à URL de redirecionamento (opcional)
    const params = new URLSearchParams();
    
    if (currentTransactionId) params.append('transactionId', currentTransactionId);
    if (URL_PARAMS.name) params.append('name', URL_PARAMS.name);
    if (URL_PARAMS.email) params.append('email', URL_PARAMS.email);
    if (URL_PARAMS.sck) params.append('sck', URL_PARAMS.sck);
    if (URL_PARAMS.src) params.append('src', URL_PARAMS.src);
    if (URL_PARAMS.utm_campaign) params.append('utm_campaign', URL_PARAMS.utm_campaign);
    if (URL_PARAMS.utm_content) params.append('utm_content', URL_PARAMS.utm_content);
    if (URL_PARAMS.utm_medium) params.append('utm_medium', URL_PARAMS.utm_medium);
    if (URL_PARAMS.utm_source) params.append('utm_source', URL_PARAMS.utm_source);
    if (URL_PARAMS.utm_term) params.append('utm_term', URL_PARAMS.utm_term);
    
    const queryString = params.toString();
    if (queryString) {
        redirectUrl += (redirectUrl.includes('?') ? '&' : '?') + queryString;
    }
    
    console.log('🚀 Redirecionando para:', redirectUrl);
    window.location.href = redirectUrl;
}

function redirectToErrorPage() {
    if (REDIRECT_URL_ERROR) {
        window.location.href = REDIRECT_URL_ERROR;
    }
}

// ============================================
// POLLING DE STATUS COM REDIRECIONAMENTO
// ============================================
function startPaymentPolling(transactionId) {
    if (!transactionId) {
        console.warn('⚠️ Transaction ID não fornecido para polling');
        return;
    }
    
    // Limpa qualquer polling anterior
    if (pollingInterval) {
        clearInterval(pollingInterval);
    }
    
    pollingAttempts = 0;
    console.log('🔄 Iniciando polling de status para:', transactionId);
    
    pollingInterval = setInterval(async function() {
        pollingAttempts++;
        console.log(`📊 Verificando status... (tentativa ${pollingAttempts}/${MAX_POLLING_ATTEMPTS})`);
        
        const statusResult = await checkPaymentStatus(transactionId);
        const status = statusResult.status ? statusResult.status.toLowerCase() : 'pending';
        
        console.log('📊 Status atual:', status, '| isPaid:', statusResult.isPaid);
        
        // Status de pagamento confirmado
        if (statusResult.isPaid || ['paid', 'approved', 'confirmed', 'completed', 'success'].includes(status)) {
            clearInterval(pollingInterval);
            console.log('✅ Pagamento confirmado!');
            
            showToast('Pagamento confirmado! Redirecionando... 🎉', 'success');
            
            // Aguarda 2 segundos para o usuário ver a mensagem e redireciona
            setTimeout(() => {
                redirectToSuccessPage();
            }, 2000);
            
            return;
        }
        
        // Status de erro/cancelamento
        if (['expired', 'cancelled', 'canceled', 'failed', 'refunded', 'error'].includes(status)) {
            clearInterval(pollingInterval);
            console.log('❌ Pagamento falhou ou expirou:', status);
            
            showToast('Pagamento expirado ou cancelado', 'error');
            
            // Opcional: redirecionar para página de erro após alguns segundos
            // setTimeout(() => { redirectToErrorPage(); }, 3000);
            
            return;
        }
        
        // Verifica limite de tentativas (evita polling infinito)
        if (pollingAttempts >= MAX_POLLING_ATTEMPTS) {
            clearInterval(pollingInterval);
            console.log('⏰ Tempo limite de verificação atingido');
            showToast('Tempo de verificação expirado. Atualize a página.', 'error');
        }
        
    }, 5000); // Verifica a cada 5 segundos
}

// ============================================
// FLUXO PRINCIPAL
// ============================================
async function initializePayment() {
    showLoading(true);
    
    // Validações
    if (!URL_PARAMS.name) {
        showLoading(false);
        showError('Nome não informado. Use: ?name=joaosilva');
        return;
    }
    if (!URL_PARAMS.document) {
        showLoading(false);
        showError('CPF não informado. Use: ?cpf=12345678909');
        return;
    }
    if (!URL_PARAMS.email) {
        showLoading(false);
        showError('Email não informado. Use: ?email=email@gmail.com');
        return;
    }
    
    try {
        const pixData = await createPixPayment();
        console.log('✅ Pagamento criado:', pixData);
        
        currentTransactionId = pixData.transactionId;
        currentPixCode = pixData.pixCode;
        
        if (pixData.pixCode) {
            updatePixCode(pixData.pixCode);
            generateQRCode(pixData.pixCode);
            showToast('Código PIX gerado com sucesso!', 'success');
        } else {
            console.warn('⚠️ PIX Code não disponível');
            updatePixCode('Código PIX não retornado - verifique o console');
        }
        
        showLoading(false);
        showPaymentCard(true);
        
        // Inicia o polling de status
        if (pixData.transactionId) {
            startPaymentPolling(pixData.transactionId);
        }

    } catch (error) {
        console.error('❌ Erro:', error);
        showLoading(false);
        showError(error.message);
    }
}

// ============================================
// FUNÇÕES DE UI
// ============================================
function generateQRCode(pixCode) {
    const container = document.getElementById('qrcode');
    if (container) {
        container.innerHTML = '';
        if (pixCode) {
            new QRCode(container, { 
                text: pixCode, 
                width: 180, 
                height: 180, 
                colorDark: '#000000', 
                colorLight: '#ffffff', 
                correctLevel: QRCode.CorrectLevel.M 
            });
        }
    }
}

function updatePixCode(code) {
    const element = document.getElementById('pixCode');
    if (element) {
        element.textContent = code || 'Código não disponível';
    }
}

function updateLoadingText(text) {
    const element = document.getElementById('loadingText');
    if (element) {
        element.textContent = text;
    }
}

function showLoading(show) {
    const element = document.getElementById('loadingOverlay');
    if (element) {
        element.classList.toggle('show', show);
    }
}

function showPaymentCard(show) {
    const paymentCard = document.getElementById('paymentCard');
    const errorState = document.getElementById('errorState');
    
    if (paymentCard) {
        paymentCard.style.display = show ? 'block' : 'none';
    }
    if (errorState) {
        errorState.classList.remove('show');
    }
}

function showError(message) {
    const paymentCard = document.getElementById('paymentCard');
    const errorMessage = document.getElementById('errorMessage');
    const errorState = document.getElementById('errorState');
    
    if (paymentCard) {
        paymentCard.style.display = 'none';
    }
    if (errorMessage) {
        errorMessage.textContent = message;
    }
    if (errorState) {
        errorState.classList.add('show');
    }
}

function copyPixCode() {
    const code = currentPixCode || document.getElementById('pixCode')?.textContent;
    if (!code || code === 'Gerando código...' || code.includes('não')) {
        showToast('Código PIX ainda não disponível', 'error');
        return;
    }
    navigator.clipboard.writeText(code).then(() => {
        showToast('Código PIX copiado!', 'success');
    }).catch(() => {
        const textarea = document.createElement('textarea');
        textarea.value = code;
        document.body.appendChild(textarea);
        textarea.select();
        document.execCommand('copy');
        document.body.removeChild(textarea);
        showToast('Código PIX copiado!', 'success');
    });
}

function showToast(message, type) {
    const toast = document.getElementById('toast');
    if (toast) {
        toast.textContent = message;
        toast.className = 'toast ' + (type || '');
        toast.classList.add('show');
        setTimeout(() => toast.classList.remove('show'), 3000);
    }
}

function retryPayment() {
    initializePayment();
}

// ============================================
// ANIMAÇÕES
// ============================================
function startTransition() {
    const expandCircle = document.getElementById('expandCircle');
    const checkIcon = document.getElementById('checkIcon');
    const mainContent = document.getElementById('mainContent');

    if (expandCircle) expandCircle.classList.add('show');
    
    setTimeout(() => { 
        if (checkIcon) {
            checkIcon.classList.add('show'); 
            checkIcon.classList.add('draw'); 
        }
    }, 300);
    
    setTimeout(() => {
        if (expandCircle) expandCircle.classList.add('expand');
    }, 800);
    
    setTimeout(() => { 
        if (mainContent) mainContent.classList.add('visible'); 
        if (expandCircle) expandCircle.style.display = 'none'; 
        initializePayment(); 
    }, 1400);
}

window.onload = function() {
    setTimeout(() => {
        const preloader = document.getElementById('preloader');
        if (preloader) {
            preloader.classList.add('hidden');
            setTimeout(startTransition, 350);
        } else {
            // Se não houver preloader, inicia direto
            startTransition();
        }
    }, 3200);
};

// ============================================
// FUNÇÕES AUXILIARES EXISTENTES
// ============================================
function getQueryParams() {
    var params = {};
    var parser = document.createElement('a');
    parser.href = window.location.href;
    var query = parser.search.substring(1);
    var vars = query.split('&');
    for (var i = 0; i < vars.length; i++) {
        var pair = vars[i].split('=');
        if (pair[0]) {
            params[pair[0]] = decodeURIComponent(pair[1] || '');
        }
    }
    return params;
}