<?php
/**
 * Webhook UTMify - Integração Umbrellapag
 * Recebe webhooks da Umbrellapag e envia para UTMify
 * 
 * IMPORTANTE: UTMify usa UTC para datas
 * 
 * Configure este URL como webhook na Umbrellapag:
 * https://seusite.com/api/webhook_utmify_umbrella.php
 */

// ============================================
// CONFIGURAÇÕES
// ============================================
$UTMify_Token = '2kN5TCos2BgJfIF19fCj7VbPohKrnzuFXZDv';
$UTMify_url = 'https://api.utmify.com.br/api-credentials/orders';

// Diretório onde os UTMs são salvos
define('UTM_STORAGE_DIR', __DIR__ . '/utm_storage/');

// ============================================
// FUNÇÃO PARA REGISTRAR LOGS
// ============================================
function logMessage($file, $message) {
    $logDir = __DIR__ . '/logs/';
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logDir . $file, "[" . date("Y-m-d H:i:s") . "] " . $message . "\n", FILE_APPEND);
}

// ============================================
// FUNÇÃO PARA OBTER DATA EM UTC (ISO 8601)
// ============================================
function getUtcDateTime() {
    $dt = new DateTime('now', new DateTimeZone('UTC'));
    return $dt->format('Y-m-d H:i:s');
}

// ============================================
// FUNÇÃO PARA CONVERTER DATA PARA UTC
// ============================================
function convertToUtc($dateString) {
    if (empty($dateString)) {
        return getUtcDateTime();
    }
    
    try {
        $dt = new DateTime($dateString);
        $dt->setTimezone(new DateTimeZone('UTC'));
        return $dt->format('Y-m-d H:i:s');
    } catch (Exception $e) {
        return getUtcDateTime();
    }
}

// ============================================
// FUNÇÃO PARA CONVERTER VALOR PARA NULL SE VAZIO
// ============================================
function emptyToNull($value) {
    if ($value === null || $value === '' || (is_string($value) && trim($value) === '')) {
        return null;
    }
    return $value;
}

// ============================================
// CAPTURA O PAYLOAD DO WEBHOOK
// ============================================
$input = file_get_contents('php://input');
logMessage('webhook_raw.log', "Payload bruto recebido: $input");

// Decodifica os dados
$data = json_decode($input, true);
if (!$data) {
    logMessage('webhook_error.log', "Dados do webhook inválidos: $input");
    http_response_code(400);
    exit("Erro: Dados inválidos.");
}

// Verificar se os dados estão dentro da chave `data`
$transactionData = $data['data'] ?? $data;

logMessage('webhook_processed.log', "Dados processados: " . json_encode($transactionData, JSON_UNESCAPED_UNICODE));

// ============================================
// EXTRAIR INFORMAÇÕES DA TRANSAÇÃO
// ============================================
$id_transacao = $transactionData['id'] ?? null;
$status = $transactionData['status'] ?? null;
$price = $transactionData['amount'] ?? null;
$paymentMethodRaw = $transactionData['paymentMethod'] ?? 'pix';

// ✅ Converte datas para UTC
$createdAt = convertToUtc($transactionData['createdAt'] ?? null);
$updatedAt = isset($transactionData['updatedAt']) ? convertToUtc($transactionData['updatedAt']) : null;

// ============================================
// MAPEAR PAYMENT METHOD
// ============================================
function mapPaymentMethod($method) {
    $mapping = [
        'pix' => 'pix',
        'creditcard' => 'credit_card',
        'credit_card' => 'credit_card',
        'boleto' => 'boleto',
        'billet' => 'boleto',
    ];
    return $mapping[strtolower($method ?? 'pix')] ?? 'pix';
}

$paymentMethod = mapPaymentMethod($paymentMethodRaw);

// ============================================
// EXTRAIR DADOS DO CLIENTE
// ============================================
$customer = $transactionData['customer'] ?? [];
$customer_name = $customer['name'] ?? null;
$customer_email = $customer['email'] ?? null;
$customer_phone = $customer['phone'] ?? null;

if (isset($customer['document'])) {
    $customer_document = is_array($customer['document']) 
        ? ($customer['document']['number'] ?? null) 
        : $customer['document'];
} else {
    $customer_document = null;
}

// ============================================
// VALIDAÇÃO
// ============================================
$missing_fields = [];
if (!$id_transacao) $missing_fields[] = 'id';
if (!$status) $missing_fields[] = 'status';
if (!$price) $missing_fields[] = 'amount';
if (!$customer_name) $missing_fields[] = 'customer.name';
if (!$customer_email) $missing_fields[] = 'customer.email';

if (!empty($missing_fields)) {
    logMessage('webhook_error.log', "Campos ausentes na transação $id_transacao: " . implode(', ', $missing_fields));
    http_response_code(400);
    exit("Erro: Campos obrigatórios ausentes.");
}

// ============================================
// RECUPERAR UTMs SALVAS
// ============================================
$utm_file = UTM_STORAGE_DIR . $id_transacao . '.json';
$utm_data = [
    'utm_source' => null,
    'utm_medium' => null,
    'utm_campaign' => null,
    'utm_term' => null,
    'utm_content' => null,
    'sck' => null,
    'src' => null,
    'customer_ip' => null
];

if (file_exists($utm_file)) {
    $saved_utm = json_decode(file_get_contents($utm_file), true);
    if ($saved_utm) {
        foreach ($saved_utm as $key => $value) {
            $utm_data[$key] = emptyToNull($value);
        }
        logMessage('webhook_utm.log', "UTMs recuperadas para $id_transacao: " . json_encode($saved_utm));
    }
} else {
    logMessage('webhook_utm.log', "Arquivo UTM não encontrado para $id_transacao");
}

// ============================================
// EXTRAIR PRODUTOS
// ============================================
$products = $transactionData['products'] ?? $transactionData['items'] ?? [];
$product_info = [];

if (!empty($products)) {
    foreach ($products as $index => $product) {
        $product_info[] = [
            'id' => 'PROD' . str_pad($index + 1, 2, '0', STR_PAD_LEFT),
            'name' => $product['title'] ?? $product['name'] ?? 'Produto',
            'planId' => 'PLAN01',
            'planName' => 'Plano Único',
            'quantity' => (int) ($product['quantity'] ?? 1),
            'priceInCents' => (int) ($product['unitPrice'] ?? $product['priceInCents'] ?? $price),
        ];
    }
}

if (empty($product_info)) {
    $product_info[] = [
        'id' => 'PROD01',
        'name' => 'Produto Digital',
        'planId' => 'PLAN01',
        'planName' => 'Plano Único',
        'quantity' => 1,
        'priceInCents' => (int) $price,
    ];
}

// ============================================
// MAPEAR STATUS PARA UTMIFY
// ============================================
function mapStatusToUtmify($status) {
    $statusMap = [
        'pending' => 'waiting_payment',
        'waiting_payment' => 'waiting_payment',
        'processing' => 'waiting_payment',
        'paid' => 'paid',
        'approved' => 'paid',
        'completed' => 'paid',
        'confirmed' => 'paid',
        'success' => 'paid',
        'cancelled' => 'refused',
        'canceled' => 'refused',
        'failed' => 'refused',
        'refused' => 'refused',
        'expired' => 'refused',
        'refunded' => 'refunded',
        'chargeback' => 'chargeback',
    ];
    return $statusMap[strtolower($status)] ?? $status;
}

$utmifyStatus = mapStatusToUtmify($status);

// ============================================
// PREPARAR DADOS PARA UTMIFY
// ============================================

// Formata telefone
$formattedPhone = null;
if (!empty($customer_phone)) {
    $cleanPhone = preg_replace('/\D/', '', $customer_phone);
    $formattedPhone = (strpos($customer_phone, '+') === 0) ? $customer_phone : '+55' . $cleanPhone;
}

$infoVenda = [
    'orderId' => (string) $id_transacao,
    'platform' => 'Umbrellapag',
    'paymentMethod' => $paymentMethod,
    'status' => $utmifyStatus,
    'createdAt' => $createdAt, // ✅ UTC
    'approvedDate' => ($utmifyStatus === 'paid') ? ($updatedAt ?? $createdAt) : null, // ✅ UTC
    'refundedAt' => ($utmifyStatus === 'refunded') ? $updatedAt : null,
    'customer' => [
        'name' => $customer_name,
        'email' => $customer_email,
        'phone' => $formattedPhone,
        'document' => $customer_document,
        'country' => 'BR',
        'ip' => $utm_data['customer_ip'] ?? $_SERVER['REMOTE_ADDR'] ?? null,
    ],
    'products' => $product_info,
    'trackingParameters' => [
        'src' => emptyToNull($utm_data['src']),
        'sck' => emptyToNull($utm_data['sck']),
        'utm_source' => emptyToNull($utm_data['utm_source']),
        'utm_medium' => emptyToNull($utm_data['utm_medium']),
        'utm_campaign' => emptyToNull($utm_data['utm_campaign']),
        'utm_term' => emptyToNull($utm_data['utm_term']),
        'utm_content' => emptyToNull($utm_data['utm_content']),
    ],
    'commission' => [
        'totalPriceInCents' => (int) $price,
        'gatewayFeeInCents' => (int) ($transactionData['fee']['fixedAmount'] ?? $transactionData['fees'] ?? 0),
        'userCommissionInCents' => (int) ($transactionData['fee']['netAmount'] ?? ($price - ($transactionData['fees'] ?? 0))),
        'currency' => 'BRL',
    ],
    'isTest' => false,
];

$arrayInfo = json_encode($infoVenda, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
logMessage('utmify_request.log', "Enviando para UTMify (Transação $id_transacao | Status: $utmifyStatus | CreatedAt: $createdAt): $arrayInfo");

// ============================================
// ENVIAR PARA UTMIFY
// ============================================
$UTMify_curl = curl_init();
curl_setopt_array($UTMify_curl, [
    CURLOPT_URL => $UTMify_url,
    CURLOPT_POST => true,
    CURLOPT_POSTFIELDS => $arrayInfo,
    CURLOPT_HTTPHEADER => [
        'Content-Type: application/json',
        'x-api-token: ' . $UTMify_Token,
    ],
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_SSL_VERIFYHOST => 2,
    CURLOPT_SSL_VERIFYPEER => true,
]);

$UTMify_response = curl_exec($UTMify_curl);
$UTMify_httpCode = curl_getinfo($UTMify_curl, CURLINFO_HTTP_CODE);
$UTMify_error = curl_error($UTMify_curl);
curl_close($UTMify_curl);

if ($UTMify_error) {
    logMessage('utmify_error.log', "Erro cURL (Transação $id_transacao): $UTMify_error");
    http_response_code(500);
    exit("Erro ao enviar para UTMify.");
}

logMessage('utmify_response.log', "Resposta UTMify (Transação $id_transacao | HTTP $UTMify_httpCode): $UTMify_response");

// ============================================
// RESPOSTA
// ============================================
if ($UTMify_httpCode >= 200 && $UTMify_httpCode < 300) {
    logMessage('webhook_success.log', "✅ Webhook processado: $id_transacao (Status: $utmifyStatus)");
    http_response_code(200);
    echo json_encode(['success' => true, 'transactionId' => $id_transacao, 'status' => $utmifyStatus]);
} else {
    logMessage('utmify_error.log', "❌ Erro HTTP UTMify (Transação $id_transacao): HTTP $UTMify_httpCode | Response: $UTMify_response");
    http_response_code(200);
    echo json_encode(['success' => false, 'httpCode' => $UTMify_httpCode, 'response' => $UTMify_response]);
}