<?php
/**
 * Status Check - Umbrellapag API
 * Verifica o status de transações PIX na Umbrellapag
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Responde OPTIONS para CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// ============================================
// CONFIGURAÇÃO DA API UMBRELLAPAG
// ============================================
define('API_BASE_URL', 'https://api-gateway.umbrellapag.com');
define('API_TOKEN', 'd46fa92f-9484-409f-a82a-e2734029979e');
define('USER_AGENT', 'AtivoB2B/1.0');

// ============================================
// RECEBIDO VIA POST
// ============================================
$id_transacao = $_POST['id_transacao_post'] ?? $_GET['id_transacao'] ?? '';
$sck = $_POST['sck_post'] ?? '';
$src = $_POST['src_post'] ?? '';
$utm_campaign = $_POST['utm_campaign_post'] ?? '';
$utm_content = $_POST['utm_content_post'] ?? '';
$utm_medium = $_POST['utm_medium_post'] ?? '';
$utm_source = $_POST['utm_source_post'] ?? '';
$utm_term = $_POST['utm_term_post'] ?? '';

// ============================================
// FUNÇÃO PARA REGISTRAR LOGS
// ============================================
function logMessage($file, $message) {
    $logDir = dirname($file);
    if ($logDir !== '.' && !is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($file, "[" . date("Y-m-d H:i:s") . "] " . $message . "\n", FILE_APPEND);
}

// ============================================
// VERIFICAÇÃO INICIAL
// ============================================
if (empty($id_transacao)) {
    logMessage('logs/error_log.log', "ID da transação ausente");
    echo json_encode([
        'success' => false,
        'error' => 'Erro: ID da transação não fornecido.'
    ]);
    exit;
}

// ============================================
// FUNÇÃO PARA FAZER REQUISIÇÃO À API UMBRELLAPAG
// ============================================
function makeRequest($url, $method = 'GET', $data = null) {
    $headers = [
        'Content-Type: application/json',
        'User-Agent: ' . USER_AGENT,
        'x-api-key: ' . API_TOKEN
    ];
    
    $curl = curl_init($url);
    curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $method);
    curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($curl, CURLOPT_TIMEOUT, 30);
    
    if ($data && $method === 'POST') {
        curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($data));
    }
    
    $response = curl_exec($curl);
    $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    
    if ($response === false) {
        $error = curl_error($curl);
        curl_close($curl);
        return [
            'success' => false,
            'error' => $error,
            'httpCode' => $httpCode
        ];
    }
    
    curl_close($curl);
    
    return [
        'success' => ($httpCode >= 200 && $httpCode < 300),
        'response' => json_decode($response, true),
        'httpCode' => $httpCode
    ];
}

// ============================================
// CONSULTAR STATUS DA TRANSAÇÃO PIX
// ============================================
function getPixTransactionStatus($transactionId) {
    $url = API_BASE_URL . '/api/user/transactions/' . $transactionId;
    return makeRequest($url, 'GET');
}

// ============================================
// OBTER STATUS DA TRANSAÇÃO
// ============================================
$transactionStatus = getPixTransactionStatus($id_transacao);

// ============================================
// PROCESSAR E RETORNAR RESPOSTA
// ============================================
if ($transactionStatus['success'] && isset($transactionStatus['response'])) {
    // Log do retorno da API
    logMessage('logs/api_response_log.log', "ID: $id_transacao | Resposta: " . json_encode($transactionStatus['response']));
    
    // Extrai os dados da transação
    $txData = $transactionStatus['response']['data'] ?? $transactionStatus['response'];
    
    // Mapeia o status para um formato padronizado
    $status = $txData['status'] ?? 'unknown';
    $isPaid = in_array(strtolower($status), ['paid', 'approved', 'completed', 'success']);
    
    echo json_encode([
        'success' => true,
        'transactionId' => $id_transacao,
        'status' => $status,
        'isPaid' => $isPaid,
        'amount' => $txData['amount'] ?? null,
        'paidAt' => $txData['paidAt'] ?? $txData['paid_at'] ?? null,
        'createdAt' => $txData['createdAt'] ?? $txData['created_at'] ?? null,
        'customer' => $txData['customer'] ?? null,
        'raw' => $txData // Dados completos para debug
    ]);
    
} else {
    // Log de erro
    $errorMsg = $transactionStatus['error'] ?? 'Erro desconhecido';
    $httpCode = $transactionStatus['httpCode'] ?? 0;
    
    logMessage('logs/api_error_log.log', "ID: $id_transacao | HTTP: $httpCode | Erro: $errorMsg | Response: " . json_encode($transactionStatus['response'] ?? []));
    
    echo json_encode([
        'success' => false,
        'transactionId' => $id_transacao,
        'error' => 'Erro na comunicação com a API.',
        'httpCode' => $httpCode,
        'details' => $transactionStatus['response'] ?? null
    ]);
}
?>