<?php
/**
 * Backend - API Umbrellapag
 * Este arquivo faz a comunicação com a API de pagamentos
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Responde OPTIONS para CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// ============================================
// CONFIGURAÇÃO DA API
// ============================================
define('API_BASE_URL', 'https://api-gateway.umbrellapag.com');
define('API_TOKEN', 'd46fa92f-9484-409f-a82a-e2734029979e');
define('USER_AGENT', 'AtivoB2B/1.0');

// ============================================
// FUNÇÃO PARA FAZER REQUISIÇÕES À API
// ============================================
function apiRequest($endpoint, $method = 'POST', $data = null) {
    $url = API_BASE_URL . $endpoint;
    
    $headers = [
        'Content-Type: application/json',
        'User-Agent: ' . USER_AGENT,
        'x-api-key: ' . API_TOKEN
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    if ($method === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($data) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        return [
            'success' => false,
            'error' => $error,
            'httpCode' => $httpCode
        ];
    }
    
    $decoded = json_decode($response, true);
    
    return [
        'success' => ($httpCode >= 200 && $httpCode < 300),
        'httpCode' => $httpCode,
        'data' => $decoded
    ];
}

// ============================================
// CRIAR TRANSAÇÃO PIX
// ============================================
function createPixTransaction($params) {
    // VALOR FIXO (em reais) - altere aqui
    $valorFixo = 47.90;
    
    // DADOS PADRÃO - usados quando não vem na URL
    $defaults = [
        'name' => 'Pedro Silva',
        'email' => 'cliente@gmail.com',
        'document' => '00895889919', // CPF válido para testes
        'phone' => '11999999999',
        'productTitle' => 'Produto DigitalZ'
    ];
    
    // Usa o valor da URL ou o padrão
    $name = !empty($params['name']) ? $params['name'] : $defaults['name'];
    $email = !empty($params['email']) ? $params['email'] : $defaults['email'];
    $document = !empty($params['document']) ? $params['document'] : $defaults['document'];
    $phone = !empty($params['phone']) ? $params['phone'] : $defaults['phone'];
    $productTitle = !empty($params['productTitle']) ? $params['productTitle'] : $defaults['productTitle'];
    
    // Converte o valor para centavos (API espera valor em centavos)
    $amountInCents = intval($valorFixo * 100);
    
    $data = [
        'amount' => $amountInCents,
        'currency' => 'BRL',
        'paymentMethod' => 'pix',
        'customer' => [
            'name' => $name,
            'email' => $email,
            'document' => [
                'number' => preg_replace('/\D/', '', $document),
                'type' => strlen(preg_replace('/\D/', '', $document)) > 11 ? 'CNPJ' : 'CPF'
            ],
            'phone' => preg_replace('/\D/', '', $phone)
        ],
        'items' => [
            [
                'title' => $productTitle,
                'unitPrice' => $amountInCents,
                'quantity' => 1,
                'tangible' => false
            ]
        ],
        'pix' => [
            'expiresInDays' => 1
        ]
    ];
    
    $result = apiRequest('/api/user/transactions', 'POST', $data);
    
    return $result;
}

// ============================================
// VERIFICAR STATUS DO PAGAMENTO
// ============================================
function checkPaymentStatus($transactionId) {
    $result = apiRequest('/api/user/transactions/' . $transactionId, 'GET');
    return $result;
}

// ============================================
// ROTEAMENTO
// ============================================
$action = $_GET['action'] ?? $_POST['action'] ?? '';

// Pega os dados do POST (JSON ou form-data)
$input = json_decode(file_get_contents('php://input'), true);
if (!$input) {
    $input = $_POST;
}

switch ($action) {
    case 'create-pix':
        // Validações
        $required = ['name', 'email', 'document', 'phone', 'amount'];
        $missing = [];
        
        foreach ($required as $field) {
            if (empty($input[$field])) {
                $missing[] = $field;
            }
        }
        
        if (!empty($missing)) {
            echo json_encode([
                'success' => false,
                'error' => 'Campos obrigatórios faltando: ' . implode(', ', $missing)
            ]);
            exit;
        }
        
        $result = createPixTransaction($input);
        
        if ($result['success'] && isset($result['data']['data'])) {
            $txData = $result['data']['data'];
            
            // Extrai o código PIX
            $pixCode = $txData['qrCode'] 
                ?? $txData['pix']['qrCode'] 
                ?? $txData['pix']['brCode'] 
                ?? $txData['pix']['emv'] 
                ?? $txData['brCode'] 
                ?? $txData['emv'] 
                ?? null;
            
            echo json_encode([
                'success' => true,
                'transactionId' => $txData['id'],
                'pixCode' => $pixCode,
                'amount' => $txData['amount'] ?? $input['amount'],
                'status' => $txData['status'] ?? 'pending',
                'raw' => $txData // Para debug
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'error' => $result['data']['message'] ?? $result['data']['error'] ?? 'Erro ao criar transação',
                'httpCode' => $result['httpCode'],
                'raw' => $result['data'] // Para debug
            ]);
        }
        break;
        
    case 'check-status':
        $transactionId = $input['transactionId'] ?? $_GET['transactionId'] ?? '';
        
        if (empty($transactionId)) {
            echo json_encode([
                'success' => false,
                'error' => 'ID da transação não informado'
            ]);
            exit;
        }
        
        $result = checkPaymentStatus($transactionId);
        
        if ($result['success'] && isset($result['data']['data'])) {
            echo json_encode([
                'success' => true,
                'status' => $result['data']['data']['status'] ?? 'pending',
                'raw' => $result['data']['data']
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'error' => 'Erro ao verificar status',
                'raw' => $result['data']
            ]);
        }
        break;
        
    default:
        echo json_encode([
            'success' => false,
            'error' => 'Ação não especificada. Use: ?action=create-pix ou ?action=check-status'
        ]);
        break;
}