<?php
/**
 * ============================================
 * PIX API - GERAÇÃO DE PAGAMENTO
 * ============================================
 * Suporta múltiplos gateways (Umbrella, TechByNet)
 * Salva dados para tracking (Facebook, UTMify)
 */

require_once __DIR__ . '/config.php';
@include_once __DIR__ . '/utmify.php';

setCorsHeaders();

$input = json_decode(file_get_contents('php://input'), true);

if (!$input) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Dados inválidos']);
    exit;
}

// Validação - aceita formato antigo (cliente) ou novo (campos diretos)
$cliente = $input['cliente'] ?? null;

if ($cliente) {
    // Formato antigo: { cliente: {...}, tracking: {...} }
    $nome = $cliente['nome'] ?? '';
    $cpf = preg_replace('/\D/', '', $cliente['cpf'] ?? '');
    $email = $cliente['email'] ?? '';
    $telefone = preg_replace('/\D/', '', $cliente['celular'] ?? '');
    $cotas = $cliente['cotas'] ?? 1;
    $valorCentavos = intval($input['valor'] ?? 0);
    $tracking = $input['tracking'] ?? [];
} else {
    // Formato novo: campos diretos
    $nome = trim($input['nome'] ?? '');
    $cpf = preg_replace('/\D/', '', $input['cpf'] ?? '');
    $email = trim($input['email'] ?? '');
    $telefone = preg_replace('/\D/', '', $input['telefone'] ?? '');
    $cotas = $input['cotas'] ?? 1;
    $valorCentavos = intval($input['valor'] ?? 990);
    $tracking = $input['tracking'] ?? [];
}

// Validações
if (strlen($nome) < 3) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Nome inválido']);
    exit;
}
if (strlen($cpf) !== 11) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'CPF inválido']);
    exit;
}
if ($valorCentavos <= 0) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Valor inválido']);
    exit;
}

$valorReais = $valorCentavos / 100;

logCheckout('pix.log', 'Criando PIX', [
    'valor' => $valorReais,
    'cliente' => $nome,
    'cotas' => $cotas,
    'produto' => PRODUTO_NOME
]);

try {
    // ============================================
    // OBTÉM CONFIGURAÇÃO DO GATEWAY ATIVO
    // ============================================
    $gateway = getGatewayConfig();
    $gatewayName = getGatewayName();
    
    logCheckout('pix.log', "Usando gateway: {$gateway['name']}", ['gateway' => $gatewayName]);
    
    // ============================================
    // MONTA PAYLOAD CONFORME O GATEWAY
    // ============================================
    
    if ($gatewayName === 'umbrella') {
        // ========== UMBRELLA ==========
        $txRef = 'TX' . time() . rand(1000, 9999);
        
        $payload = [
            'amount' => $valorCentavos,
            'currency' => 'BRL',
            'paymentMethod' => 'PIX',
            'installments' => 1,
            'customer' => [
                'name' => $nome,
                'email' => $email ?: 'cliente@email.com',
                'document' => [
                    'number' => $cpf,
                    'type' => strlen($cpf) > 11 ? 'CNPJ' : 'CPF'
                ],
                'phone' => $telefone,
                'externalRef' => $txRef,
                'address' => [
                    'street' => 'Avenida Paulista',
                    'streetNumber' => '1000',
                    'complement' => '',
                    'zipCode' => '01310100',
                    'neighborhood' => 'Bela Vista',
                    'city' => 'São Paulo',
                    'state' => 'SP',
                    'country' => 'BR'
                ]
            ],
            'items' => [[
                'title' => PRODUTO_NOME . ' - ' . $cotas . ' números',
                'unitPrice' => $valorCentavos,
                'quantity' => 1,
                'tangible' => false,
                'externalRef' => 'item-' . $txRef
            ]],
            'pix' => [
                'expiresInDays' => 1
            ],
            'shipping' => [
                'fee' => 0,
                'address' => [
                    'street' => 'Avenida Paulista',
                    'streetNumber' => '1000',
                    'complement' => '',
                    'zipCode' => '01310100',
                    'neighborhood' => 'Bela Vista',
                    'city' => 'São Paulo',
                    'state' => 'SP',
                    'country' => 'BR'
                ]
            ],
            'postbackUrl' => WEBHOOK_URL,
            'metadata' => json_encode(['pedido' => $txRef]),
            'traceable' => true,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1'
        ];
        
        $url = $gateway['api_url'] . '/api/user/transactions';
        $headers = [
            'Content-Type: application/json',
            'User-Agent: UMBRELLAB2B/1.0',
            'x-api-key: ' . $gateway['api_key']
        ];
        
    } elseif ($gatewayName === 'techbynet') {
        // ========== TECHBYNET ==========
        $txRef = 'TX' . time() . rand(1000, 9999);
        
        $payload = [
            'amount' => $valorCentavos,
            'currency' => 'BRL',
            'paymentMethod' => 'PIX',
            'installments' => 1,
            'customer' => [
                'name' => $nome,
                'email' => $email ?: 'cliente@email.com',
                'document' => [
                    'number' => $cpf,
                    'type' => 'CPF'
                ],
                'phone' => $telefone,
                'externalRef' => $txRef,
                'address' => [
                    'street' => 'Avenida Paulista',
                    'streetNumber' => '1000',
                    'complement' => '',
                    'zipCode' => '01310100',
                    'neighborhood' => 'Bela Vista',
                    'city' => 'São Paulo',
                    'state' => 'SP',
                    'country' => 'BR'
                ]
            ],
            'items' => [[
                'title' => PRODUTO_NOME . ' - ' . $cotas . ' números',
                'unitPrice' => $valorCentavos,
                'quantity' => 1,
                'tangible' => false,
                'externalRef' => 'item-' . $txRef
            ]],
            'pix' => [
                'expiresInDays' => 1
            ],
            'shipping' => [
                'fee' => 0,
                'address' => [
                    'street' => 'Avenida Paulista',
                    'streetNumber' => '1000',
                    'complement' => '',
                    'zipCode' => '01310100',
                    'neighborhood' => 'Bela Vista',
                    'city' => 'São Paulo',
                    'state' => 'SP',
                    'country' => 'BR'
                ]
            ],
            'postbackUrl' => WEBHOOK_URL,
            'metadata' => json_encode(['pedido' => $txRef]),
            'traceable' => true,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1'
        ];
        
        $url = $gateway['api_url'];
        $headers = [
            'Content-Type: application/json',
            'x-api-key: ' . $gateway['api_key'],
            'User-Agent: ' . ($gateway['user_agent'] ?: 'SmartCheckout/2.0')
        ];
        
        logCheckout('pix.log', 'TechByNet Request', [
            'url' => $url,
            'txRef' => $txRef
        ]);
        
    } else {
        throw new Exception("Gateway '$gatewayName' não suportado");
    }
    
    // ============================================
    // FAZ REQUISIÇÃO PARA O GATEWAY
    // ============================================
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => true
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    logCheckout('pix.log', 'Resposta do gateway', [
        'gateway' => $gatewayName,
        'http_code' => $httpCode,
        'response' => substr($response, 0, 500)
    ]);
    
    if ($error) {
        throw new Exception("Erro cURL: $error");
    }
    
    $responseData = json_decode($response, true);
    
    if (!$responseData) {
        throw new Exception("Resposta inválida do gateway");
    }
    
    // Log completo para debug
    logCheckout('pix.log', 'Resposta COMPLETA do gateway', [
        'gateway' => $gatewayName,
        'http_code' => $httpCode,
        'data' => $responseData
    ]);
    
    // ============================================
    // EXTRAI DADOS DA RESPOSTA
    // ============================================
    $txId = null;
    $pixCode = null;
    $pixBase64 = null;
    
    if ($gatewayName === 'umbrella') {
        // Umbrella: dados podem estar em vários níveis
        $txData = $responseData['data'] ?? $responseData;
        
        // Se data contém outro data (estrutura aninhada)
        if (isset($txData['data']) && is_array($txData['data'])) {
            $txData = $txData['data'];
        }
        
        $txId = $txData['id'] ?? $responseData['id'] ?? null;
        
        // Extrai QR Code PIX (TODOS os locais possíveis)
        $pixCode = $txData['qrCode'] 
            ?? $txData['qrcode']
            ?? $txData['brCode']
            ?? $txData['emv']
            ?? $txData['pix']['qrCode'] 
            ?? $txData['pix']['brCode'] 
            ?? $txData['pix']['emv']
            ?? $txData['pix']['qrcode']
            ?? $responseData['qrCode']
            ?? $responseData['qrcode']
            ?? $responseData['brCode']
            ?? $responseData['emv']
            ?? $responseData['data']['qrCode']
            ?? $responseData['data']['qrcode']
            ?? $responseData['data']['brCode']
            ?? null;
            
        $pixBase64 = $txData['qrCodeBase64'] 
            ?? $txData['qrcodeBase64']
            ?? $txData['pix']['qrCodeBase64']
            ?? $responseData['qrCodeBase64']
            ?? $responseData['qrcodeBase64']
            ?? null;
            
    } elseif ($gatewayName === 'techbynet') {
        // TechByNet: dados em data.data
        $txData = $responseData['data']['data'] ?? $responseData['data'] ?? $responseData;
        
        $txId = $txData['id'] ?? $responseData['data']['id'] ?? null;
            
        $pixCode = $txData['qrCode'] 
            ?? $txData['qrcode']
            ?? $txData['brCode']
            ?? $responseData['data']['qrCode']
            ?? $responseData['data']['data']['qrCode']
            ?? null;
            
        $pixBase64 = $txData['qrCodeBase64'] 
            ?? $txData['qrcodeBase64']
            ?? null;
            
        // payUrl também pode ser útil
        $payUrl = $txData['payUrl'] ?? null;
        if ($payUrl) {
            logCheckout('pix.log', 'TechByNet payUrl', ['url' => $payUrl]);
        }
    }
    
    // Log do que foi extraído
    logCheckout('pix.log', 'Dados extraídos', [
        'txId' => $txId,
        'pixCode_length' => $pixCode ? strlen($pixCode) : 0,
        'pixCode_preview' => $pixCode ? substr($pixCode, 0, 50) . '...' : null,
        'pixBase64' => $pixBase64 ? 'presente' : 'ausente'
    ]);
    
    // Se não encontrou txId, gera um
    if (!$txId) {
        $txId = 'TX' . time() . rand(1000, 9999);
        logCheckout('pix.log', 'txId gerado localmente', ['txId' => $txId]);
    }
    
    if (!$pixCode) {
        logCheckout('pix.log', 'ERRO: QR Code não encontrado na resposta', [
            'keys_nivel1' => array_keys($responseData),
            'keys_data' => isset($responseData['data']) ? array_keys($responseData['data']) : 'N/A'
        ]);
        throw new Exception("QR Code PIX não encontrado na resposta do gateway");
    }
    
    // ============================================
    // SALVA DADOS PARA WEBHOOK/TRACKING
    // ============================================
    $utmData = [
        'transaction_id' => $txId,
        'gateway' => $gatewayName,
        'created_at' => date('Y-m-d H:i:s'),
        'valor_centavos' => $valorCentavos,
        'cotas' => $cotas,
        'produto' => PRODUTO_NOME,
        
        // UTMs do tracking
        'utm_source' => emptyToNull($tracking['utm_source'] ?? null),
        'utm_medium' => emptyToNull($tracking['utm_medium'] ?? null),
        'utm_campaign' => emptyToNull($tracking['utm_campaign'] ?? null),
        'utm_term' => emptyToNull($tracking['utm_term'] ?? null),
        'utm_content' => emptyToNull($tracking['utm_content'] ?? null),
        'sck' => emptyToNull($tracking['sck'] ?? null),
        'src' => emptyToNull($tracking['src'] ?? null),
        'fbc' => emptyToNull($tracking['fbc'] ?? null),
        'fbp' => emptyToNull($tracking['fbp'] ?? null),
        
        // IP do cliente
        'customer_ip' => $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '',
        
        // Dados do cliente
        'customer' => [
            'name' => $nome,
            'email' => $email,
            'phone' => $telefone,
            'document' => $cpf
        ]
    ];
    
    // Salva JSON para webhook
    $saved = @file_put_contents(
        UTM_STORAGE_DIR . $txId . '.json',
        json_encode($utmData, JSON_PRETTY_PRINT)
    );
    
    logCheckout('pix.log', 'PIX criado com sucesso', [
        'tx_id' => $txId,
        'gateway' => $gatewayName,
        'valor' => $valorReais,
        'json_saved' => $saved !== false
    ]);
    
    // ============================================
    // ENVIA PARA UTMIFY (STATUS PENDENTE)
    // ============================================
    if (class_exists('Utmify')) {
        $utmify = new Utmify();
        $utmifyResult = $utmify->sendPending(
            $txId,
            $valorCentavos,
            [
                'name' => $nome,
                'email' => $email,
                'phone' => $telefone,
                'document' => $cpf,
                'ip' => $utmData['customer_ip']
            ],
            [
                'id' => PRODUTO_ID,
                'name' => PRODUTO_NOME,
                'quantity' => $cotas
            ],
            [
                'utm_source' => $tracking['utm_source'] ?? '',
                'utm_medium' => $tracking['utm_medium'] ?? '',
                'utm_campaign' => $tracking['utm_campaign'] ?? '',
                'utm_content' => $tracking['utm_content'] ?? '',
                'utm_term' => $tracking['utm_term'] ?? '',
                'sck' => $tracking['sck'] ?? '',
                'src' => $tracking['src'] ?? ''
            ]
        );
        
        logCheckout('pix.log', 'UTMify Pendente enviado', [
            'success' => $utmifyResult['success'] ?? false,
            'produto' => PRODUTO_NOME
        ]);
    }
    
    // ============================================
    // RESPOSTA DE SUCESSO
    // ============================================
    echo json_encode([
        'success' => true,
        'gateway' => $gateway['name'],
        'id' => $txId,
        'txid' => $txId,
        'pix' => [
            'qrcode' => $pixCode,
            'qrcodeBase64' => $pixBase64
        ],
        'qrcode' => $pixCode,
        'qrcodeBase64' => $pixBase64,
        'amount' => $valorCentavos,
        'valor' => $valorReais,
        'status' => 'pending'
    ]);
    
} catch (Exception $e) {
    logCheckout('pix.log', 'ERRO: ' . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}