<?php
/**
 * ============================================
 * WEBHOOK - NOTIFICAÇÕES DE PAGAMENTO
 * ============================================
 * Recebe notificações de todos os gateways
 */

require_once __DIR__ . '/config.php';
@include_once __DIR__ . '/facebook_capi.php';
@include_once __DIR__ . '/utmify.php';

// Log da requisição
$rawInput = file_get_contents('php://input');
$data = json_decode($rawInput, true);

logCheckout('webhook.log', 'Webhook recebido', [
    'method' => $_SERVER['REQUEST_METHOD'],
    'content_type' => $_SERVER['CONTENT_TYPE'] ?? '',
    'raw_length' => strlen($rawInput),
    'data_preview' => substr($rawInput, 0, 500)
]);

if (!$data) {
    http_response_code(400);
    echo json_encode(['error' => 'Payload inválido']);
    exit;
}

// ============================================
// DETECTA O GATEWAY AUTOMATICAMENTE
// ============================================
function detectGateway($data) {
    // Genesys: tem external_id e total_amount
    if (isset($data['external_id']) && isset($data['total_amount'])) {
        return 'genesys';
    }
    
    // AllowPay: tem objectId e type
    if (isset($data['objectId']) && isset($data['type'])) {
        return 'allowpay';
    }
    
    // Umbrella: tem objectId mas estrutura diferente
    if (isset($data['objectId']) && isset($data['data']['data'])) {
        return 'umbrella';
    }
    
    // TechByNet: tem transaction_id
    if (isset($data['transaction_id']) || isset($data['pix_code'])) {
        return 'techbynet';
    }
    
    // Fallback: usa gateway ativo
    return getGatewayName();
}

$detectedGateway = detectGateway($data);

logCheckout('webhook.log', 'Gateway detectado', ['gateway' => $detectedGateway]);

// ============================================
// EXTRAI DADOS DO WEBHOOK
// ============================================
$txId = null;
$status = null;
$amount = null;

if ($detectedGateway === 'genesys') {
    // Genesys Finance - formato simples
    $txId = $data['id'] ?? null;
    $status = $data['status'] ?? null;
    // total_amount está em REAIS, converter para centavos
    $amount = isset($data['total_amount']) ? intval($data['total_amount'] * 100) : null;
    
} elseif ($detectedGateway === 'allowpay') {
    // AllowPay
    $txId = $data['data']['id'] ?? $data['objectId'] ?? null;
    $status = $data['data']['status'] ?? null;
    $amount = $data['data']['amount'] ?? null;
    
} elseif ($detectedGateway === 'umbrella') {
    // Umbrella
    $txData = $data['data']['data'][0] ?? $data['data'] ?? [];
    $txId = $txData['id'] ?? $data['objectId'] ?? null;
    $status = $txData['status'] ?? null;
    $amount = $txData['amount'] ?? null;
    
} elseif ($detectedGateway === 'techbynet') {
    // TechByNet
    $txData = $data['data']['data'] ?? $data['data'] ?? $data;
    $txId = $txData['id'] ?? $data['transaction_id'] ?? null;
    $status = $txData['status'] ?? null;
    $amount = $txData['amount'] ?? null;
    
} else {
    // Genérico
    $txId = $data['data']['id'] ?? $data['id'] ?? $data['objectId'] ?? null;
    $status = $data['data']['status'] ?? $data['status'] ?? null;
    $amount = $data['data']['amount'] ?? $data['amount'] ?? null;
}

logCheckout('webhook.log', 'Dados extraídos', [
    'txId' => $txId,
    'status' => $status,
    'amount' => $amount
]);

if (!$txId) {
    logCheckout('webhook.log', 'ERRO: txId não encontrado');
    http_response_code(400);
    echo json_encode(['error' => 'Transaction ID não encontrado']);
    exit;
}

// ============================================
// VERIFICA SE É PAGAMENTO CONFIRMADO
// ============================================
$isPaid = isPaidStatus($status);

if ($isPaid) {
    logCheckout('webhook.log', 'PAGAMENTO CONFIRMADO', [
        'txId' => $txId,
        'status' => $status,
        'amount' => $amount
    ]);
    
    // Carrega dados do UTM salvos
    $utmFile = UTM_STORAGE_DIR . $txId . '.json';
    $utmData = [];
    
    if (file_exists($utmFile)) {
        $utmData = json_decode(file_get_contents($utmFile), true) ?: [];
        logCheckout('webhook.log', 'UTM data carregado', ['file' => $utmFile]);
    }
    
    // ============================================
    // FACEBOOK CAPI - PURCHASE
    // ============================================
    if (class_exists('FacebookCAPI')) {
        try {
            $fb = new FacebookCAPI();
            $fb->sendPurchase(
                $txId,
                ($amount ?: $utmData['valor_centavos'] ?? 0) / 100,
                $utmData
            );
            logCheckout('webhook.log', 'Facebook CAPI Purchase enviado');
        } catch (Exception $e) {
            logCheckout('webhook.log', 'Erro Facebook CAPI: ' . $e->getMessage());
        }
    }
    
    // ============================================
    // UTMIFY - PAID
    // ============================================
    if (class_exists('Utmify')) {
        try {
            $utmify = new Utmify();
            
            // Prepara dados do cliente
            $customer = [
                'name' => $utmData['customer']['name'] ?? $utmData['nome'] ?? '',
                'email' => $utmData['customer']['email'] ?? $utmData['email'] ?? '',
                'phone' => $utmData['customer']['phone'] ?? $utmData['telefone'] ?? '',
                'document' => $utmData['customer']['document'] ?? $utmData['cpf'] ?? '',
                'ip' => $utmData['customer_ip'] ?? $_SERVER['REMOTE_ADDR'] ?? ''
            ];
            
            // Prepara dados do produto
            $product = [
                'id' => $utmData['produto_id'] ?? (defined('PRODUTO_ID') ? PRODUTO_ID : 'produto-001'),
                'name' => $utmData['produto'] ?? (defined('PRODUTO_NOME') ? PRODUTO_NOME : 'Produto'),
                'quantity' => $utmData['cotas'] ?? 1
            ];
            
            // UTMs
            $utms = [
                'utm_source' => $utmData['utm_source'] ?? '',
                'utm_medium' => $utmData['utm_medium'] ?? '',
                'utm_campaign' => $utmData['utm_campaign'] ?? '',
                'utm_content' => $utmData['utm_content'] ?? '',
                'utm_term' => $utmData['utm_term'] ?? '',
                'sck' => $utmData['sck'] ?? '',
                'src' => $utmData['src'] ?? ''
            ];
            
            $utmifyResult = $utmify->sendPaid(
                $txId,
                $amount ?: $utmData['valor_centavos'] ?? 0,
                $customer,
                $product,
                $utms
            );
            
            logCheckout('webhook.log', 'UTMify Paid enviado', $utmifyResult);
        } catch (Exception $e) {
            logCheckout('webhook.log', 'Erro UTMify: ' . $e->getMessage());
        }
    }
    
    // Marca como processado
    $utmData['paid_at'] = date('Y-m-d H:i:s');
    $utmData['webhook_status'] = $status;
    @file_put_contents($utmFile, json_encode($utmData, JSON_PRETTY_PRINT));
}

// Resposta de sucesso
http_response_code(200);
echo json_encode([
    'success' => true,
    'txId' => $txId,
    'status' => $status,
    'isPaid' => $isPaid,
    'gateway' => $detectedGateway
]);