<?php
/**
 * ============================================
 * UPLOAD API - COM APROVAÇÃO AUTOMÁTICA
 * Usando OCR.space para extrair dados
 * ============================================
 */
error_reporting(0);
date_default_timezone_set('America/Sao_Paulo');

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// ========== CONFIGURAÇÕES ==========
define('OCR_API_KEY', 'K82516147688957');
define('OCR_API_URL', 'https://api.ocr.space/parse/image');
define('SCORE_MINIMO', 70);
define('VALOR_TOLERANCIA', 0.50);
define('LOG_FILE', __DIR__ . '/../logs/upload-auto.log');

/**
 * Log
 */
function logMsg($msg) {
    $logDir = dirname(LOG_FILE);
    if (!is_dir($logDir)) @mkdir($logDir, 0755, true);
    @file_put_contents(LOG_FILE, "[" . date('Y-m-d H:i:s') . "] $msg\n", FILE_APPEND);
}

/**
 * Extrai texto usando API OCR.space
 */
function extrairTextoOCR($imagePath) {
    if (!file_exists($imagePath)) {
        logMsg("Arquivo não existe: $imagePath");
        return '';
    }
    
    $ext = strtolower(pathinfo($imagePath, PATHINFO_EXTENSION));
    $mimeTypes = [
        'jpg' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'png' => 'image/png',
        'gif' => 'image/gif',
        'webp' => 'image/webp',
        'pdf' => 'application/pdf'
    ];
    $mimeType = $mimeTypes[$ext] ?? 'image/jpeg';
    
    $curl = curl_init();
    
    $postFields = [
        'apikey' => OCR_API_KEY,
        'language' => 'por',
        'isOverlayRequired' => 'false',
        'detectOrientation' => 'true',
        'scale' => 'true',
        'OCREngine' => '2',
        'file' => new CURLFile($imagePath, $mimeType, basename($imagePath))
    ];
    
    curl_setopt_array($curl, [
        CURLOPT_URL => OCR_API_URL,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $postFields,
        CURLOPT_TIMEOUT => 60,
        CURLOPT_SSL_VERIFYPEER => false
    ]);
    
    $response = curl_exec($curl);
    $error = curl_error($curl);
    curl_close($curl);
    
    if ($error) {
        logMsg("Erro CURL: $error");
        return '';
    }
    
    $result = json_decode($response, true);
    
    if (isset($result['ParsedResults'][0]['ParsedText'])) {
        $texto = $result['ParsedResults'][0]['ParsedText'];
        logMsg("OCR extraiu " . strlen($texto) . " caracteres");
        return $texto;
    }
    
    logMsg("OCR não extraiu texto");
    return '';
}

/**
 * Extrai dados do comprovante
 */
function extrairDados($texto) {
    $dados = [
        'valor' => null,
        'cpf' => null,
        'nome' => null,
        'data' => null
    ];
    
    $texto = mb_strtoupper($texto, 'UTF-8');
    $textoLimpo = preg_replace('/\s+/', ' ', $texto);
    
    // VALOR
    $padroesValor = [
        '/R\$\s*(\d{1,3}(?:\.\d{3})*[,]\d{2})/',
        '/R\$\s*(\d{1,3}[,]\d{2})/',
        '/VALOR[:\s]*R?\$?\s*(\d{1,3}(?:\.\d{3})*[,]\d{2})/i',
        '/(\d{2,3})[,](\d{2})(?=\s|$|[^0-9])/'
    ];
    
    foreach ($padroesValor as $padrao) {
        if (preg_match($padrao, $textoLimpo, $m)) {
            $valorStr = isset($m[2]) ? $m[1] . ',' . $m[2] : $m[1];
            $valorStr = str_replace('.', '', $valorStr);
            $valorStr = str_replace(',', '.', $valorStr);
            $valor = floatval($valorStr);
            if ($valor > 0 && $valor < 100000) {
                $dados['valor'] = $valor;
                break;
            }
        }
    }
    
    // CPF
    if (preg_match('/(\d{3})[\.\s]?(\d{3})[\.\s]?(\d{3})[\-\.\s]?(\d{2})/', $textoLimpo, $m)) {
        $cpf = $m[1] . $m[2] . $m[3] . $m[4];
        if (strlen($cpf) === 11) {
            $dados['cpf'] = $cpf;
        }
    }
    
    // NOME
    $padroesNome = [
        '/(?:PAGADOR|ORIGEM|REMETENTE|DE|NOME)[:\s]+([A-ZÁÀÂÃÉÈÊÍÌÎÓÒÔÕÚÙÛÇ][A-ZÁÀÂÃÉÈÊÍÌÎÓÒÔÕÚÙÛÇ\s]{2,40})(?=\s{2}|CPF|CNPJ|\d|$)/i',
        '/(?:TRANSFERIU|ENVIOU|PAGOU)[:\s]+([A-ZÁÀÂÃÉÈÊÍÌÎÓÒÔÕÚÙÛÇ][A-ZÁÀÂÃÉÈÊÍÌÎÓÒÔÕÚÙÛÇ\s]{2,40})/i'
    ];
    
    foreach ($padroesNome as $padrao) {
        if (preg_match($padrao, $textoLimpo, $m)) {
            $nome = trim($m[1]);
            $ignorar = ['PIX', 'BANCO', 'COMPROVANTE', 'TRANSFERENCIA', 'PAGAMENTO', 'VALOR'];
            $ehNome = true;
            foreach ($ignorar as $ig) {
                if (stripos($nome, $ig) !== false) {
                    $ehNome = false;
                    break;
                }
            }
            if ($ehNome && strlen($nome) >= 3) {
                $dados['nome'] = $nome;
                break;
            }
        }
    }
    
    // DATA
    if (preg_match('/(\d{2})[\/\-](\d{2})[\/\-](\d{4})/', $textoLimpo, $m)) {
        $dados['data'] = $m[3] . '-' . $m[2] . '-' . $m[1];
    }
    
    return $dados;
}

/**
 * Calcula score de compatibilidade
 */
function calcularScore($dadosOCR, $cliente) {
    $score = 0;
    $detalhes = [];
    
    $valorCliente = floatval($cliente['valor'] ?? 0);
    $cpfCliente = preg_replace('/\D/', '', $cliente['cpf'] ?? '');
    $nomeCliente = mb_strtoupper(trim($cliente['nome'] ?? ''), 'UTF-8');
    
    // VALOR (40 pontos)
    if ($dadosOCR['valor'] && $valorCliente > 0) {
        $diff = abs($dadosOCR['valor'] - $valorCliente);
        
        if ($diff <= VALOR_TOLERANCIA) {
            $score += 40;
            $detalhes[] = "✅ Valor EXATO";
        } elseif ($diff <= 2.00) {
            $score += 25;
            $detalhes[] = "⚠️ Valor próximo";
        } elseif ($diff <= 10.00) {
            $score += 10;
            $detalhes[] = "⚠️ Valor similar";
        } else {
            $detalhes[] = "❌ Valor diferente";
        }
    } else {
        $detalhes[] = "⚠️ Valor não identificado";
    }
    
    // CPF (30 pontos)
    if ($dadosOCR['cpf'] && $cpfCliente) {
        if ($dadosOCR['cpf'] === $cpfCliente) {
            $score += 30;
            $detalhes[] = "✅ CPF conferido";
        } elseif (substr($dadosOCR['cpf'], 0, 6) === substr($cpfCliente, 0, 6)) {
            $score += 15;
            $detalhes[] = "⚠️ CPF parcial";
        } else {
            $detalhes[] = "❌ CPF diferente";
        }
    } else {
        $detalhes[] = "⚠️ CPF não identificado";
    }
    
    // NOME (20 pontos)
    if ($dadosOCR['nome'] && $nomeCliente) {
        $nomeOCR = mb_strtoupper(trim($dadosOCR['nome']), 'UTF-8');
        
        if ($nomeOCR === $nomeCliente) {
            $score += 20;
            $detalhes[] = "✅ Nome conferido";
        } else {
            $primeiroOCR = explode(' ', $nomeOCR)[0];
            $primeiroCliente = explode(' ', $nomeCliente)[0];
            
            if ($primeiroOCR === $primeiroCliente && strlen($primeiroOCR) >= 3) {
                $score += 15;
                $detalhes[] = "✅ Primeiro nome confere";
            } else {
                similar_text($nomeOCR, $nomeCliente, $percent);
                if ($percent >= 70) {
                    $score += 10;
                    $detalhes[] = "⚠️ Nome similar";
                } else {
                    $detalhes[] = "❌ Nome diferente";
                }
            }
        }
    } else {
        $detalhes[] = "⚠️ Nome não identificado";
    }
    
    // DATA (10 pontos)
    if ($dadosOCR['data']) {
        $dataOCR = strtotime($dadosOCR['data']);
        $hoje = strtotime('today');
        $diffDias = abs($dataOCR - $hoje) / 86400;
        
        if ($diffDias <= 3) {
            $score += 10;
            $detalhes[] = "✅ Data recente";
        } elseif ($diffDias <= 7) {
            $score += 5;
            $detalhes[] = "⚠️ Data ok";
        }
    }
    
    // BÔNUS
    $temValor = $dadosOCR['valor'] && abs($dadosOCR['valor'] - $valorCliente) <= VALOR_TOLERANCIA;
    $temCpfOuNome = ($dadosOCR['cpf'] === $cpfCliente) || 
                    (mb_strtoupper($dadosOCR['nome'] ?? '') === $nomeCliente);
    
    if ($temValor && $temCpfOuNome) {
        $score += 10;
        $detalhes[] = "🎯 Bônus";
    }
    
    return [
        'score' => $score,
        'detalhes' => $detalhes,
        'aprovado' => $score >= SCORE_MINIMO,
        'dados_ocr' => $dadosOCR
    ];
}

// ========== MAIN ==========
try {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Método não permitido');
    }
    
    if (!isset($_FILES['comprovante'])) {
        throw new Exception('Nenhum arquivo enviado');
    }
    
    if ($_FILES['comprovante']['error'] !== UPLOAD_ERR_OK) {
        $erros = [
            UPLOAD_ERR_INI_SIZE => 'Arquivo muito grande',
            UPLOAD_ERR_FORM_SIZE => 'Arquivo muito grande',
            UPLOAD_ERR_PARTIAL => 'Upload incompleto',
            UPLOAD_ERR_NO_FILE => 'Nenhum arquivo',
            UPLOAD_ERR_NO_TMP_DIR => 'Erro no servidor',
            UPLOAD_ERR_CANT_WRITE => 'Erro ao salvar',
        ];
        throw new Exception($erros[$_FILES['comprovante']['error']] ?? 'Erro no upload');
    }
    
    $file = $_FILES['comprovante'];
    $pixId = $_POST['pix_id'] ?? '';
    $clienteJson = $_POST['cliente'] ?? '{}';
    
    if (empty($pixId)) {
        throw new Exception('ID do PIX não informado');
    }
    
    // Valida extensão
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $allowedExt = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'pdf'];
    
    if (!in_array($ext, $allowedExt)) {
        throw new Exception('Tipo de arquivo não permitido. Use: ' . implode(', ', $allowedExt));
    }
    
    // Valida tamanho (max 10MB)
    if ($file['size'] > 10 * 1024 * 1024) {
        throw new Exception('Arquivo muito grande (máx 10MB)');
    }
    
    // Cria pasta de comprovantes
    $uploadDir = __DIR__ . '/../comprovantes/';
    if (!is_dir($uploadDir)) {
        if (!mkdir($uploadDir, 0755, true)) {
            throw new Exception('Erro ao criar pasta de uploads');
        }
    }
    
    if (!is_writable($uploadDir)) {
        throw new Exception('Sem permissão para salvar arquivos');
    }
    
    // Remove do rejeitados.json se existir
    $rejeitadosFile = $uploadDir . 'rejeitados.json';
    if (file_exists($rejeitadosFile)) {
        $rejeitados = json_decode(file_get_contents($rejeitadosFile), true) ?: [];
        $totalAntes = count($rejeitados);
        $rejeitados = array_filter($rejeitados, function($r) use ($pixId) {
            return ($r['pix_id'] ?? '') !== $pixId;
        });
        if (count($rejeitados) < $totalAntes) {
            file_put_contents($rejeitadosFile, json_encode(array_values($rejeitados), JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        }
    }
    
    // Gera nome único
    $filename = date('Y-m-d_H-i-s') . '_' . preg_replace('/[^a-zA-Z0-9]/', '', $pixId) . '.' . $ext;
    $filepath = $uploadDir . $filename;
    
    // Move arquivo
    if (!move_uploaded_file($file['tmp_name'], $filepath)) {
        throw new Exception('Erro ao salvar arquivo');
    }
    
    // Parse cliente
    $cliente = json_decode($clienteJson, true) ?: [];
    
    logMsg("=== UPLOAD: $filename ===");
    logMsg("Cliente: " . ($cliente['nome'] ?? 'N/A') . " | Valor: R$ " . ($cliente['valor'] ?? 0));
    
    // ========== TENTA APROVAÇÃO AUTOMÁTICA ==========
    $autoAprovado = false;
    $scoreInfo = null;
    
    // Extrai texto via OCR
    $texto = extrairTextoOCR($filepath);
    
    if (!empty($texto)) {
        // Extrai dados
        $dadosOCR = extrairDados($texto);
        
        logMsg("Dados OCR: valor=" . ($dadosOCR['valor'] ?? 'N/A') . " | cpf=" . ($dadosOCR['cpf'] ?? 'N/A') . " | nome=" . ($dadosOCR['nome'] ?? 'N/A'));
        
        // Calcula score
        $scoreInfo = calcularScore($dadosOCR, $cliente);
        
        logMsg("Score: {$scoreInfo['score']} | " . implode(' | ', $scoreInfo['detalhes']));
        
        if ($scoreInfo['aprovado']) {
            $autoAprovado = true;
            logMsg("✅ AUTO-APROVADO!");
        } else {
            logMsg("❌ Score insuficiente ({$scoreInfo['score']} < " . SCORE_MINIMO . ")");
        }
    } else {
        logMsg("⚠️ OCR não extraiu texto - vai para análise manual");
    }
    
    // Monta registro
    $registro = [
        'id' => uniqid('comp_'),
        'pix_id' => $pixId,
        'arquivo' => $filename,
        'cliente' => [
            'nome' => $cliente['nome'] ?? '',
            'cpf' => $cliente['cpf'] ?? '',
            'email' => $cliente['email'] ?? '',
            'celular' => $cliente['celular'] ?? '',
            'valor' => $cliente['valor'] ?? 0,
            'cotas' => $cliente['cotas'] ?? 0
        ],
        'data' => date('Y-m-d H:i:s'),
        'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
    ];
    
    if ($autoAprovado) {
        // Salva em aprovados.json
        $aprovadosFile = $uploadDir . 'aprovados.json';
        $aprovados = file_exists($aprovadosFile) ? json_decode(file_get_contents($aprovadosFile), true) : [];
        
        $registro['status'] = 'aprovado';
        $registro['aprovado_em'] = date('Y-m-d H:i:s');
        $registro['aprovado_por'] = 'AUTO_OCR';
        $registro['auto_approve'] = $scoreInfo;
        
        $aprovados[] = $registro;
        file_put_contents($aprovadosFile, json_encode($aprovados, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        
        echo json_encode([
            'success' => true,
            'message' => '✅ Comprovante aprovado automaticamente!',
            'auto_aprovado' => true,
            'score' => $scoreInfo['score']
        ]);
    } else {
        // Salva em pendentes.json
        $pendentesFile = $uploadDir . 'pendentes.json';
        $pendentes = file_exists($pendentesFile) ? json_decode(file_get_contents($pendentesFile), true) : [];
        
        $registro['status'] = 'pendente';
        if ($scoreInfo) {
            $registro['auto_check'] = $scoreInfo;
        }
        
        $pendentes[] = $registro;
        file_put_contents($pendentesFile, json_encode($pendentes, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        
        echo json_encode([
            'success' => true,
            'message' => '⏳ Comprovante enviado! Aguardando análise...',
            'auto_aprovado' => false,
            'score' => $scoreInfo['score'] ?? 0
        ]);
    }
    
} catch (Exception $e) {
    logMsg("ERRO: " . $e->getMessage());
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}