<?php
/**
 * ============================================
 * WEBHOOK - NOTIFICAÇÕES DE PAGAMENTO
 * ============================================
 * Versão: 3.0 - Usa configurações do gateways.json
 * 
 * Gateways suportados:
 * - AllowPay (ativo)
 * - Umbrella
 * - TechByNet
 * - Genesys Finance
 * 
 * IMPORTANTE: Este é o ÚNICO lugar que envia Purchase!
 */

date_default_timezone_set('America/Sao_Paulo');

require_once __DIR__ . '/config.php';
@include_once __DIR__ . '/facebook_capi.php';
@include_once __DIR__ . '/utmify.php';

// ============================================
// LOG DETALHADO
// ============================================
$rawInput = file_get_contents('php://input');
$data = json_decode($rawInput, true);

// Headers recebidos
$headers = [];
foreach ($_SERVER as $key => $value) {
    if (strpos($key, 'HTTP_') === 0) {
        $headers[str_replace('HTTP_', '', $key)] = $value;
    }
}

logCheckout('webhook.log', '========== WEBHOOK RECEBIDO ==========', [
    'method' => $_SERVER['REQUEST_METHOD'],
    'content_type' => $_SERVER['CONTENT_TYPE'] ?? 'não definido',
    'raw_length' => strlen($rawInput),
    'remote_ip' => $_SERVER['REMOTE_ADDR'] ?? 'desconhecido'
]);

// Log do payload completo para debug
logCheckout('webhook_payloads.log', 'Payload', [
    'timestamp' => date('Y-m-d H:i:s'),
    'raw' => substr($rawInput, 0, 3000)
]);

if (!$data) {
    logCheckout('webhook.log', '❌ ERRO: Payload inválido');
    http_response_code(400);
    echo json_encode(['error' => 'Payload inválido']);
    exit;
}

// ============================================
// CARREGA CONFIGURAÇÕES DOS GATEWAYS
// ============================================
$gatewaysConfig = loadGatewaysConfig();
$allGateways = $gatewaysConfig['gateways'] ?? [];

// ============================================
// DETECTA O GATEWAY AUTOMATICAMENTE
// ============================================
function detectGateway($data, $allGateways) {
    
    // 1. GENESYS: tem external_id OU (id + total_amount sem objectId)
    if (isset($data['external_id'])) {
        return 'genesys';
    }
    if (isset($data['id']) && isset($data['total_amount']) && !isset($data['objectId'])) {
        return 'genesys';
    }
    
    // 2. UMBRELLA: tem objectId E data.data como ARRAY
    if (isset($data['objectId']) && isset($data['data']['data']) && is_array($data['data']['data']) && isset($data['data']['data'][0])) {
        return 'umbrella';
    }
    
    // 3. TECHBYNET: tem data.data como OBJETO (não array)
    if (isset($data['data']['data']) && is_array($data['data']['data']) && !isset($data['data']['data'][0])) {
        return 'techbynet';
    }
    
    // 4. ALLOWPAY: tem id e status no root, sem data.data
    if (isset($data['id']) && isset($data['status']) && !isset($data['data']['data']) && !isset($data['total_amount'])) {
        return 'allowpay';
    }
    
    // 5. ALLOWPAY alternativo: tem objectId mas não tem data.data
    if (isset($data['objectId']) && !isset($data['data']['data'])) {
        return 'allowpay';
    }
    
    // 6. TECHBYNET: tem transaction_id ou pix_code
    if (isset($data['transaction_id']) || isset($data['pix_code'])) {
        return 'techbynet';
    }
    
    // 7. Fallback: usa gateway ativo da config
    return getGatewayName();
}

$detectedGateway = detectGateway($data, $allGateways);
$gatewayConfig = $allGateways[$detectedGateway] ?? [];

logCheckout('webhook.log', '🔍 Gateway: ' . strtoupper($detectedGateway), [
    'has_config' => !empty($gatewayConfig),
    'keys' => array_keys($data)
]);

// ============================================
// EXTRAI DADOS USANDO CONFIGURAÇÃO DO GATEWAY
// ============================================

/**
 * Extrai valor usando path como "data.data[0].status" ou "data.data.status"
 */
function extractValue($data, $path) {
    if (!$path || !$data) return null;
    
    // Se já é o valor direto
    if (!is_string($path)) return $path;
    
    $parts = explode('.', $path);
    $current = $data;
    
    foreach ($parts as $part) {
        if ($current === null) return null;
        
        // Verifica se é acesso a array tipo "data[0]"
        if (preg_match('/^(.+)\[(\d+)\]$/', $part, $matches)) {
            $key = $matches[1];
            $index = (int)$matches[2];
            
            if (!isset($current[$key]) || !is_array($current[$key])) {
                return null;
            }
            $current = $current[$key][$index] ?? null;
        } else {
            if (!is_array($current) || !isset($current[$part])) {
                return null;
            }
            $current = $current[$part];
        }
    }
    
    return $current;
}

$txId = null;
$status = null;
$amount = null;
$customerData = [];

// Usa campos configurados no gateways.json
$statusField = $gatewayConfig['status_field'] ?? 'status';
$txidField = $gatewayConfig['txid_field'] ?? 'id';
$valorEmReais = $gatewayConfig['valor_em_reais'] ?? false;

switch ($detectedGateway) {
    
    // ========== ALLOWPAY ==========
    case 'allowpay':
        // AllowPay: dados geralmente no root ou em data
        $txId = $data['id'] ?? $data['objectId'] ?? extractValue($data, $txidField);
        $status = $data['status'] ?? extractValue($data, $statusField);
        $amount = $data['amount'] ?? $data['value'] ?? null;
        
        // Se dados estão em "data"
        if (!$txId && isset($data['data'])) {
            $txId = $data['data']['id'] ?? null;
            $status = $data['data']['status'] ?? $status;
            $amount = $data['data']['amount'] ?? $data['data']['value'] ?? $amount;
        }
        
        // Customer
        $customer = $data['customer'] ?? $data['data']['customer'] ?? $data['payer'] ?? [];
        if ($customer) {
            $customerData = [
                'name' => $customer['name'] ?? null,
                'email' => $customer['email'] ?? null,
                'phone' => $customer['phone'] ?? $customer['cellphone'] ?? null,
                'document' => $customer['document'] ?? $customer['cpf'] ?? $customer['tax_id'] ?? null
            ];
        }
        break;
    
    // ========== UMBRELLA ==========
    case 'umbrella':
        // Umbrella: data.data[0].xxx
        $txData = $data['data']['data'][0] ?? [];
        
        $txId = $txData['id'] ?? $data['objectId'] ?? extractValue($data, $txidField);
        $status = $txData['status'] ?? extractValue($data, $statusField);
        $amount = $txData['amount'] ?? $txData['value'] ?? null;
        
        // Customer
        $customer = $txData['customer'] ?? $txData['payer'] ?? [];
        if ($customer) {
            $customerData = [
                'name' => $customer['name'] ?? null,
                'email' => $customer['email'] ?? null,
                'phone' => $customer['phone'] ?? $customer['phoneNumber'] ?? null,
                'document' => $customer['document'] ?? $customer['cpf'] ?? $customer['taxId'] ?? null
            ];
        }
        break;
    
    // ========== TECHBYNET ==========
    case 'techbynet':
        // TechByNet: data.data.xxx (objeto, não array)
        $txData = $data['data']['data'] ?? $data['data'] ?? $data;
        
        // Se data.data é um array com índice 0, pega o primeiro
        if (isset($txData[0])) {
            $txData = $txData[0];
        }
        
        $txId = $txData['id'] ?? $data['transaction_id'] ?? $data['id'] ?? extractValue($data, $txidField);
        $status = $txData['status'] ?? $data['status'] ?? extractValue($data, $statusField);
        $amount = $txData['amount'] ?? $txData['value'] ?? null;
        
        // Customer
        $customer = $txData['customer'] ?? $txData['payer'] ?? $data['customer'] ?? [];
        if ($customer) {
            $customerData = [
                'name' => $customer['name'] ?? null,
                'email' => $customer['email'] ?? null,
                'phone' => $customer['phone'] ?? null,
                'document' => $customer['document'] ?? $customer['cpf'] ?? null
            ];
        }
        break;
    
    // ========== GENESYS ==========
    case 'genesys':
        // Genesys: dados no root, total_amount em REAIS
        $txId = $data['id'] ?? $data['external_id'] ?? extractValue($data, $txidField);
        $status = $data['status'] ?? extractValue($data, $statusField);
        
        // Genesys envia em REAIS (config: valor_em_reais: true)
        if (isset($data['total_amount'])) {
            $amount = intval(floatval($data['total_amount']) * 100);
            $valorEmReais = true; // Já convertido
        } elseif (isset($data['amount'])) {
            // Se usa amount, pode ser centavos ou reais
            $amt = $data['amount'];
            if ($valorEmReais || $amt < 1000) {
                // Provavelmente em reais
                $amount = intval(floatval($amt) * 100);
            } else {
                $amount = intval($amt);
            }
        }
        
        // Customer
        $customer = $data['customer'] ?? [];
        if ($customer) {
            $customerData = [
                'name' => $customer['name'] ?? null,
                'email' => $customer['email'] ?? null,
                'phone' => $customer['phone'] ?? null,
                'document' => $customer['document'] ?? $customer['cpf'] ?? null
            ];
        }
        break;
    
    // ========== GENÉRICO ==========
    default:
        // Tenta extrair usando campos configurados
        $txId = extractValue($data, $txidField) ?? $data['id'] ?? $data['objectId'] ?? $data['transaction_id'];
        $status = extractValue($data, $statusField) ?? $data['status'];
        $amount = $data['amount'] ?? $data['value'] ?? $data['total_amount'];
        
        // Se total_amount, converte de reais
        if (isset($data['total_amount']) && $valorEmReais) {
            $amount = intval(floatval($data['total_amount']) * 100);
        } elseif ($amount) {
            $amount = intval($amount);
        }
        
        $customer = $data['customer'] ?? [];
        if ($customer) {
            $customerData = [
                'name' => $customer['name'] ?? null,
                'email' => $customer['email'] ?? null,
                'phone' => $customer['phone'] ?? null,
                'document' => $customer['document'] ?? null
            ];
        }
}

// Garante que amount é inteiro (centavos)
if ($amount !== null) {
    $amount = intval($amount);
}

logCheckout('webhook.log', '📦 Dados extraídos', [
    'txId' => $txId,
    'status' => $status,
    'amount_centavos' => $amount,
    'amount_reais' => $amount ? 'R$ ' . number_format($amount / 100, 2, ',', '.') : null,
    'has_customer' => !empty(array_filter($customerData))
]);

// ============================================
// VALIDAÇÃO
// ============================================
if (!$txId) {
    logCheckout('webhook.log', '❌ ERRO: txId não encontrado');
    http_response_code(400);
    echo json_encode(['error' => 'Transaction ID não encontrado', 'gateway' => $detectedGateway]);
    exit;
}

// ============================================
// VERIFICA SE É PAGAMENTO CONFIRMADO
// ============================================
$isPaid = isPaidStatus($status);

logCheckout('webhook.log', 'Status', [
    'original' => $status,
    'mapped' => mapStatus($status),
    'is_paid' => $isPaid
]);

// ============================================
// PROCESSA PAGAMENTO CONFIRMADO
// ============================================
if ($isPaid) {
    logCheckout('webhook.log', '💰 ========== PAGAMENTO CONFIRMADO ==========', [
        'txId' => $txId,
        'gateway' => $detectedGateway,
        'valor' => $amount ? 'R$ ' . number_format($amount / 100, 2, ',', '.') : 'não informado'
    ]);
    
    // Carrega dados UTM salvos
    $utmFile = UTM_STORAGE_DIR . $txId . '.json';
    $utmData = [];
    
    if (file_exists($utmFile)) {
        $utmData = json_decode(file_get_contents($utmFile), true) ?: [];
        logCheckout('webhook.log', '📂 UTM carregado', [
            'has_fbc' => !empty($utmData['fbc']),
            'has_fbp' => !empty($utmData['fbp']),
            'has_email' => !empty($utmData['customer']['email'] ?? $utmData['email'])
        ]);
    } else {
        logCheckout('webhook.log', '⚠️ UTM não encontrado, usando dados do webhook');
        if (!empty($customerData)) {
            $utmData['customer'] = $customerData;
        }
    }
    
    // Calcula valor
    $valorCentavos = $amount ?: ($utmData['valor_centavos'] ?? 0);
    $valorReais = $valorCentavos / 100;
    
    if ($valorReais <= 0) {
        $valorReais = defined('VALOR_PADRAO_REAIS') ? VALOR_PADRAO_REAIS : 59.90;
        $valorCentavos = intval($valorReais * 100);
        logCheckout('webhook.log', '⚠️ Usando valor padrão: R$ ' . number_format($valorReais, 2, ',', '.'));
    }
    
    // ========== FACEBOOK CAPI - PURCHASE ==========
    $fbResult = ['success' => false];
    
    if (class_exists('FacebookCAPI')) {
        try {
            $fb = new FacebookCAPI();
            
            if (empty($utmData['customer']) && !empty($customerData)) {
                $utmData['customer'] = $customerData;
            }
            $utmData['valor_centavos'] = $valorCentavos;
            
            $fbResult = $fb->sendPurchase($txId, $valorReais, $utmData);
            
            logCheckout('webhook.log', '📘 Facebook CAPI', [
                'success' => $fbResult['success'] ?? false,
                'skipped' => $fbResult['skipped'] ?? false,
                'event_id' => $fbResult['event_id'] ?? null,
                'events_received' => $fbResult['events_received'] ?? 0
            ]);
        } catch (Exception $e) {
            logCheckout('webhook.log', '❌ FB CAPI erro: ' . $e->getMessage());
        }
    }
    
    // ========== UTMIFY ==========
    $utmifyResult = ['success' => false];
    
    if (class_exists('Utmify')) {
        try {
            $utmify = new Utmify();
            
            if ($utmify->isConfigured()) {
                $customer = [
                    'name' => $utmData['customer']['name'] ?? $utmData['nome'] ?? $customerData['name'] ?? '',
                    'email' => $utmData['customer']['email'] ?? $utmData['email'] ?? $customerData['email'] ?? '',
                    'phone' => $utmData['customer']['phone'] ?? $utmData['telefone'] ?? $customerData['phone'] ?? '',
                    'document' => $utmData['customer']['document'] ?? $utmData['cpf'] ?? $customerData['document'] ?? '',
                    'ip' => $utmData['customer_ip'] ?? ''
                ];
                
                $product = [
                    'id' => $utmData['produto_id'] ?? PRODUTO_ID ?? 'produto-001',
                    'name' => $utmData['produto'] ?? PRODUTO_NOME ?? 'Produto',
                    'quantity' => $utmData['cotas'] ?? 1
                ];
                
                $utms = [
                    'utm_source' => $utmData['utm_source'] ?? '',
                    'utm_medium' => $utmData['utm_medium'] ?? '',
                    'utm_campaign' => $utmData['utm_campaign'] ?? '',
                    'utm_content' => $utmData['utm_content'] ?? '',
                    'utm_term' => $utmData['utm_term'] ?? '',
                    'sck' => $utmData['sck'] ?? '',
                    'src' => $utmData['src'] ?? ''
                ];
                
                $utmifyResult = $utmify->sendPaid($txId, $valorCentavos, $customer, $product, $utms);
                
                logCheckout('webhook.log', '📊 UTMify', [
                    'success' => $utmifyResult['success'] ?? false
                ]);
            }
        } catch (Exception $e) {
            logCheckout('webhook.log', '❌ UTMify erro: ' . $e->getMessage());
        }
    }
    
    // Salva resultado
    $utmData['paid'] = true;
    $utmData['paid_at'] = date('Y-m-d H:i:s');
    $utmData['webhook_gateway'] = $detectedGateway;
    $utmData['webhook_status'] = $status;
    $utmData['webhook_amount'] = $amount;
    $utmData['fb_success'] = $fbResult['success'] ?? false;
    
    @file_put_contents($utmFile, json_encode($utmData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    logCheckout('webhook.log', '✅ CONCLUÍDO', [
        'txId' => $txId,
        'valor' => 'R$ ' . number_format($valorReais, 2, ',', '.'),
        'fb' => $fbResult['success'] ?? false,
        'utmify' => $utmifyResult['success'] ?? false
    ]);
    
} else {
    logCheckout('webhook.log', '⏳ Status pendente: ' . $status);
}

// Resposta
http_response_code(200);
echo json_encode([
    'success' => true,
    'txId' => $txId,
    'status' => $status,
    'isPaid' => $isPaid,
    'gateway' => $detectedGateway
]);