/**
 * ============================================
 * SMART TRACKING v6 - QUALIDADE MÁXIMA
 * ============================================
 * 
 * CORREÇÕES:
 * - Pixel ID correto: 2357252418054828
 * - Valor padrão: R$ 59,90
 * - Purchase BLOQUEADO no frontend
 * - Anti-duplicação robusta
 * - Coleta completa de dados
 * 
 * INSTALAÇÃO:
 * <script>
 *   !function(f,b,e,v,n,t,s){...}
 *   fbq('init', '2357252418054828');
 * </script>
 * <script src="/api/smart-tracking.js"></script>
 * <script>fbq('track', 'PageView');</script>
 */

(function() {
    'use strict';
    
    // ============================================
    // CONFIGURAÇÃO
    // ============================================
    var CONFIG = {
        pixelId: '2357252418054828',
        trackingEndpoint: '/api/tracking.php',
        defaultValue: 59.90,
        defaultCurrency: 'BRL',
        defaultContentName: 'Tarifa BrasiCred',
        defaultContentId: 'tarifa-001',
        debug: false,
        // IMPORTANTE: Purchase é enviado APENAS pelo webhook
        blockPurchaseFromFrontend: true
    };
    
    var sentEventIds = {};
    
    // ============================================
    // UTILITÁRIOS
    // ============================================
    function log(msg, data) {
        if (CONFIG.debug) console.log('[SmartTracking]', msg, data || '');
    }
    
    function warn(msg, data) {
        console.warn('[SmartTracking]', msg, data || '');
    }
    
    function generateEventId(eventName) {
        return eventName.toLowerCase() + '_' + Math.random().toString(36).substr(2, 9) + '_' + Date.now();
    }
    
    function getFbCookies() {
        var cookies = {};
        document.cookie.split(';').forEach(function(c) {
            var parts = c.trim().split('=');
            if (parts.length === 2) cookies[parts[0]] = parts[1];
        });
        
        var fbc = cookies._fbc || null;
        var fbp = cookies._fbp || null;
        
        // Cria fbc a partir do fbclid
        if (!fbc) {
            var params = new URLSearchParams(window.location.search);
            var fbclid = params.get('fbclid');
            if (fbclid) {
                fbc = 'fb.1.' + Date.now() + '.' + fbclid;
                try { document.cookie = '_fbc=' + fbc + '; max-age=7776000; path=/; SameSite=Lax'; } catch(e) {}
            }
        }
        
        // Cria fbp se não existir
        if (!fbp) {
            fbp = 'fb.1.' + Date.now() + '.' + Math.floor(Math.random() * 10000000000);
            try { document.cookie = '_fbp=' + fbp + '; max-age=7776000; path=/; SameSite=Lax'; } catch(e) {}
        }
        
        return { fbc: fbc, fbp: fbp };
    }
    
    function getUtms() {
        var params = new URLSearchParams(window.location.search);
        var utms = {};
        ['utm_source', 'utm_medium', 'utm_campaign', 'utm_content', 'utm_term', 'sck', 'src', 'fbclid'].forEach(function(key) {
            utms[key] = params.get(key) || sessionStorage.getItem(key) || '';
            if (utms[key]) try { sessionStorage.setItem(key, utms[key]); } catch(e) {}
        });
        return utms;
    }
    
    function getClientData() {
        try {
            var stored = JSON.parse(localStorage.getItem('cliente') || '{}');
            var params = new URLSearchParams(window.location.search);
            
            return {
                email: stored.email || params.get('email') || null,
                phone: stored.celular || stored.telefone || params.get('celular') || null,
                name: stored.nome || params.get('nome') || null,
                document: stored.cpf || params.get('cpf') || null
            };
        } catch(e) { return {}; }
    }
    
    function getProductValue() {
        try {
            // localStorage (centavos)
            var valorCentavos = localStorage.getItem('valor');
            if (valorCentavos && parseInt(valorCentavos) > 0) {
                return parseInt(valorCentavos) / 100;
            }
            
            // Elemento na página
            var el = document.querySelector('#txtValor, .precoNew, [data-valor]');
            if (el) {
                var texto = el.textContent || el.getAttribute('data-valor') || '';
                var numero = texto.replace(/[^\d,]/g, '').replace(',', '.');
                if (numero && parseFloat(numero) > 0) return parseFloat(numero);
            }
            
            return CONFIG.defaultValue;
        } catch(e) { return CONFIG.defaultValue; }
    }
    
    function getQuantity() {
        try {
            var cotas = localStorage.getItem('cotas');
            if (cotas && parseInt(cotas) > 0) return parseInt(cotas);
            return 1;
        } catch(e) { return 1; }
    }
    
    // ============================================
    // ANTI-DUPLICAÇÃO
    // ============================================
    function isEventDuplicate(eventId) {
        if (!eventId) return false;
        if (sentEventIds[eventId]) return true;
        try { if (sessionStorage.getItem('st_' + eventId)) return true; } catch(e) {}
        return false;
    }
    
    function markEventSent(eventId) {
        if (!eventId) return;
        sentEventIds[eventId] = Date.now();
        try { sessionStorage.setItem('st_' + eventId, '1'); } catch(e) {}
        
        // Limpa eventos antigos
        var now = Date.now();
        Object.keys(sentEventIds).forEach(function(id) {
            if (now - sentEventIds[id] > 600000) delete sentEventIds[id];
        });
    }
    
    // ============================================
    // ENVIO PARA CAPI
    // ============================================
    function sendToCAPI(eventName, eventId, customData) {
        // BLOQUEIO: Purchase não deve ser enviado pelo frontend
        if (CONFIG.blockPurchaseFromFrontend && eventName === 'Purchase') {
            warn('⛔ Purchase BLOQUEADO - deve ser enviado apenas pelo webhook');
            return Promise.resolve(null);
        }
        
        if (isEventDuplicate(eventId)) {
            log('⚠️ Duplicado:', eventId);
            return Promise.resolve(eventId);
        }
        
        markEventSent(eventId);
        customData = customData || {};
        
        var fbCookies = getFbCookies();
        var utms = getUtms();
        var clientData = getClientData();
        
        var value = customData.value;
        if (!value || value === 0) value = getProductValue();
        
        var payload = {
            event: eventName,
            event_id: eventId,
            source_url: window.location.href,
            user_agent: navigator.userAgent,
            fbc: fbCookies.fbc,
            fbp: fbCookies.fbp,
            email: clientData.email,
            phone: clientData.phone,
            name: clientData.name,
            document: clientData.document,
            utm_source: utms.utm_source,
            utm_medium: utms.utm_medium,
            utm_campaign: utms.utm_campaign,
            utm_content: utms.utm_content,
            utm_term: utms.utm_term,
            value: parseFloat(value) || CONFIG.defaultValue,
            currency: customData.currency || CONFIG.defaultCurrency,
            content_name: customData.content_name || CONFIG.defaultContentName,
            content_id: (customData.content_ids && customData.content_ids[0]) || customData.content_id || CONFIG.defaultContentId,
            num_items: parseInt(customData.num_items || customData.quantity || getQuantity()) || 1
        };
        
        // Remove campos vazios
        Object.keys(payload).forEach(function(key) {
            if (payload[key] === null || payload[key] === undefined || payload[key] === '') {
                delete payload[key];
            }
        });
        
        log('📤 CAPI:', { event: eventName, event_id: eventId, value: payload.value });
        
        return new Promise(function(resolve) {
            fetch(CONFIG.trackingEndpoint, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload),
                keepalive: true
            }).then(function(response) {
                if (response.ok) log('✅ CAPI OK:', eventName);
                resolve(eventId);
            }).catch(function(err) {
                warn('❌ CAPI erro:', err.message);
                resolve(eventId);
            });
        });
    }
    
    // ============================================
    // INTERCEPTA O PIXEL DO FACEBOOK
    // ============================================
    var originalFbq = window.fbq;
    
    function newFbq() {
        var args = Array.prototype.slice.call(arguments);
        var action = args[0];
        var eventName = args[1];
        var eventData = args[2] || {};
        var options = args[3] || {};
        
        if ((action === 'track' || action === 'trackCustom') && eventName) {
            
            // BLOQUEIO: Purchase
            if (CONFIG.blockPurchaseFromFrontend && eventName === 'Purchase') {
                warn('⛔ fbq("track", "Purchase") BLOQUEADO');
                warn('   O Purchase é enviado automaticamente pelo webhook');
                return;
            }
            
            var eventId = options.eventID || generateEventId(eventName);
            if (!options.eventID) {
                options.eventID = eventId;
                args[3] = options;
            }
            
            // Adiciona value/currency se necessário
            var eventsWithValue = ['ViewContent', 'AddToCart', 'InitiateCheckout', 'AddPaymentInfo'];
            if (eventsWithValue.indexOf(eventName) !== -1) {
                if (!eventData.value || eventData.value === 0) eventData.value = getProductValue();
                if (!eventData.currency) eventData.currency = CONFIG.defaultCurrency;
                if (!eventData.num_items) eventData.num_items = getQuantity();
                if (!eventData.content_name) eventData.content_name = CONFIG.defaultContentName;
                if (!eventData.content_ids) eventData.content_ids = [CONFIG.defaultContentId];
                args[2] = eventData;
            }
            
            log('📡 Pixel:', { event: eventName, event_id: eventId, value: eventData.value });
            
            // Envia para CAPI
            sendToCAPI(eventName, eventId, eventData);
        }
        
        // Chama função original
        if (originalFbq) return originalFbq.apply(window, args);
    }
    
    // Copia propriedades do fbq original
    if (originalFbq) {
        for (var key in originalFbq) {
            if (originalFbq.hasOwnProperty(key)) newFbq[key] = originalFbq[key];
        }
        if (originalFbq.queue) newFbq.queue = originalFbq.queue;
        if (originalFbq.loaded !== undefined) newFbq.loaded = originalFbq.loaded;
        if (originalFbq.version) newFbq.version = originalFbq.version;
        if (originalFbq.push) newFbq.push = newFbq;
    }
    
    window.fbq = newFbq;
    if (window._fbq === originalFbq) window._fbq = newFbq;
    
    // ============================================
    // API PÚBLICA
    // ============================================
    window.SmartTracking = {
        version: '6.0.0',
        
        sendEvent: function(eventName, data) {
            if (CONFIG.blockPurchaseFromFrontend && eventName === 'Purchase') {
                warn('⛔ SmartTracking.sendEvent("Purchase") BLOQUEADO');
                return Promise.resolve(null);
            }
            
            data = data || {};
            var eventId = generateEventId(eventName);
            
            if (!data.value || data.value === 0) data.value = getProductValue();
            if (!data.currency) data.currency = CONFIG.defaultCurrency;
            
            if (typeof window.fbq !== 'undefined') {
                window.fbq('track', eventName, data, { eventID: eventId });
            } else {
                sendToCAPI(eventName, eventId, data);
            }
            
            return Promise.resolve(eventId);
        },
        
        track: function(eventName, data) { return this.sendEvent(eventName, data); },
        
        getFbCookies: getFbCookies,
        getUtms: getUtms,
        getClientData: getClientData,
        getProductValue: getProductValue,
        
        enableDebug: function() { CONFIG.debug = true; console.log('[SmartTracking] Debug ON'); },
        disableDebug: function() { CONFIG.debug = false; },
        
        init: function() {
            getFbCookies();
            getUtms();
            log('✅ Inicializado', { pixelId: CONFIG.pixelId, defaultValue: CONFIG.defaultValue });
        }
    };
    
    SmartTracking.init();
    log('✅ Smart Tracking v6 carregado');
    
})();