<?php
/**
 * ============================================
 * PIX API - GERAÇÃO DE PAGAMENTO
 * ============================================
 * Tratamento robusto de dados para todos os gateways
 * Suporta: Umbrella, TechByNet, AllowPay, Genesys
 * 
 * CORREÇÃO: Captura fbc/fbp de múltiplas fontes
 */

require_once __DIR__ . '/config.php';
@include_once __DIR__ . '/facebook_capi.php';
@include_once __DIR__ . '/utmify.php';

setCorsHeaders();

// ============================================
// FUNÇÕES DE TRATAMENTO DE DADOS
// ============================================

/**
 * Limpa e formata CPF (remove tudo que não é número)
 */
function limparCPF($cpf) {
    $cpf = preg_replace('/\D/', '', $cpf ?? '');
    
    // Se veio vazio ou muito curto, retorna CPF padrão
    if (strlen($cpf) < 11) {
        return '00000000000';
    }
    
    // Se veio maior que 11, trunca (pode ser CNPJ ou erro)
    if (strlen($cpf) > 11) {
        $cpf = substr($cpf, 0, 11);
    }
    
    return $cpf;
}

/**
 * Limpa e formata telefone
 */
function limparTelefone($telefone) {
    $telefone = preg_replace('/\D/', '', $telefone ?? '');
    
    // Se veio vazio ou muito curto
    if (strlen($telefone) < 10) {
        return '11999999999';
    }
    
    // Remove 0 inicial se houver (0xx)
    if (strlen($telefone) > 11 && $telefone[0] === '0') {
        $telefone = substr($telefone, 1);
    }
    
    // Limita a 11 dígitos
    if (strlen($telefone) > 11) {
        $telefone = substr($telefone, 0, 11);
    }
    
    // Se tem 10 dígitos (fixo), adiciona 9 para virar celular
    if (strlen($telefone) === 10) {
        $ddd = substr($telefone, 0, 2);
        $numero = substr($telefone, 2);
        $telefone = $ddd . '9' . $numero;
    }
    
    return $telefone;
}

/**
 * Limpa e valida email - GERA ALEATÓRIO SE VAZIO
 */
function limparEmail($email) {
    $email = trim($email ?? '');
    
    // Decodifica se vier URL encoded
    $email = urldecode($email);
    
    // Se vazio ou inválido, GERA EMAIL ALEATÓRIO
    if (empty($email) || strpos($email, '@') === false || strpos($email, '.') === false) {
        // Gera email único baseado em timestamp + random
        $random = strtolower(bin2hex(random_bytes(4))); // 8 caracteres hex
        $timestamp = time();
        return "cliente{$timestamp}{$random}@pagamento.com";
    }
    
    // Remove espaços e converte para minúsculo
    $email = strtolower(str_replace(' ', '', $email));
    
    return $email;
}

/**
 * Limpa e formata nome
 */
function limparNome($nome) {
    $nome = trim($nome ?? '');
    
    // Decodifica URL encoding
    $nome = urldecode($nome);
    
    // Substitui + por espaço (URL encoding alternativo)
    $nome = str_replace('+', ' ', $nome);
    
    // Remove caracteres especiais mantendo letras, espaços, hífens e apóstrofos
    $nome = preg_replace('/[^\p{L}\s\-\'\.]/u', '', $nome);
    
    // Remove espaços múltiplos
    $nome = preg_replace('/\s+/', ' ', $nome);
    
    // Trim novamente
    $nome = trim($nome);
    
    // Se ficou muito curto, usa padrão
    if (strlen($nome) < 2) {
        return 'Cliente';
    }
    
    // Limita tamanho (alguns gateways têm limite)
    if (strlen($nome) > 100) {
        $nome = substr($nome, 0, 100);
    }
    
    return $nome;
}

/**
 * Converte valor para centavos
 */
function valorParaCentavos($valor) {
    // Se já é inteiro maior que 100, provavelmente já está em centavos
    if (is_int($valor) && $valor > 100) {
        return $valor;
    }
    
    // Converte para string para processar
    $valorStr = strval($valor ?? '0');
    
    // Limpa o valor
    $valorStr = str_replace(['R$', ' '], '', $valorStr);
    
    // Detecta formato brasileiro (1.234,56) vs americano (1,234.56)
    // Se tem vírgula depois do ponto, é brasileiro
    $posVirgula = strrpos($valorStr, ',');
    $posPonto = strrpos($valorStr, '.');
    
    if ($posVirgula !== false && ($posPonto === false || $posVirgula > $posPonto)) {
        // Formato brasileiro: 1.234,56
        $valorStr = str_replace('.', '', $valorStr); // Remove pontos de milhar
        $valorStr = str_replace(',', '.', $valorStr); // Troca vírgula por ponto
    } else {
        // Formato americano ou só número: remove vírgulas de milhar
        $valorStr = str_replace(',', '', $valorStr);
    }
    
    $valorFloat = floatval($valorStr);
    
    // Se parece estar em reais (menor que 10000 e tem decimais ou é pequeno)
    if ($valorFloat < 10000 && ($valorFloat != intval($valorFloat) || $valorFloat < 500)) {
        return intval(round($valorFloat * 100));
    }
    
    // Já está em centavos
    return intval($valorFloat);
}

// ============================================
// RECEBER E PROCESSAR DADOS
// ============================================

// Tenta ler JSON do body
$rawInput = file_get_contents('php://input');
$input = json_decode($rawInput, true);

// Se não veio JSON válido, tenta form-urlencoded
if (!$input && !empty($rawInput)) {
    parse_str($rawInput, $input);
}

// Se não veio no body, tenta POST
if (!$input && !empty($_POST)) {
    $input = $_POST;
}

// Se não veio POST, tenta GET (para debug)
if (!$input && !empty($_GET)) {
    $input = $_GET;
}

logCheckout('pix.log', 'Dados recebidos', [
    'method' => $_SERVER['REQUEST_METHOD'],
    'content_type' => $_SERVER['CONTENT_TYPE'] ?? 'não definido',
    'input_keys' => $input ? array_keys($input) : 'vazio',
    'raw_preview' => substr($rawInput, 0, 300)
]);

if (!$input) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Nenhum dado recebido']);
    exit;
}

// ============================================
// EXTRAIR DADOS (SUPORTA MÚLTIPLOS FORMATOS)
// ============================================

// Verifica se veio no formato antigo (com objeto cliente)
$cliente = $input['cliente'] ?? null;

if ($cliente && is_array($cliente)) {
    // Formato antigo: { cliente: {...}, tracking: {...}, valor: ... }
    $nome = $cliente['nome'] ?? $cliente['name'] ?? '';
    $cpf = $cliente['cpf'] ?? $cliente['document'] ?? $cliente['doc_num'] ?? '';
    $email = $cliente['email'] ?? '';
    $telefone = $cliente['celular'] ?? $cliente['telefone'] ?? $cliente['phone'] ?? '';
    $cotas = $cliente['cotas'] ?? 1;
    $valorCentavos = $input['valor'] ?? 0;
    $tracking = $input['tracking'] ?? [];
} else {
    // Formato novo: campos diretos
    $nome = $input['nome'] ?? $input['name'] ?? '';
    $cpf = $input['cpf'] ?? $input['document'] ?? $input['doc_num'] ?? '';
    $email = $input['email'] ?? '';
    $telefone = $input['telefone'] ?? $input['celular'] ?? $input['phone'] ?? $input['telephone'] ?? '';
    $cotas = $input['cotas'] ?? $input['quantidade'] ?? 1;
    $valorCentavos = $input['valor'] ?? $input['amount'] ?? $input['value'] ?? 0;
    $tracking = $input['tracking'] ?? [];
    
    // Se tracking não veio como objeto, monta dos campos
    if (empty($tracking)) {
        $tracking = [
            'utm_source' => $input['utm_source'] ?? null,
            'utm_medium' => $input['utm_medium'] ?? null,
            'utm_campaign' => $input['utm_campaign'] ?? null,
            'utm_content' => $input['utm_content'] ?? null,
            'utm_term' => $input['utm_term'] ?? null,
            'sck' => $input['sck'] ?? null,
            'src' => $input['src'] ?? null
        ];
    }
}

// ============================================
// CAPTURA FBC/FBP DE MÚLTIPLAS FONTES
// ============================================
// Prioridade: input direto > tracking > cookies
$fbc = $input['fbc'] ?? $tracking['fbc'] ?? $_COOKIE['_fbc'] ?? null;
$fbp = $input['fbp'] ?? $tracking['fbp'] ?? $_COOKIE['_fbp'] ?? null;
$fbclid = $input['fbclid'] ?? $tracking['fbclid'] ?? null;

// Se tem fbclid mas não tem fbc, gera fbc
if (!$fbc && $fbclid) {
    $fbc = 'fb.1.' . time() . '.' . $fbclid;
}

// Se não tem fbp, gera um novo
if (!$fbp) {
    $fbp = 'fb.1.' . round(microtime(true) * 1000) . '.' . mt_rand(1000000000, 9999999999);
}

// ============================================
// LIMPAR E VALIDAR TODOS OS DADOS
// ============================================
$nome = limparNome($nome);
$cpf = limparCPF($cpf);
$email = limparEmail($email);
$telefone = limparTelefone($telefone);
$valorCentavos = valorParaCentavos($valorCentavos);
$cotas = max(1, intval($cotas));

// Validação final
if ($valorCentavos <= 0) {
    logCheckout('pix.log', 'Valor inválido', ['valor_original' => $input['valor'] ?? 'não informado']);
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Valor inválido: ' . ($input['valor'] ?? 'não informado')]);
    exit;
}

$valorReais = $valorCentavos / 100;

logCheckout('pix.log', 'Dados processados', [
    'nome' => $nome,
    'cpf' => substr($cpf, 0, 3) . '***' . substr($cpf, -2),
    'email' => $email,
    'telefone' => substr($telefone, 0, 2) . '****' . substr($telefone, -4),
    'valor_centavos' => $valorCentavos,
    'valor_reais' => $valorReais,
    'cotas' => $cotas,
    'has_fbc' => !empty($fbc),
    'has_fbp' => !empty($fbp)
]);

// ============================================
// PRODUTO DINÂMICO
// ============================================
$produto = $input['produto'] ?? [];
$produtoNome = $produto['nome'] ?? $input['produto_nome'] ?? (defined('PRODUTO_NOME') ? PRODUTO_NOME : 'Taxa de Verificacao');
$produtoId = $produto['id'] ?? $input['produto_id'] ?? (defined('PRODUTO_ID') ? PRODUTO_ID : 'taxa-001');
$produtoDescricao = $produto['descricao'] ?? $input['produto_descricao'] ?? null;

try {
    // ============================================
    // OBTER CONFIGURAÇÃO DO GATEWAY
    // ============================================
    $gateway = getGatewayConfig();
    $gatewayName = getGatewayName();
    
    if (!$gateway) {
        throw new Exception("Gateway '$gatewayName' não configurado");
    }
    
    logCheckout('pix.log', "Usando gateway: {$gateway['name']}", ['gateway' => $gatewayName]);
    
    // ============================================
    // MONTAR PAYLOAD CONFORME O GATEWAY
    // ============================================
    $txRef = generateTxRef();
    $webhookUrl = defined('WEBHOOK_URL') ? WEBHOOK_URL : ('https://' . ($_SERVER['HTTP_HOST'] ?? 'localhost') . '/api/webhook.php');
    $ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
    if (strpos($ip, ',') !== false) {
        $ip = trim(explode(',', $ip)[0]);
    }
    $descricaoItem = $produtoDescricao ?: $produtoNome;
    
    $payload = [];
    
    // ========== ALLOWPAY ==========
    if ($gatewayName === 'allowpay') {
        $payload = [
            'amount' => $valorCentavos,
            'description' => $descricaoItem,
            'paymentMethod' => 'PIX',
            'postbackUrl' => $webhookUrl,
            'ip' => $ip,
            'customer' => [
                'name' => $nome,
                'email' => $email,
                'phone' => $telefone,
                'document' => [
                    'type' => 'CPF',
                    'number' => $cpf
                ]
            ],
            'shipping' => [
                'address' => [
                    'street' => 'Avenida Paulista',
                    'streetNumber' => '1000',
                    'complement' => '',
                    'neighborhood' => 'Bela Vista',
                    'city' => 'Sao Paulo',
                    'state' => 'SP',
                    'zipCode' => '01310100'
                ]
            ],
            'pix' => [
                'expiresInDays' => 1
            ],
            'items' => [[
                'title' => $descricaoItem,
                'unitPrice' => $valorCentavos,
                'quantity' => 1,
                'externalRef' => $txRef
            ]]
        ];
    }
    // ========== GENESYS ==========
    elseif ($gatewayName === 'genesys') {
        // Genesys usa valor em REAIS (não centavos)
        $payload = [
            'external_id' => $txRef,
            'total_amount' => $valorReais,
            'payment_method' => 'PIX',
            'webhook_url' => $webhookUrl,
            'ip' => $ip,
            'items' => [[
                'id' => $txRef,
                'title' => $produtoNome,
                'description' => $descricaoItem,
                'price' => $valorReais,
                'quantity' => 1,
                'is_physical' => false
            ]],
            'customer' => [
                'name' => $nome,
                'email' => $email,
                'phone' => $telefone,
                'document_type' => 'CPF',
                'document' => $cpf,
                'utm_source' => $tracking['utm_source'] ?? '',
                'utm_medium' => $tracking['utm_medium'] ?? '',
                'utm_campaign' => $tracking['utm_campaign'] ?? '',
                'utm_content' => $tracking['utm_content'] ?? '',
                'utm_term' => $tracking['utm_term'] ?? ''
            ]
        ];
    }
    // ========== UMBRELLA / TECHBYNET / PADRÃO ==========
    else {
        $payload = [
            'amount' => $valorCentavos,
            'currency' => 'BRL',
            'paymentMethod' => 'PIX',
            'installments' => 1,
            'customer' => [
                'name' => $nome,
                'email' => $email,
                'document' => [
                    'number' => $cpf,
                    'type' => 'CPF'
                ],
                'phone' => $telefone,
                'externalRef' => $txRef,
                'address' => [
                    'street' => 'Avenida Paulista',
                    'streetNumber' => '1000',
                    'complement' => '',
                    'zipCode' => '01310100',
                    'neighborhood' => 'Bela Vista',
                    'city' => 'Sao Paulo',
                    'state' => 'SP',
                    'country' => 'BR'
                ]
            ],
            'items' => [[
                'title' => $descricaoItem,
                'unitPrice' => $valorCentavos,
                'quantity' => 1,
                'tangible' => false,
                'externalRef' => 'item-' . $txRef
            ]],
            'pix' => [
                'expiresInDays' => 1
            ],
            'shipping' => [
                'fee' => 0,
                'address' => [
                    'street' => 'Avenida Paulista',
                    'streetNumber' => '1000',
                    'complement' => '',
                    'zipCode' => '01310100',
                    'neighborhood' => 'Bela Vista',
                    'city' => 'Sao Paulo',
                    'state' => 'SP',
                    'country' => 'BR'
                ]
            ],
            'postbackUrl' => $webhookUrl,
            'metadata' => json_encode(['pedido' => $txRef, 'cotas' => $cotas, 'produto' => $produtoNome]),
            'traceable' => true,
            'ip' => $ip
        ];
    }
    
    // ============================================
    // MONTAR URL E HEADERS
    // ============================================
    $endpoint = $gateway['endpoints']['pix_create'] ?? '';
    $url = rtrim($gateway['api_url'], '/');
    if ($endpoint) {
        $url .= $endpoint;
    }
    
    // Headers de autenticação
    $authType = $gateway['auth_type'] ?? 'apikey';
    $headers = [
        'Content-Type: application/json',
        'Accept: application/json',
        'User-Agent: ' . ($gateway['user_agent'] ?: 'SmartCheckout/2.0')
    ];
    
    if ($authType === 'basic') {
        // Basic Auth (AllowPay)
        $credentials = $gateway['api_key'] . ':' . ($gateway['company_id'] ?? '');
        $headers[] = 'Authorization: Basic ' . base64_encode($credentials);
    } elseif ($authType === 'header') {
        // Header customizado (Genesys)
        $authHeader = $gateway['auth_header'] ?? 'api-secret';
        $headers[] = $authHeader . ': ' . $gateway['api_key'];
    } else {
        // API Key padrão (Umbrella, TechByNet)
        $authHeader = $gateway['auth_header'] ?? 'x-api-key';
        $authPrefix = $gateway['auth_prefix'] ?? '';
        $headers[] = $authHeader . ': ' . $authPrefix . $gateway['api_key'];
    }
    
    logCheckout('pix.log', 'Enviando para gateway', [
        'url' => $url,
        'txRef' => $txRef,
        'gateway' => $gatewayName
    ]);
    
    // ============================================
    // FAZ REQUISIÇÃO CURL
    // ============================================
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => true
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    logCheckout('pix.log', 'Resposta do gateway', [
        'gateway' => $gatewayName,
        'http_code' => $httpCode,
        'curl_error' => $curlError ?: 'nenhum',
        'response_preview' => substr($response, 0, 1000)
    ]);
    
    if ($curlError) {
        throw new Exception("Erro de conexão: $curlError");
    }
    
    if (empty($response)) {
        throw new Exception("Gateway retornou resposta vazia (HTTP $httpCode)");
    }
    
    $responseData = json_decode($response, true);
    
    if (!$responseData) {
        logCheckout('pix.log', 'JSON inválido', ['raw' => $response]);
        throw new Exception("Resposta inválida do gateway: " . substr($response, 0, 200));
    }
    
    // ============================================
    // VERIFICAR ERRO NA RESPOSTA
    // ============================================
    $hasData = isset($responseData['data']) || isset($responseData['id']) || isset($responseData['pix']);
    $hasError = isset($responseData['error']) || isset($responseData['errors']);
    $httpSuccess = $httpCode >= 200 && $httpCode < 300;
    
    if ($hasError && !$hasData) {
        $errorMsg = $responseData['error'] ?? $responseData['errors'] ?? $responseData['message'] ?? 'Erro desconhecido';
        if (is_array($errorMsg)) {
            $errorMsg = json_encode($errorMsg, JSON_UNESCAPED_UNICODE);
        }
        logCheckout('pix.log', 'Erro do gateway', ['error' => $errorMsg]);
        throw new Exception("Gateway: $errorMsg");
    }
    
    // ============================================
    // EXTRAIR QR CODE E ID
    // ============================================
    $txId = null;
    $pixCode = null;
    $pixBase64 = null;
    
    // Paths configurados
    $txidField = $gateway['txid_field'] ?? 'data.id';
    $qrcodeField = $gateway['qrcode_field'] ?? 'data.qrCode';
    
    $txId = extractByPath($responseData, $txidField);
    $pixCode = extractByPath($responseData, $qrcodeField);
    
    // Fallbacks para QR Code
    if (!$pixCode) {
        $qrcodePaths = [
            'pix.payload', 'pix.qrcode', 'pix.qrCode',
            'data.qrCode', 'data.qrcode', 'data.data.qrCode',
            'data.pix.qrCode', 'data.pix.payload',
            'qrCode', 'qrcode', 'payload', 'brCode', 'emv'
        ];
        
        foreach ($qrcodePaths as $path) {
            $pixCode = extractByPath($responseData, $path);
            if ($pixCode && strlen($pixCode) > 50) {
                logCheckout('pix.log', "QR Code em: $path");
                break;
            }
        }
    }
    
    // Fallbacks para ID
    if (!$txId) {
        $idPaths = ['id', 'data.id', 'data.data.id', 'objectId', 'transaction_id'];
        foreach ($idPaths as $path) {
            $txId = extractByPath($responseData, $path);
            if ($txId) break;
        }
    }
    
    // QR Code Base64
    $base64Paths = ['data.qrCodeBase64', 'data.data.qrCodeBase64', 'qrCodeBase64', 'pix.qrCodeBase64'];
    foreach ($base64Paths as $path) {
        $pixBase64 = extractByPath($responseData, $path);
        if ($pixBase64) break;
    }
    
    // Se não encontrou txId, usa o txRef local
    if (!$txId) {
        $txId = $txRef;
    }
    
    if (!$pixCode) {
        logCheckout('pix.log', 'QR Code não encontrado', [
            'response_keys' => array_keys($responseData),
            'response_data' => $responseData
        ]);
        throw new Exception("QR Code PIX não encontrado na resposta");
    }
    
    logCheckout('pix.log', 'PIX gerado com sucesso', [
        'txId' => $txId,
        'pixCode_length' => strlen($pixCode),
        'gateway' => $gatewayName
    ]);
    
    // ============================================
    // SALVAR DADOS PARA WEBHOOK/TRACKING
    // ============================================
    $utmStorageDir = defined('UTM_STORAGE_DIR') ? UTM_STORAGE_DIR : (__DIR__ . '/utm_data/');
    if (!is_dir($utmStorageDir)) {
        @mkdir($utmStorageDir, 0755, true);
    }
    
    $utmData = [
        'transaction_id' => $txId,
        'gateway' => $gatewayName,
        'created_at' => date('Y-m-d H:i:s'),
        'valor_centavos' => $valorCentavos,
        'cotas' => $cotas,
        'produto' => $produtoNome,
        'produto_id' => $produtoId,
        // UTMs
        'utm_source' => $tracking['utm_source'] ?? null,
        'utm_medium' => $tracking['utm_medium'] ?? null,
        'utm_campaign' => $tracking['utm_campaign'] ?? null,
        'utm_term' => $tracking['utm_term'] ?? null,
        'utm_content' => $tracking['utm_content'] ?? null,
        'sck' => $tracking['sck'] ?? null,
        'src' => $tracking['src'] ?? null,
        // FACEBOOK - ESSENCIAL PARA QUALIDADE
        'fbc' => $fbc,
        'fbp' => $fbp,
        'fbclid' => $fbclid,
        // DADOS TÉCNICOS
        'customer_ip' => $ip,
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
        'source_url' => $_SERVER['HTTP_REFERER'] ?? '',
        // DADOS DO CLIENTE - ESSENCIAL PARA NOME/SOBRENOME
        'customer' => [
            'name' => $nome,
            'email' => $email,
            'phone' => $telefone,
            'document' => $cpf
        ],
        // ALIASES PARA COMPATIBILIDADE
        'nome' => $nome,
        'email' => $email,
        'telefone' => $telefone,
        'cpf' => $cpf
    ];
    
    // Salva arquivo UTM
    $utmFile = $utmStorageDir . $txId . '.json';
    $saved = @file_put_contents($utmFile, json_encode($utmData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    // Log de cobertura para debug
    $isRealName = !empty($nome) && $nome !== 'Cliente';
    $isRealEmail = !empty($email) && strpos($email, '@pagamento.com') === false; // Email gerado não conta
    $isRealPhone = !empty($telefone) && $telefone !== '11999999999';
    $isRealCpf = !empty($cpf) && $cpf !== '00000000000';
    
    logCheckout('pix.log', '📊 COBERTURA DE DADOS SALVOS', [
        'txId' => $txId,
        'nome' => $isRealName ? '✓ ' . $nome : '✗ PADRÃO',
        'email' => $isRealEmail ? '✓' : '✗ PADRÃO',
        'telefone' => $isRealPhone ? '✓' : '✗ PADRÃO',
        'cpf' => $isRealCpf ? '✓' : '✗ PADRÃO',
        'fbc' => !empty($fbc) ? '✓' : '✗ VAZIO',
        'fbp' => !empty($fbp) ? '✓' : '✗ VAZIO',
        'utm_source' => !empty($tracking['utm_source']) ? '✓' : '✗',
        'arquivo_salvo' => $saved !== false
    ]);
    
    // Aviso se cobertura está baixa
    $coberturaOk = $isRealName && $isRealEmail;
    if (!$coberturaOk) {
        logCheckout('pix.log', '⚠️ ATENÇÃO: Dados do cliente incompletos - qualidade do evento será baixa', [
            'problema' => !$isRealName ? 'NOME não informado na URL' : 'EMAIL não informado na URL'
        ]);
    }
    
    // ============================================
    // UTMIFY (STATUS PENDENTE)
    // ============================================
    if (class_exists('Utmify')) {
        try {
            $utmify = new Utmify();
            $utmify->sendPending(
                $txId,
                $valorCentavos,
                ['name' => $nome, 'email' => $email, 'phone' => $telefone, 'document' => $cpf, 'ip' => $ip],
                ['id' => $produtoId, 'name' => $produtoNome, 'quantity' => $cotas],
                $tracking
            );
        } catch (Exception $e) {
            logCheckout('pix.log', 'Erro UTMify: ' . $e->getMessage());
        }
    }
    
    // ============================================
    // RESPOSTA DE SUCESSO
    // ============================================
    echo json_encode([
        'success' => true,
        'gateway' => $gateway['name'],
        'id' => $txId,
        'txid' => $txId,
        'pix' => [
            'qrcode' => $pixCode,
            'qrcodeBase64' => $pixBase64
        ],
        'qrcode' => $pixCode,
        'qrcodeBase64' => $pixBase64,
        'amount' => $valorCentavos,
        'valor' => $valorReais,
        'status' => 'pending'
    ]);
    
} catch (Exception $e) {
    logCheckout('pix.log', 'ERRO: ' . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}