<?php
/**
 * ============================================
 * WEBHOOK - NOTIFICAÇÕES DE PAGAMENTO
 * ============================================
 * Versão: 3.3 - CORRIGIDO para AllowPay (status em data.status)
 * 
 * Gateways suportados:
 * - AllowPay (ativo) - status em data.status
 * - Umbrella
 * - TechByNet
 * - Genesys Finance
 * 
 * IMPORTANTE: Este é o ÚNICO lugar que envia Purchase!
 */

date_default_timezone_set('America/Sao_Paulo');

require_once __DIR__ . '/config.php';

// ============================================
// VERIFICAÇÃO DE ARQUIVOS CAPI
// ============================================
$fbCapiFile = __DIR__ . '/facebook_capi.php';
$utmifyFile = __DIR__ . '/utmify.php';

if (file_exists($fbCapiFile)) {
    require_once $fbCapiFile;
} else {
    logCheckout('webhook.log', '⚠️ AVISO: facebook_capi.php NÃO ENCONTRADO em ' . $fbCapiFile);
}

if (file_exists($utmifyFile)) {
    require_once $utmifyFile;
}

// ============================================
// LOG DETALHADO
// ============================================
$rawInput = file_get_contents('php://input');
$data = json_decode($rawInput, true);

logCheckout('webhook.log', '========== WEBHOOK RECEBIDO ==========', [
    'method' => $_SERVER['REQUEST_METHOD'],
    'content_type' => $_SERVER['CONTENT_TYPE'] ?? 'não definido',
    'raw_length' => strlen($rawInput),
    'remote_ip' => $_SERVER['REMOTE_ADDR'] ?? 'desconhecido'
]);

// Log do payload completo para debug
logCheckout('webhook_payloads.log', 'Payload', [
    'timestamp' => date('Y-m-d H:i:s'),
    'raw' => substr($rawInput, 0, 3000)
]);

if (!$data) {
    logCheckout('webhook.log', '❌ ERRO: Payload inválido ou vazio');
    http_response_code(400);
    echo json_encode(['error' => 'Payload inválido']);
    exit;
}

// ============================================
// CARREGA CONFIGURAÇÕES DOS GATEWAYS
// ============================================
$gatewaysConfig = loadGatewaysConfig();
$allGateways = $gatewaysConfig['gateways'] ?? [];

// ============================================
// DETECTA O GATEWAY AUTOMATICAMENTE
// ============================================
function detectGateway($data, $allGateways) {
    
    // 1. GENESYS: tem external_id OU (id + total_amount sem objectId)
    if (isset($data['external_id'])) {
        return 'genesys';
    }
    if (isset($data['id']) && isset($data['total_amount']) && !isset($data['objectId'])) {
        return 'genesys';
    }
    
    // 2. ALLOWPAY: tem objectId E data.id (estrutura típica do AllowPay)
    if (isset($data['objectId']) && isset($data['data']['id'])) {
        return 'allowpay';
    }
    
    // 3. UMBRELLA: tem objectId E data.data como ARRAY
    if (isset($data['objectId']) && isset($data['data']['data']) && is_array($data['data']['data']) && isset($data['data']['data'][0])) {
        return 'umbrella';
    }
    
    // 4. TECHBYNET: tem data.data como OBJETO (não array)
    if (isset($data['data']['data']) && is_array($data['data']['data']) && !isset($data['data']['data'][0])) {
        return 'techbynet';
    }
    
    // 5. ALLOWPAY alternativo: tem type "transaction"
    if (isset($data['type']) && $data['type'] === 'transaction') {
        return 'allowpay';
    }
    
    // 6. TECHBYNET: tem transaction_id ou pix_code
    if (isset($data['transaction_id']) || isset($data['pix_code'])) {
        return 'techbynet';
    }
    
    // 7. Fallback: usa gateway ativo da config
    return getGatewayName();
}

$detectedGateway = detectGateway($data, $allGateways);
$gatewayConfig = $allGateways[$detectedGateway] ?? [];

logCheckout('webhook.log', '🔍 Gateway detectado: ' . strtoupper($detectedGateway), [
    'has_config' => !empty($gatewayConfig),
    'keys_root' => array_keys($data),
    'has_data_object' => isset($data['data']),
    'status_root' => $data['status'] ?? 'N/A',
    'status_data' => $data['data']['status'] ?? 'N/A'
]);

// ============================================
// EXTRAI DADOS USANDO CONFIGURAÇÃO DO GATEWAY
// ============================================

$txId = null;
$status = null;
$amount = null;
$customerData = [];

switch ($detectedGateway) {
    
    // ========== ALLOWPAY ==========
    // IMPORTANTE: AllowPay envia dados dentro de "data"
    case 'allowpay':
        // O AllowPay envia: { id, type, objectId, data: { id, status, amount, customer, ... } }
        $txData = $data['data'] ?? [];
        
        // ID da transação (usa o de dentro de data)
        $txId = $txData['id'] ?? $data['objectId'] ?? $data['id'] ?? null;
        
        // STATUS - CRÍTICO: está dentro de data.status
        $status = $txData['status'] ?? $data['status'] ?? null;
        
        // Valor
        $amount = $txData['amount'] ?? $txData['value'] ?? $data['amount'] ?? null;
        
        // Customer - está dentro de data.customer
        $customer = $txData['customer'] ?? $data['customer'] ?? [];
        if ($customer) {
            $customerData = [
                'name' => $customer['name'] ?? null,
                'email' => $customer['email'] ?? null,
                'phone' => $customer['phone'] ?? $customer['cellphone'] ?? null,
                'document' => $customer['document'] ?? $customer['cpf'] ?? null
            ];
        }
        
        logCheckout('webhook.log', '📦 AllowPay - Extração', [
            'txId' => $txId,
            'status' => $status,
            'amount' => $amount,
            'customer_name' => $customerData['name'] ?? 'N/A'
        ]);
        break;
    
    // ========== UMBRELLA ==========
    case 'umbrella':
        // Umbrella: data.data[0].xxx
        $txData = $data['data']['data'][0] ?? [];
        
        $txId = $txData['id'] ?? $data['objectId'] ?? null;
        $status = $txData['status'] ?? null;
        $amount = $txData['amount'] ?? $txData['value'] ?? null;
        
        $customer = $txData['customer'] ?? $txData['payer'] ?? [];
        if ($customer) {
            $customerData = [
                'name' => $customer['name'] ?? null,
                'email' => $customer['email'] ?? null,
                'phone' => $customer['phone'] ?? $customer['phoneNumber'] ?? null,
                'document' => $customer['document'] ?? $customer['cpf'] ?? $customer['taxId'] ?? null
            ];
        }
        break;
    
    // ========== TECHBYNET ==========
    case 'techbynet':
        $txData = $data['data']['data'] ?? $data['data'] ?? $data;
        
        if (isset($txData[0])) {
            $txData = $txData[0];
        }
        
        $txId = $txData['id'] ?? $data['transaction_id'] ?? $data['id'] ?? null;
        $status = $txData['status'] ?? $data['status'] ?? null;
        $amount = $txData['amount'] ?? $txData['value'] ?? null;
        
        $customer = $txData['customer'] ?? $txData['payer'] ?? $data['customer'] ?? [];
        if ($customer) {
            $customerData = [
                'name' => $customer['name'] ?? null,
                'email' => $customer['email'] ?? null,
                'phone' => $customer['phone'] ?? null,
                'document' => $customer['document'] ?? $customer['cpf'] ?? null
            ];
        }
        break;
    
    // ========== GENESYS ==========
    case 'genesys':
        $txId = $data['id'] ?? $data['external_id'] ?? null;
        $status = $data['status'] ?? null;
        
        // Genesys envia em REAIS
        if (isset($data['total_amount'])) {
            $amount = intval(floatval($data['total_amount']) * 100);
        } elseif (isset($data['amount'])) {
            $amt = $data['amount'];
            if ($amt < 1000) {
                $amount = intval(floatval($amt) * 100);
            } else {
                $amount = intval($amt);
            }
        }
        
        $customer = $data['customer'] ?? [];
        if ($customer) {
            $customerData = [
                'name' => $customer['name'] ?? null,
                'email' => $customer['email'] ?? null,
                'phone' => $customer['phone'] ?? null,
                'document' => $customer['document'] ?? $customer['cpf'] ?? null
            ];
        }
        break;
    
    // ========== GENÉRICO ==========
    default:
        $txId = $data['id'] ?? $data['objectId'] ?? $data['transaction_id'] ?? null;
        $status = $data['status'] ?? $data['data']['status'] ?? null;
        $amount = $data['amount'] ?? $data['value'] ?? $data['total_amount'] ?? null;
        
        if ($amount) {
            $amount = intval($amount);
        }
        
        $customer = $data['customer'] ?? [];
        if ($customer) {
            $customerData = [
                'name' => $customer['name'] ?? null,
                'email' => $customer['email'] ?? null,
                'phone' => $customer['phone'] ?? null,
                'document' => $customer['document'] ?? null
            ];
        }
}

// Garante que amount é inteiro (centavos)
if ($amount !== null) {
    $amount = intval($amount);
}

logCheckout('webhook.log', '📦 Dados extraídos FINAL', [
    'txId' => $txId,
    'status' => $status,
    'amount_centavos' => $amount,
    'amount_reais' => $amount ? 'R$ ' . number_format($amount / 100, 2, ',', '.') : null,
    'has_customer' => !empty(array_filter($customerData))
]);

// ============================================
// VALIDAÇÃO
// ============================================
if (!$txId) {
    logCheckout('webhook.log', '❌ ERRO: txId não encontrado', [
        'payload_keys' => array_keys($data)
    ]);
    http_response_code(400);
    echo json_encode(['error' => 'Transaction ID não encontrado', 'gateway' => $detectedGateway]);
    exit;
}

// ============================================
// VERIFICA SE É PAGAMENTO CONFIRMADO
// ============================================
$isPaid = isPaidStatus($status);

logCheckout('webhook.log', '🔍 Análise de Status', [
    'status_original' => $status,
    'status_lower' => strtolower(trim($status ?? '')),
    'is_paid' => $isPaid ? '✅ SIM' : '❌ NÃO'
]);

// ============================================
// PROCESSA PAGAMENTO CONFIRMADO
// ============================================
if ($isPaid) {
    logCheckout('webhook.log', '💰 ========== PAGAMENTO CONFIRMADO ==========', [
        'txId' => $txId,
        'gateway' => $detectedGateway,
        'status' => $status,
        'valor' => $amount ? 'R$ ' . number_format($amount / 100, 2, ',', '.') : 'não informado'
    ]);
    
    // Carrega dados UTM salvos
    $utmFile = UTM_STORAGE_DIR . $txId . '.json';
    $utmData = [];
    
    if (file_exists($utmFile)) {
        $utmData = json_decode(file_get_contents($utmFile), true) ?: [];
        logCheckout('webhook.log', '📂 UTM carregado', [
            'file' => basename($utmFile),
            'has_fbc' => !empty($utmData['fbc']),
            'has_fbp' => !empty($utmData['fbp']),
            'has_email' => !empty($utmData['customer']['email'] ?? $utmData['email'])
        ]);
    } else {
        logCheckout('webhook.log', '⚠️ UTM não encontrado: ' . $utmFile);
        if (!empty($customerData)) {
            $utmData['customer'] = $customerData;
        }
    }
    
    // Calcula valor
    $valorCentavos = $amount ?: ($utmData['valor_centavos'] ?? 0);
    $valorReais = $valorCentavos / 100;
    
    if ($valorReais <= 0) {
        $valorReais = defined('VALOR_PADRAO_REAIS') ? VALOR_PADRAO_REAIS : 59.90;
        $valorCentavos = intval($valorReais * 100);
        logCheckout('webhook.log', '⚠️ Usando valor padrão: R$ ' . number_format($valorReais, 2, ',', '.'));
    }
    
    // ========== FACEBOOK CAPI - PURCHASE ==========
    $fbResult = ['success' => false, 'error' => 'Classe não carregada'];
    
    $fbClassExists = class_exists('FacebookCAPI');
    logCheckout('webhook.log', '📘 Facebook CAPI - Verificação', [
        'class_exists' => $fbClassExists ? 'SIM' : 'NÃO'
    ]);
    
    if ($fbClassExists) {
        try {
            $fb = new FacebookCAPI();
            
            // Garante que customer está preenchido
            if (empty($utmData['customer']) && !empty($customerData)) {
                $utmData['customer'] = $customerData;
            }
            $utmData['valor_centavos'] = $valorCentavos;
            
            logCheckout('webhook.log', '📘 Enviando Purchase...', [
                'txId' => $txId,
                'valor' => $valorReais,
                'fbc' => !empty($utmData['fbc']) ? 'presente' : 'VAZIO',
                'fbp' => !empty($utmData['fbp']) ? 'presente' : 'VAZIO'
            ]);
            
            $fbResult = $fb->sendPurchase($txId, $valorReais, $utmData);
            
            logCheckout('webhook.log', '📘 Facebook CAPI - Resultado', [
                'success' => $fbResult['success'] ?? false,
                'skipped' => $fbResult['skipped'] ?? false,
                'event_id' => $fbResult['event_id'] ?? null,
                'events_received' => $fbResult['events_received'] ?? 0
            ]);
        } catch (Exception $e) {
            logCheckout('webhook.log', '❌ FB CAPI EXCEPTION: ' . $e->getMessage());
            $fbResult = ['success' => false, 'error' => $e->getMessage()];
        }
    } else {
        logCheckout('webhook.log', '❌ ERRO: Classe FacebookCAPI não existe!');
    }
    
    // ========== UTMIFY ==========
    $utmifyResult = ['success' => false];
    
    if (class_exists('Utmify')) {
        try {
            $utmify = new Utmify();
            
            if ($utmify->isConfigured()) {
                $customer = [
                    'name' => $utmData['customer']['name'] ?? $utmData['nome'] ?? $customerData['name'] ?? '',
                    'email' => $utmData['customer']['email'] ?? $utmData['email'] ?? $customerData['email'] ?? '',
                    'phone' => $utmData['customer']['phone'] ?? $utmData['telefone'] ?? $customerData['phone'] ?? '',
                    'document' => $utmData['customer']['document'] ?? $utmData['cpf'] ?? $customerData['document'] ?? '',
                    'ip' => $utmData['customer_ip'] ?? ''
                ];
                
                $product = [
                    'id' => $utmData['produto_id'] ?? PRODUTO_ID ?? 'produto-001',
                    'name' => $utmData['produto'] ?? PRODUTO_NOME ?? 'Produto',
                    'quantity' => $utmData['cotas'] ?? 1
                ];
                
                $utms = [
                    'utm_source' => $utmData['utm_source'] ?? '',
                    'utm_medium' => $utmData['utm_medium'] ?? '',
                    'utm_campaign' => $utmData['utm_campaign'] ?? '',
                    'utm_content' => $utmData['utm_content'] ?? '',
                    'utm_term' => $utmData['utm_term'] ?? '',
                    'sck' => $utmData['sck'] ?? '',
                    'src' => $utmData['src'] ?? ''
                ];
                
                $utmifyResult = $utmify->sendPaid($txId, $valorCentavos, $customer, $product, $utms);
                
                logCheckout('webhook.log', '📊 UTMify', [
                    'success' => $utmifyResult['success'] ?? false
                ]);
            }
        } catch (Exception $e) {
            logCheckout('webhook.log', '❌ UTMify erro: ' . $e->getMessage());
        }
    }
    
    // Salva resultado
    $utmData['paid'] = true;
    $utmData['paid_at'] = date('Y-m-d H:i:s');
    $utmData['webhook_gateway'] = $detectedGateway;
    $utmData['webhook_status'] = $status;
    $utmData['webhook_amount'] = $amount;
    $utmData['fb_success'] = $fbResult['success'] ?? false;
    $utmData['fb_result'] = $fbResult;
    
    @file_put_contents($utmFile, json_encode($utmData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    logCheckout('webhook.log', '✅ PROCESSAMENTO CONCLUÍDO', [
        'txId' => $txId,
        'valor' => 'R$ ' . number_format($valorReais, 2, ',', '.'),
        'fb_success' => $fbResult['success'] ?? false,
        'utmify_success' => $utmifyResult['success'] ?? false
    ]);
    
} else {
    logCheckout('webhook.log', '⏳ Status NÃO é pagamento', [
        'status' => $status,
        'tipo' => $data['type'] ?? 'N/A'
    ]);
}

// Resposta
http_response_code(200);
echo json_encode([
    'success' => true,
    'txId' => $txId,
    'status' => $status,
    'isPaid' => $isPaid,
    'gateway' => $detectedGateway
]);