<?php
/**
 * ============================================
 * TRACKING.PHP - COM DEBUG v4.1
 * ============================================
 */

// Captura erros fatais
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        $logDir = __DIR__ . '/logs';
        if (!is_dir($logDir)) @mkdir($logDir, 0755, true);
        $msg = "[" . date('Y-m-d H:i:s') . "] ❌ FATAL: {$error['message']} em {$error['file']}:{$error['line']}\n";
        @file_put_contents($logDir . '/tracking_errors.log', $msg, FILE_APPEND);
        
        if (!headers_sent()) {
            header('Content-Type: application/json');
            http_response_code(500);
        }
        echo json_encode([
            'success' => false,
            'error' => 'Fatal error: ' . $error['message'],
            'file' => basename($error['file']),
            'line' => $error['line']
        ]);
    }
});

date_default_timezone_set('America/Sao_Paulo');

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { 
    http_response_code(200); 
    exit; 
}

// ============================================
// LOG
// ============================================
function logTracking($message, $data = []) {
    $logDir = __DIR__ . '/logs';
    if (!is_dir($logDir)) @mkdir($logDir, 0755, true);
    
    $timestamp = date('Y-m-d H:i:s');
    $logLine = "[$timestamp] $message";
    if (!empty($data)) $logLine .= ' | ' . json_encode($data, JSON_UNESCAPED_UNICODE);
    @file_put_contents($logDir . '/tracking.log', $logLine . "\n", FILE_APPEND);
}

// ============================================
// VERIFICA DEPENDÊNCIAS
// ============================================
$configPath = __DIR__ . '/config.php';
$capiPath = __DIR__ . '/facebook_capi.php';

if (!file_exists($configPath)) {
    logTracking("❌ config.php não encontrado");
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'config.php não encontrado']);
    exit;
}

if (!file_exists($capiPath)) {
    logTracking("❌ facebook_capi.php não encontrado");
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'facebook_capi.php não encontrado']);
    exit;
}

require_once $configPath;
require_once $capiPath;

// Verifica se classe existe
if (!class_exists('FacebookCAPI')) {
    logTracking("❌ Classe FacebookCAPI não encontrada");
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Classe FacebookCAPI não encontrada']);
    exit;
}

// ============================================
// ANTI-DUPLICAÇÃO
// ============================================
function isEventDuplicate($eventId) {
    if (empty($eventId)) return false;
    
    $cacheDir = __DIR__ . '/cache';
    if (!is_dir($cacheDir)) @mkdir($cacheDir, 0755, true);
    
    $cacheFile = $cacheDir . '/events_' . date('Y-m-d-H') . '.json';
    $cache = file_exists($cacheFile) ? json_decode(file_get_contents($cacheFile), true) ?: [] : [];
    
    if (isset($cache[$eventId])) return true;
    
    $cache[$eventId] = time();
    
    // Limpa eventos antigos
    $now = time();
    foreach ($cache as $id => $time) {
        if ($now - $time > 600) unset($cache[$id]);
    }
    
    @file_put_contents($cacheFile, json_encode($cache));
    return false;
}

// ============================================
// RECEBE DADOS
// ============================================
$rawInput = file_get_contents('php://input');
$input = json_decode($rawInput, true);

if (!$input) {
    logTracking("❌ JSON inválido", ['raw' => substr($rawInput, 0, 200)]);
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Dados inválidos']);
    exit;
}

$event = $input['event'] ?? $input['event_name'] ?? null;
$eventId = $input['event_id'] ?? null;

if (empty($event)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Evento não especificado']);
    exit;
}

// ============================================
// BLOQUEIO: PURCHASE NÃO PERMITIDO
// ============================================
if ($event === 'Purchase') {
    logTracking("⛔ PURCHASE BLOQUEADO", ['event_id' => $eventId]);
    http_response_code(403);
    echo json_encode([
        'success' => false,
        'error' => 'Purchase deve ser enviado pelo webhook'
    ]);
    exit;
}

// Anti-duplicação
if (isEventDuplicate($eventId)) {
    logTracking("⚠️ DUPLICADO: $event", ['event_id' => $eventId]);
    echo json_encode(['success' => true, 'skipped' => true, 'reason' => 'duplicate']);
    exit;
}

// ============================================
// EXTRAI DADOS
// ============================================
$sourceUrl = $input['source_url'] ?? ($_SERVER['HTTP_REFERER'] ?? '');
$value = floatval($input['value'] ?? 0);
if ($value <= 0) {
    $value = defined('VALOR_PADRAO_REAIS') ? VALOR_PADRAO_REAIS : 59.90;
}

$numItems = intval($input['num_items'] ?? 1);

// Facebook cookies
$fbc = $input['fbc'] ?? $_COOKIE['_fbc'] ?? null;
$fbp = $input['fbp'] ?? $_COOKIE['_fbp'] ?? null;
$fbclid = $input['fbclid'] ?? null;

// Valida fbc
if (!empty($fbc) && strpos($fbc, 'fb.') !== 0) {
    logTracking("⚠️ FBC inválido: $fbc");
    $fbc = null;
}

// Gera fbc do fbclid
if (empty($fbc) && !empty($fbclid)) {
    $fbc = 'fb.1.' . time() . '000.' . $fbclid;
}

// Gera fbp se necessário
if (empty($fbp) || strpos($fbp, 'fb.') !== 0) {
    $fbp = 'fb.1.' . round(microtime(true) * 1000) . '.' . mt_rand(1000000000, 9999999999);
}

// IP
$clientIp = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['HTTP_X_REAL_IP'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
if (strpos($clientIp, ',') !== false) $clientIp = trim(explode(',', $clientIp)[0]);

$userData = [
    'email' => $input['email'] ?? null,
    'phone' => $input['phone'] ?? null,
    'name' => $input['name'] ?? null,
    'document' => $input['document'] ?? null,
    'fbc' => $fbc,
    'fbp' => $fbp,
    'fbclid' => $fbclid,
    'user_agent' => $input['user_agent'] ?? $_SERVER['HTTP_USER_AGENT'] ?? '',
    'ip' => $clientIp,
    'content_name' => $input['content_name'] ?? (defined('PRODUTO_NOME') ? PRODUTO_NOME : 'Produto'),
    'content_id' => $input['content_id'] ?? (defined('PRODUTO_ID') ? PRODUTO_ID : 'hvzu616mv7'),
    // UTMs
    'utm_source' => $input['utm_source'] ?? null,
    'utm_medium' => $input['utm_medium'] ?? null,
    'utm_campaign' => $input['utm_campaign'] ?? null,
    'utm_content' => $input['utm_content'] ?? null,
    'utm_term' => $input['utm_term'] ?? null
];

logTracking("📥 $event", [
    'event_id' => $eventId,
    'value' => $value,
    'has_fbc' => !empty($fbc),
    'has_fbp' => !empty($fbp)
]);

// ============================================
// ENVIA PARA FACEBOOK CAPI
// ============================================
try {
    $fb = new FacebookCAPI();
    $result = ['success' => false, 'error' => 'Evento não processado'];
    
    switch ($event) {
        case 'PageView':
            $result = $fb->pageView($userData, $sourceUrl, $eventId);
            break;
            
        case 'ViewContent':
            $result = $fb->viewContent($userData, $value, $sourceUrl, $eventId);
            break;
            
        case 'AddToCart':
            if (method_exists($fb, 'addToCart')) {
                $result = $fb->addToCart($userData, $value, $numItems, $sourceUrl, $eventId);
            } else {
                $result = ['success' => true, 'skipped' => true, 'reason' => 'method_not_exists'];
            }
            break;
            
        case 'InitiateCheckout':
            if (method_exists($fb, 'initiateCheckout')) {
                // Verifica assinatura do método
                $ref = new ReflectionMethod($fb, 'initiateCheckout');
                $paramCount = $ref->getNumberOfParameters();
                
                if ($paramCount >= 5) {
                    // Assinatura: ($data, $value, $numItems, $sourceUrl, $eventId)
                    $result = $fb->initiateCheckout($userData, $value, $numItems, $sourceUrl, $eventId);
                } else {
                    // Assinatura: ($data, $value, $sourceUrl, $eventId)
                    $result = $fb->initiateCheckout($userData, $value, $sourceUrl, $eventId);
                }
            } else {
                $result = ['success' => true, 'skipped' => true, 'reason' => 'method_not_exists'];
            }
            break;
            
        case 'AddPaymentInfo':
            if (method_exists($fb, 'addPaymentInfo')) {
                $result = $fb->addPaymentInfo($userData, $value, $sourceUrl, $eventId);
            } else {
                $result = ['success' => true, 'skipped' => true, 'reason' => 'method_not_exists'];
            }
            break;
            
        default:
            $result = ['success' => false, 'error' => "Evento '$event' não suportado"];
    }
    
    logTracking("📤 $event", ['success' => $result['success'] ?? false]);
    echo json_encode($result);
    
} catch (Exception $e) {
    logTracking("❌ Exception: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
} catch (Error $e) {
    logTracking("❌ Error: " . $e->getMessage(), [
        'file' => basename($e->getFile()),
        'line' => $e->getLine()
    ]);
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}