/**
 * ============================================
 * SMART TRACKING v8 - QUALIDADE MÁXIMA
 * ============================================
 * 
 * MELHORIAS v8:
 * - Captura TODOS os dados do cliente da URL
 * - nome, cpf, email, telefone, nascimento, sexo
 * - Envia para CAPI com qualidade máxima
 * - Produto ID corrigido para catálogo Facebook
 * 
 * INSTALAÇÃO:
 * <script src="/seguro/consulta/tarifa/api/smart-tracking.js"></script>
 */

(function() {
    'use strict';
    
    // ============================================
    // CONFIGURAÇÃO
    // ============================================
    var CONFIG = {
        pixelId: '2357252418054828',
        trackingEndpoint: '/seguro/consulta/tarifa/api/tracking.php',
        defaultValue: 59.90,
        defaultCurrency: 'BRL',
        // PRODUTO DO CATÁLOGO FACEBOOK
        defaultContentName: 'ctv04 - video',
        defaultContentId: 'hvzu616mv7',
        debug: true,
        blockPurchaseFromFrontend: true
    };
    
    var sentEventIds = {};
    
    // ============================================
    // UTILITÁRIOS
    // ============================================
    function log(msg, data) {
        if (CONFIG.debug) console.log('[SmartTracking]', msg, data || '');
    }
    
    function warn(msg, data) {
        console.warn('[SmartTracking]', msg, data || '');
    }
    
    function getCookie(name) {
        try {
            var value = '; ' + document.cookie;
            var parts = value.split('; ' + name + '=');
            if (parts.length === 2) {
                return parts.pop().split(';').shift();
            }
        } catch(e) {}
        return null;
    }
    
    function setCookie(name, value, days) {
        try {
            days = days || 90;
            var expires = new Date(Date.now() + days * 864e5).toUTCString();
            document.cookie = name + '=' + value + '; expires=' + expires + '; path=/; SameSite=Lax';
        } catch(e) {}
    }
    
    function getLocal(name) {
        try {
            return localStorage.getItem(name);
        } catch(e) {
            return null;
        }
    }
    
    function setLocal(name, value) {
        try {
            localStorage.setItem(name, value);
        } catch(e) {}
    }
    
    function getUrlParam(param) {
        try {
            var urlParams = new URLSearchParams(window.location.search);
            return urlParams.get(param);
        } catch(e) {
            var regex = new RegExp('[?&]' + param + '=([^&#]*)');
            var results = regex.exec(window.location.search);
            return results ? decodeURIComponent(results[1]) : null;
        }
    }
    
    // ============================================
    // FBC - Facebook Click ID
    // ============================================
    function getFbc() {
        var fbc = null;
        
        fbc = getCookie('_fbc');
        if (fbc && fbc.indexOf('fb.') === 0) {
            setLocal('_fbc', fbc);
            return fbc;
        }
        
        fbc = getLocal('_fbc');
        if (fbc && fbc.indexOf('fb.') === 0) {
            setCookie('_fbc', fbc);
            return fbc;
        }
        
        var fbclid = getUrlParam('fbclid') || getLocal('fbclid');
        if (fbclid) {
            fbc = 'fb.1.' + Date.now() + '.' + fbclid;
            setCookie('_fbc', fbc);
            setLocal('_fbc', fbc);
            return fbc;
        }
        
        return null;
    }
    
    // ============================================
    // FBP - Facebook Browser ID
    // ============================================
    function getFbp() {
        var fbp = null;
        
        fbp = getCookie('_fbp');
        if (fbp && fbp.indexOf('fb.') === 0) {
            setLocal('_fbp', fbp);
            return fbp;
        }
        
        fbp = getLocal('_fbp');
        if (fbp && fbp.indexOf('fb.') === 0) {
            setCookie('_fbp', fbp);
            return fbp;
        }
        
        var randomId = Math.floor(Math.random() * 10000000000);
        fbp = 'fb.1.' + Date.now() + '.' + randomId;
        setCookie('_fbp', fbp);
        setLocal('_fbp', fbp);
        
        return fbp;
    }
    
    function saveFbclid() {
        var fbclid = getUrlParam('fbclid');
        if (fbclid) {
            setLocal('fbclid', fbclid);
        }
    }
    
    // ============================================
    // CAPTURA DADOS DO CLIENTE DA URL (NOVO v8!)
    // ============================================
    function getClientDataFromUrl() {
        return {
            // Nome completo
            name: getUrlParam('nome') || getUrlParam('name') || '',
            
            // Primeiro nome (se disponível separado)
            first_name: getUrlParam('primeiro_nome') || getUrlParam('first_name') || '',
            
            // CPF/Documento
            document: (getUrlParam('cpf') || getUrlParam('document') || '').replace(/\D/g, ''),
            
            // Email
            email: getUrlParam('email') || '',
            
            // Telefone (múltiplas fontes)
            phone: (getUrlParam('telefone') || getUrlParam('telephone') || getUrlParam('celular') || getUrlParam('PIX') || '').replace(/\D/g, ''),
            
            // Data de nascimento
            birth_date: getUrlParam('nascimento') || getUrlParam('data_nascimento') || getUrlParam('birth_date') || '',
            
            // Sexo/Gênero
            gender: getUrlParam('sexo') || getUrlParam('genero') || getUrlParam('gender') || '',
            
            // Valor total (para contexto)
            total: parseFloat(getUrlParam('total') || getUrlParam('valor') || '0')
        };
    }
    
    // ============================================
    // CAPTURA DADOS DO LOCALSTORAGE (NOVO v8!)
    // ============================================
    function getClientDataFromStorage() {
        var sources = ['cliente', 'trackingClienteData', 'dadosCliente'];
        
        for (var i = 0; i < sources.length; i++) {
            try {
                var data = getLocal(sources[i]);
                if (data) {
                    var parsed = JSON.parse(data);
                    if (parsed && (parsed.nome || parsed.name || parsed.cpf)) {
                        return {
                            name: parsed.nome || parsed.name || '',
                            first_name: parsed.primeiro_nome || '',
                            document: (parsed.cpf || parsed.document || '').toString().replace(/\D/g, ''),
                            email: parsed.email || '',
                            phone: (parsed.telefone || parsed.celular || parsed.telephone || '').toString().replace(/\D/g, ''),
                            birth_date: parsed.nascimento || parsed.birth_date || '',
                            gender: parsed.sexo || parsed.gender || ''
                        };
                    }
                }
            } catch(e) {}
        }
        
        return {};
    }
    
    // ============================================
    // MESCLA DADOS DE TODAS AS FONTES (NOVO v8!)
    // ============================================
    function getAllClientData() {
        var urlData = getClientDataFromUrl();
        var storageData = getClientDataFromStorage();
        
        // Prioriza URL sobre storage
        var merged = {
            name: urlData.name || storageData.name || '',
            first_name: urlData.first_name || storageData.first_name || '',
            document: urlData.document || storageData.document || '',
            email: urlData.email || storageData.email || '',
            phone: urlData.phone || storageData.phone || '',
            birth_date: urlData.birth_date || storageData.birth_date || '',
            gender: urlData.gender || storageData.gender || ''
        };
        
        // Extrai primeiro nome do nome completo se não tiver
        if (!merged.first_name && merged.name) {
            var parts = merged.name.trim().split(' ');
            merged.first_name = parts[0] || '';
            if (parts.length > 1) {
                merged.last_name = parts[parts.length - 1] || '';
            }
        }
        
        log('Dados do cliente coletados:', {
            has_name: !!merged.name,
            has_document: !!merged.document,
            has_email: !!merged.email,
            has_phone: !!merged.phone,
            has_birth_date: !!merged.birth_date,
            has_gender: !!merged.gender
        });
        
        return merged;
    }
    
    // ============================================
    // COLETA TODOS OS DADOS DO USUÁRIO
    // ============================================
    function getUserData() {
        var fbc = getFbc();
        var fbp = getFbp();
        var clientData = getAllClientData();
        
        return {
            // Facebook cookies
            fbc: fbc,
            fbp: fbp,
            fbclid: getUrlParam('fbclid') || getLocal('fbclid'),
            user_agent: navigator.userAgent,
            source_url: window.location.href,
            
            // DADOS DO CLIENTE (NOVO v8!)
            name: clientData.name,
            first_name: clientData.first_name,
            last_name: clientData.last_name || '',
            document: clientData.document,
            email: clientData.email,
            phone: clientData.phone,
            birth_date: clientData.birth_date,
            gender: clientData.gender,
            
            // UTMs
            utm_source: getUrlParam('utm_source'),
            utm_medium: getUrlParam('utm_medium'),
            utm_campaign: getUrlParam('utm_campaign'),
            utm_content: getUrlParam('utm_content'),
            utm_term: getUrlParam('utm_term'),
            sck: getUrlParam('sck'),
            src: getUrlParam('src')
        };
    }
    
    // ============================================
    // GERA EVENT ID ÚNICO
    // ============================================
    function generateEventId(eventName) {
        var random = Math.random().toString(36).substring(2, 10);
        return eventName.toLowerCase() + '_' + Date.now() + '_' + random;
    }
    
    // ============================================
    // VERIFICA DUPLICAÇÃO
    // ============================================
    function isDuplicate(eventId) {
        if (sentEventIds[eventId]) {
            return true;
        }
        sentEventIds[eventId] = Date.now();
        
        var oneHourAgo = Date.now() - 3600000;
        for (var id in sentEventIds) {
            if (sentEventIds[id] < oneHourAgo) {
                delete sentEventIds[id];
            }
        }
        
        return false;
    }
    
    // ============================================
    // ENVIA PARA SERVIDOR (CAPI)
    // ============================================
    function sendToServer(eventName, eventData, eventId) {
        if (isDuplicate(eventId)) {
            log('Evento duplicado ignorado:', eventId);
            return;
        }
        
        var userData = getUserData();
        
        var payload = {
            event_name: eventName,
            event_id: eventId,
            event_time: Math.floor(Date.now() / 1000),
            source_url: window.location.href,
            
            // FACEBOOK - ESSENCIAL PARA MATCH RATE
            fbc: userData.fbc,
            fbp: userData.fbp,
            fbclid: userData.fbclid,
            user_agent: userData.user_agent,
            
            // DADOS DO CLIENTE (NOVO v8!) - QUALIDADE MÁXIMA
            name: userData.name,
            first_name: userData.first_name,
            last_name: userData.last_name,
            document: userData.document,
            email: userData.email,
            phone: userData.phone,
            birth_date: userData.birth_date,
            gender: userData.gender,
            
            // UTMs
            utm_source: userData.utm_source,
            utm_medium: userData.utm_medium,
            utm_campaign: userData.utm_campaign,
            utm_content: userData.utm_content,
            utm_term: userData.utm_term,
            sck: userData.sck,
            src: userData.src,
            
            // Dados do evento
            value: eventData.value || CONFIG.defaultValue,
            currency: eventData.currency || CONFIG.defaultCurrency,
            content_name: eventData.content_name || CONFIG.defaultContentName,
            content_ids: eventData.content_ids || [CONFIG.defaultContentId],
            content_id: (eventData.content_ids && eventData.content_ids[0]) || CONFIG.defaultContentId,
            content_type: eventData.content_type || 'product',
            num_items: eventData.num_items || 1
        };
        
        log('📤 Enviando ' + eventName + ':', {
            has_name: !!payload.name,
            has_document: !!payload.document,
            has_birth_date: !!payload.birth_date,
            has_gender: !!payload.gender,
            has_fbc: !!payload.fbc,
            has_fbp: !!payload.fbp
        });
        
        // Usa sendBeacon se disponível
        if (navigator.sendBeacon) {
            try {
                var blob = new Blob([JSON.stringify(payload)], { type: 'application/json' });
                navigator.sendBeacon(CONFIG.trackingEndpoint, blob);
            } catch(e) {
                sendViaFetch(payload);
            }
        } else {
            sendViaFetch(payload);
        }
    }
    
    function sendViaFetch(payload) {
        try {
            fetch(CONFIG.trackingEndpoint, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload),
                keepalive: true
            }).then(function(response) {
                log('Resposta:', response.status);
            }).catch(function(err) {
                warn('Erro:', err);
            });
        } catch(e) {
            warn('Fetch não suportado');
        }
    }
    
    // ============================================
    // INTERCEPTA FBQ
    // ============================================
    function interceptFbq() {
        if (typeof window.fbq === 'undefined') {
            setTimeout(interceptFbq, 500);
            return;
        }
        
        var originalFbq = window.fbq;
        
        window.fbq = function() {
            var args = Array.prototype.slice.call(arguments);
            
            // Chama fbq original primeiro
            originalFbq.apply(this, args);
            
            // Intercepta eventos 'track'
            if (args[0] === 'track' && args[1]) {
                var eventName = args[1];
                var eventData = args[2] || {};
                
                // BLOQUEIA Purchase no frontend
                if (eventName === 'Purchase' && CONFIG.blockPurchaseFromFrontend) {
                    warn('Purchase BLOQUEADO (enviado pelo webhook)');
                    return;
                }
                
                var eventId = generateEventId(eventName);
                var eventsToTrack = ['PageView', 'ViewContent', 'AddToCart', 'InitiateCheckout', 'AddPaymentInfo'];
                
                if (eventsToTrack.indexOf(eventName) !== -1) {
                    sendToServer(eventName, eventData, eventId);
                }
            }
        };
        
        // Copia propriedades
        for (var prop in originalFbq) {
            if (originalFbq.hasOwnProperty(prop)) {
                window.fbq[prop] = originalFbq[prop];
            }
        }
        
        log('✅ fbq interceptado');
    }
    
    // ============================================
    // INICIALIZAÇÃO
    // ============================================
    function init() {
        log('🚀 SmartTracking v8 iniciando...');
        
        saveFbclid();
        
        var fbc = getFbc();
        var fbp = getFbp();
        var clientData = getAllClientData();
        
        log('FBC:', fbc ? '✅' : '❌');
        log('FBP:', fbp ? '✅' : '❌');
        log('Cliente:', clientData.name ? clientData.name : '❌ não encontrado');
        
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', function() {
                setTimeout(interceptFbq, 100);
            });
        } else {
            setTimeout(interceptFbq, 100);
        }
    }
    
    // ============================================
    // API PÚBLICA
    // ============================================
    window.SmartTracking = {
        getFbc: getFbc,
        getFbp: getFbp,
        getUserData: getUserData,
        getClientData: getAllClientData,
        sendEvent: sendToServer,
        getConfig: function() { return CONFIG; }
    };
    
    init();
    
})();