<?php
/**
 * ============================================
 * RELATÓRIO DE VENDAS POR CAMPANHA
 * ============================================
 * Agrupa vendas por UTM Source, Campaign, Medium, Content
 */

// Proteção básica (opcional - descomente para usar senha)
/*
$senha = 'carlinhos2024';
if (!isset($_GET['key']) || $_GET['key'] !== $senha) {
    http_response_code(403);
    die('Acesso negado. Use: ?key=' . $senha);
}
*/

// Configuração
$utmDir = __DIR__ . '/utm_data/';
$somentePagos = isset($_GET['pagos']) ? $_GET['pagos'] !== '0' : true;
$dataInicio = $_GET['inicio'] ?? null;
$dataFim = $_GET['fim'] ?? null;

// Lê todos os arquivos JSON
$arquivos = glob($utmDir . '*.json');
$transacoes = [];

foreach ($arquivos as $arquivo) {
    $conteudo = file_get_contents($arquivo);
    $dados = json_decode($conteudo, true);
    
    if (!$dados) continue;
    
    // Filtra por status de pagamento
    if ($somentePagos) {
        $status = strtolower($dados['webhook_status'] ?? '');
        if (!in_array($status, ['paid', 'pago', 'approved', 'confirmed', 'completed'])) {
            continue;
        }
    }
    
    // Filtra por data
    $dataCriacao = $dados['created_at'] ?? $dados['paid_at'] ?? null;
    if ($dataInicio && $dataCriacao && strtotime($dataCriacao) < strtotime($dataInicio)) continue;
    if ($dataFim && $dataCriacao && strtotime($dataCriacao) > strtotime($dataFim . ' 23:59:59')) continue;
    
    $transacoes[] = $dados;
}

// Agrupa por campanha
$porCampanha = [];
$porConjunto = [];
$porAnuncio = [];
$porPosicionamento = [];
$porFonte = [];
$totais = ['vendas' => 0, 'valor' => 0, 'cotas' => 0];

foreach ($transacoes as $t) {
    $source = $t['utm_source'] ?? 'Direto';
    $campaign = $t['utm_campaign'] ?? 'Sem campanha';
    $medium = $t['utm_medium'] ?? 'Sem conjunto';
    $content = $t['utm_content'] ?? 'Sem anúncio';
    $term = $t['utm_term'] ?? 'Sem posicionamento';
    $valor = ($t['valor_centavos'] ?? 0) / 100;
    $cotas = $t['cotas'] ?? 1;
    
    // Extrai nome e ID da campanha
    $campaignParts = explode('|', $campaign);
    $campaignNome = trim($campaignParts[0]);
    $campaignId = $campaignParts[1] ?? '';
    
    // Extrai nome e ID do conjunto
    $mediumParts = explode('|', $medium);
    $mediumNome = trim($mediumParts[0]);
    $mediumId = $mediumParts[1] ?? '';
    
    // Extrai nome e ID do anúncio
    $contentParts = explode('|', $content);
    $contentNome = trim($contentParts[0]);
    $contentId = $contentParts[1] ?? '';
    
    // Totais gerais
    $totais['vendas']++;
    $totais['valor'] += $valor;
    $totais['cotas'] += $cotas;
    
    // Por fonte
    if (!isset($porFonte[$source])) {
        $porFonte[$source] = ['vendas' => 0, 'valor' => 0, 'cotas' => 0];
    }
    $porFonte[$source]['vendas']++;
    $porFonte[$source]['valor'] += $valor;
    $porFonte[$source]['cotas'] += $cotas;
    
    // Por campanha
    if (!isset($porCampanha[$campaignNome])) {
        $porCampanha[$campaignNome] = ['id' => $campaignId, 'vendas' => 0, 'valor' => 0, 'cotas' => 0, 'conjuntos' => []];
    }
    $porCampanha[$campaignNome]['vendas']++;
    $porCampanha[$campaignNome]['valor'] += $valor;
    $porCampanha[$campaignNome]['cotas'] += $cotas;
    
    // Por conjunto (dentro da campanha)
    if (!isset($porCampanha[$campaignNome]['conjuntos'][$mediumNome])) {
        $porCampanha[$campaignNome]['conjuntos'][$mediumNome] = ['id' => $mediumId, 'vendas' => 0, 'valor' => 0, 'cotas' => 0, 'anuncios' => []];
    }
    $porCampanha[$campaignNome]['conjuntos'][$mediumNome]['vendas']++;
    $porCampanha[$campaignNome]['conjuntos'][$mediumNome]['valor'] += $valor;
    $porCampanha[$campaignNome]['conjuntos'][$mediumNome]['cotas'] += $cotas;
    
    // Por anúncio (dentro do conjunto)
    if (!isset($porCampanha[$campaignNome]['conjuntos'][$mediumNome]['anuncios'][$contentNome])) {
        $porCampanha[$campaignNome]['conjuntos'][$mediumNome]['anuncios'][$contentNome] = ['id' => $contentId, 'vendas' => 0, 'valor' => 0, 'cotas' => 0];
    }
    $porCampanha[$campaignNome]['conjuntos'][$mediumNome]['anuncios'][$contentNome]['vendas']++;
    $porCampanha[$campaignNome]['conjuntos'][$mediumNome]['anuncios'][$contentNome]['valor'] += $valor;
    $porCampanha[$campaignNome]['conjuntos'][$mediumNome]['anuncios'][$contentNome]['cotas'] += $cotas;
    
    // Por posicionamento
    if (!isset($porPosicionamento[$term])) {
        $porPosicionamento[$term] = ['vendas' => 0, 'valor' => 0, 'cotas' => 0];
    }
    $porPosicionamento[$term]['vendas']++;
    $porPosicionamento[$term]['valor'] += $valor;
    $porPosicionamento[$term]['cotas'] += $cotas;
}

// Ordena por valor (maior primeiro)
uasort($porCampanha, fn($a, $b) => $b['valor'] <=> $a['valor']);
uasort($porFonte, fn($a, $b) => $b['valor'] <=> $a['valor']);
uasort($porPosicionamento, fn($a, $b) => $b['valor'] <=> $a['valor']);

// Formata valor
function brl($v) {
    return 'R$ ' . number_format($v, 2, ',', '.');
}

// Verifica se é JSON
if (isset($_GET['json'])) {
    header('Content-Type: application/json');
    echo json_encode([
        'totais' => $totais,
        'por_fonte' => $porFonte,
        'por_campanha' => $porCampanha,
        'por_posicionamento' => $porPosicionamento,
        'transacoes' => count($transacoes)
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Relatório de Vendas por Campanha</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800&display=swap" rel="stylesheet">
    <style>
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { font-family: Inter, sans-serif; background: #f3f4f6; padding: 20px; color: #1f2937; }
        
        .container { max-width: 1200px; margin: 0 auto; }
        
        h1 { font-size: 24px; margin-bottom: 20px; display: flex; align-items: center; gap: 10px; }
        h2 { font-size: 18px; margin: 20px 0 12px; color: #374151; }
        h3 { font-size: 14px; margin: 12px 0 8px; color: #6b7280; }
        
        .filtros {
            background: #fff;
            padding: 16px;
            border-radius: 12px;
            margin-bottom: 20px;
            display: flex;
            gap: 16px;
            flex-wrap: wrap;
            align-items: center;
            border: 1px solid #e5e7eb;
        }
        .filtros label { font-size: 13px; font-weight: 600; color: #374151; }
        .filtros input[type="date"] {
            padding: 8px 12px;
            border: 1px solid #d1d5db;
            border-radius: 8px;
            font-size: 14px;
        }
        .filtros button {
            padding: 10px 20px;
            background: #2563eb;
            color: #fff;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
        }
        .filtros button:hover { background: #1d4ed8; }
        
        .totais {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 16px;
            margin-bottom: 24px;
        }
        .total-card {
            background: #fff;
            padding: 20px;
            border-radius: 12px;
            border: 1px solid #e5e7eb;
        }
        .total-card.destaque {
            background: linear-gradient(135deg, #059669, #10b981);
            color: #fff;
            border: none;
        }
        .total-card .label { font-size: 13px; opacity: 0.8; margin-bottom: 4px; }
        .total-card .valor { font-size: 28px; font-weight: 800; }
        .total-card.destaque .label { color: rgba(255,255,255,0.9); }
        
        .section {
            background: #fff;
            border-radius: 12px;
            border: 1px solid #e5e7eb;
            margin-bottom: 20px;
            overflow: hidden;
        }
        .section-header {
            padding: 16px 20px;
            background: #f9fafb;
            border-bottom: 1px solid #e5e7eb;
            font-weight: 700;
            font-size: 16px;
        }
        .section-body { padding: 0; }
        
        .campanha {
            border-bottom: 1px solid #e5e7eb;
        }
        .campanha:last-child { border-bottom: none; }
        
        .campanha-header {
            padding: 16px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            cursor: pointer;
            transition: background 0.2s;
        }
        .campanha-header:hover { background: #f9fafb; }
        
        .campanha-nome {
            font-weight: 700;
            font-size: 15px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        .campanha-nome .emoji { font-size: 18px; }
        .campanha-id { font-size: 11px; color: #9ca3af; font-weight: 400; }
        
        .campanha-stats {
            display: flex;
            gap: 20px;
            font-size: 13px;
        }
        .stat { display: flex; flex-direction: column; align-items: flex-end; }
        .stat-label { color: #6b7280; font-size: 11px; }
        .stat-valor { font-weight: 700; color: #059669; }
        .stat-vendas { font-weight: 700; color: #2563eb; }
        
        .campanha-detalhes {
            display: none;
            padding: 0 20px 16px 40px;
            background: #f9fafb;
        }
        .campanha.aberto .campanha-detalhes { display: block; }
        .campanha.aberto .campanha-header { background: #f9fafb; }
        
        .conjunto {
            padding: 12px 16px;
            background: #fff;
            border-radius: 8px;
            margin-bottom: 8px;
            border: 1px solid #e5e7eb;
        }
        .conjunto-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            cursor: pointer;
        }
        .conjunto-nome { font-weight: 600; font-size: 14px; }
        
        .anuncios {
            display: none;
            margin-top: 12px;
            padding-top: 12px;
            border-top: 1px solid #e5e7eb;
        }
        .conjunto.aberto .anuncios { display: block; }
        
        .anuncio {
            padding: 8px 12px;
            background: #f3f4f6;
            border-radius: 6px;
            margin-bottom: 6px;
            display: flex;
            justify-content: space-between;
            font-size: 13px;
        }
        .anuncio-nome { font-weight: 500; }
        
        .fonte-item, .posicionamento-item {
            padding: 12px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-bottom: 1px solid #f3f4f6;
        }
        .fonte-item:last-child, .posicionamento-item:last-child { border-bottom: none; }
        
        .badge {
            padding: 4px 10px;
            border-radius: 20px;
            font-size: 11px;
            font-weight: 700;
        }
        .badge-fb { background: #dbeafe; color: #1d4ed8; }
        .badge-google { background: #fef3c7; color: #92400e; }
        .badge-direto { background: #e5e7eb; color: #374151; }
        
        .toggle-icon {
            transition: transform 0.2s;
            font-size: 12px;
            color: #9ca3af;
        }
        .aberto .toggle-icon { transform: rotate(90deg); }
        
        @media (max-width: 640px) {
            .campanha-stats { flex-direction: column; gap: 8px; }
            .stat { align-items: flex-start; }
            .filtros { flex-direction: column; align-items: stretch; }
        }
    </style>
</head>
<body>

<div class="container">
    <h1>📊 Relatório de Vendas por Campanha</h1>
    
    <form class="filtros" method="GET">
        <div>
            <label>Data Início:</label><br>
            <input type="date" name="inicio" value="<?= htmlspecialchars($dataInicio ?? '') ?>">
        </div>
        <div>
            <label>Data Fim:</label><br>
            <input type="date" name="fim" value="<?= htmlspecialchars($dataFim ?? '') ?>">
        </div>
        <div>
            <label>Status:</label><br>
            <select name="pagos" style="padding: 8px 12px; border: 1px solid #d1d5db; border-radius: 8px;">
                <option value="1" <?= $somentePagos ? 'selected' : '' ?>>Somente pagos</option>
                <option value="0" <?= !$somentePagos ? 'selected' : '' ?>>Todos</option>
            </select>
        </div>
        <div style="margin-top: 18px;">
            <button type="submit">🔍 Filtrar</button>
        </div>
    </form>
    
    <div class="totais">
        <div class="total-card destaque">
            <div class="label">Faturamento Total</div>
            <div class="valor"><?= brl($totais['valor']) ?></div>
        </div>
        <div class="total-card">
            <div class="label">Total de Vendas</div>
            <div class="valor"><?= $totais['vendas'] ?></div>
        </div>
        <div class="total-card">
            <div class="label">Total de Cotas</div>
            <div class="valor"><?= $totais['cotas'] ?></div>
        </div>
        <div class="total-card">
            <div class="label">Ticket Médio</div>
            <div class="valor"><?= $totais['vendas'] > 0 ? brl($totais['valor'] / $totais['vendas']) : 'R$ 0,00' ?></div>
        </div>
    </div>
    
    <!-- Por Fonte -->
    <div class="section">
        <div class="section-header">📡 Por Fonte de Tráfego</div>
        <div class="section-body">
            <?php foreach ($porFonte as $fonte => $dados): ?>
            <div class="fonte-item">
                <div style="display: flex; align-items: center; gap: 10px;">
                    <span class="badge <?= strtoupper($fonte) === 'FB' ? 'badge-fb' : (strtoupper($fonte) === 'GOOGLE' ? 'badge-google' : 'badge-direto') ?>">
                        <?= htmlspecialchars($fonte) ?>
                    </span>
                </div>
                <div class="campanha-stats">
                    <div class="stat">
                        <span class="stat-label">Vendas</span>
                        <span class="stat-vendas"><?= $dados['vendas'] ?></span>
                    </div>
                    <div class="stat">
                        <span class="stat-label">Faturamento</span>
                        <span class="stat-valor"><?= brl($dados['valor']) ?></span>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
            <?php if (empty($porFonte)): ?>
            <div style="padding: 20px; text-align: center; color: #6b7280;">Nenhuma venda encontrada</div>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Por Campanha -->
    <div class="section">
        <div class="section-header">🎯 Por Campanha</div>
        <div class="section-body">
            <?php foreach ($porCampanha as $campanha => $dados): ?>
            <div class="campanha" onclick="this.classList.toggle('aberto')">
                <div class="campanha-header">
                    <div class="campanha-nome">
                        <span class="toggle-icon">▶</span>
                        <span class="emoji">📢</span>
                        <?= htmlspecialchars($campanha) ?>
                        <?php if ($dados['id']): ?>
                        <span class="campanha-id">ID: <?= htmlspecialchars($dados['id']) ?></span>
                        <?php endif; ?>
                    </div>
                    <div class="campanha-stats">
                        <div class="stat">
                            <span class="stat-label">Vendas</span>
                            <span class="stat-vendas"><?= $dados['vendas'] ?></span>
                        </div>
                        <div class="stat">
                            <span class="stat-label">Cotas</span>
                            <span><?= $dados['cotas'] ?></span>
                        </div>
                        <div class="stat">
                            <span class="stat-label">Faturamento</span>
                            <span class="stat-valor"><?= brl($dados['valor']) ?></span>
                        </div>
                    </div>
                </div>
                <div class="campanha-detalhes" onclick="event.stopPropagation()">
                    <?php foreach ($dados['conjuntos'] as $conjunto => $conjDados): ?>
                    <div class="conjunto" onclick="this.classList.toggle('aberto'); event.stopPropagation();">
                        <div class="conjunto-header">
                            <div class="conjunto-nome">
                                📦 <?= htmlspecialchars($conjunto) ?>
                                <?php if ($conjDados['id']): ?>
                                <span class="campanha-id">ID: <?= htmlspecialchars($conjDados['id']) ?></span>
                                <?php endif; ?>
                            </div>
                            <div class="campanha-stats">
                                <div class="stat">
                                    <span class="stat-vendas"><?= $conjDados['vendas'] ?> vendas</span>
                                </div>
                                <div class="stat">
                                    <span class="stat-valor"><?= brl($conjDados['valor']) ?></span>
                                </div>
                            </div>
                        </div>
                        <div class="anuncios">
                            <?php foreach ($conjDados['anuncios'] as $anuncio => $anDados): ?>
                            <div class="anuncio">
                                <div class="anuncio-nome">
                                    🎨 <?= htmlspecialchars($anuncio) ?>
                                    <?php if ($anDados['id']): ?>
                                    <span class="campanha-id" style="margin-left: 8px;">ID: <?= htmlspecialchars($anDados['id']) ?></span>
                                    <?php endif; ?>
                                </div>
                                <div>
                                    <span class="stat-vendas"><?= $anDados['vendas'] ?></span> vendas · 
                                    <span class="stat-valor"><?= brl($anDados['valor']) ?></span>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endforeach; ?>
            <?php if (empty($porCampanha)): ?>
            <div style="padding: 20px; text-align: center; color: #6b7280;">Nenhuma campanha encontrada</div>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Por Posicionamento -->
    <div class="section">
        <div class="section-header">📍 Por Posicionamento</div>
        <div class="section-body">
            <?php foreach ($porPosicionamento as $posicionamento => $dados): ?>
            <div class="posicionamento-item">
                <div>📱 <?= htmlspecialchars($posicionamento) ?></div>
                <div class="campanha-stats">
                    <div class="stat">
                        <span class="stat-label">Vendas</span>
                        <span class="stat-vendas"><?= $dados['vendas'] ?></span>
                    </div>
                    <div class="stat">
                        <span class="stat-label">Faturamento</span>
                        <span class="stat-valor"><?= brl($dados['valor']) ?></span>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
            <?php if (empty($porPosicionamento)): ?>
            <div style="padding: 20px; text-align: center; color: #6b7280;">Nenhum posicionamento encontrado</div>
            <?php endif; ?>
        </div>
    </div>
    
    <div style="text-align: center; padding: 20px; color: #9ca3af; font-size: 12px;">
        Atualizado em: <?= date('d/m/Y H:i:s') ?> · 
        <a href="?json=1<?= $dataInicio ? '&inicio=' . $dataInicio : '' ?><?= $dataFim ? '&fim=' . $dataFim : '' ?>" style="color: #2563eb;">Exportar JSON</a>
    </div>
</div>

</body>
</html>