<?php
/**
 * ============================================
 * FACEBOOK CAPI - QUALIDADE MÁXIMA v4.0
 * ============================================
 * 
 * PARÂMETROS ENVIADOS:
 * ✅ Email (em)
 * ✅ Telefone (ph)
 * ✅ Nome (fn)
 * ✅ Sobrenome (ln)
 * ✅ CPF/External ID (external_id)
 * ✅ Data de Nascimento (db) - NOVO!
 * ✅ Gênero (ge) - NOVO!
 * ✅ Cidade (ct) - NOVO!
 * ✅ Estado (st) - NOVO!
 * ✅ CEP (zp) - NOVO!
 * ✅ País (country)
 * ✅ IP (client_ip_address)
 * ✅ User Agent (client_user_agent)
 * ✅ FBC (fbc)
 * ✅ FBP (fbp)
 */

class FacebookCAPI {
    
    private $pixelId;
    private $accessToken;
    private $testEventCode;
    private $siteUrl;
    
    public function __construct() {
        $this->pixelId = defined('FB_PIXEL_ID') ? FB_PIXEL_ID : '2357252418054828';
        $this->accessToken = defined('FB_ACCESS_TOKEN') ? FB_ACCESS_TOKEN : 'EAAMchqNcUmMBQiBm6tuKZBZAkDYffLazN5u3wApky1afxti4SXtLDs5eW6vN6Pe6Jvss0KZAGvXiZCHeSc4Q2ZAqGm8aBwNbMVn7YWbcYuOOAHjvwe95DnKMq8lWawz2WwuOJcQ5Rrs259LdD6EZCZBKpNcyZBIDgqZCtT2R8Cd8ggpmdryqZBnFTGVpxSw2GmjQZDZD';
        $this->testEventCode = defined('FB_TEST_EVENT_CODE') ? FB_TEST_EVENT_CODE : '';
        $this->siteUrl = defined('SITE_URL') ? SITE_URL : $this->detectSiteUrl();
    }
    
    private function detectSiteUrl() {
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        return $protocol . '://' . $host;
    }
    
    /**
     * Anti-duplicação
     */
    private function isPurchaseAlreadySent($txId) {
        if (empty($txId)) return false;
        
        $cacheDir = __DIR__ . '/cache';
        if (!is_dir($cacheDir)) @mkdir($cacheDir, 0755, true);
        
        $cacheFile = $cacheDir . '/purchase_sent.json';
        $cache = file_exists($cacheFile) ? json_decode(file_get_contents($cacheFile), true) : [];
        
        if (isset($cache[$txId])) {
            $this->log("⚠️ Purchase já enviado para txId: $txId");
            return true;
        }
        return false;
    }
    
    private function markPurchaseSent($txId) {
        if (empty($txId)) return;
        
        $cacheDir = __DIR__ . '/cache';
        $cacheFile = $cacheDir . '/purchase_sent.json';
        
        $cache = file_exists($cacheFile) ? json_decode(file_get_contents($cacheFile), true) : [];
        $cache[$txId] = date('Y-m-d H:i:s');
        
        // Limpa cache > 24h
        $oneDayAgo = date('Y-m-d H:i:s', strtotime('-24 hours'));
        foreach ($cache as $id => $date) {
            if ($date < $oneDayAgo) unset($cache[$id]);
        }
        
        @file_put_contents($cacheFile, json_encode($cache, JSON_PRETTY_PRINT));
    }
    
    /**
     * ============================================
     * PREPARAÇÃO DE USER_DATA - QUALIDADE MÁXIMA
     * ============================================
     */
    private function prepareUserData($data) {
        $userData = [];
        $coverage = [];
        
        // ========== EMAIL ==========
        $email = $this->extractValue($data, ['email', 'customer.email']);
        if ($email && filter_var($email, FILTER_VALIDATE_EMAIL)) {
            // Não enviar emails falsos gerados
            if (strpos($email, '@pagamento.com') === false) {
                $emailNormalized = strtolower(trim($email));
                $userData['em'] = [hash('sha256', $emailNormalized)];
                $coverage['email'] = '✅';
            } else {
                $coverage['email'] = '❌ (gerado)';
            }
        } else {
            $coverage['email'] = '❌';
        }
        
        // ========== TELEFONE ==========
        $phone = $this->extractValue($data, ['phone', 'telefone', 'celular', 'customer.phone']);
        if ($phone) {
            $phoneClean = preg_replace('/\D/', '', $phone);
            // Não enviar telefone padrão
            if ($phoneClean !== '11999999999' && strlen($phoneClean) >= 10) {
                if (strlen($phoneClean) <= 11 && substr($phoneClean, 0, 2) !== '55') {
                    $phoneClean = '55' . $phoneClean;
                }
                $userData['ph'] = [hash('sha256', $phoneClean)];
                $coverage['phone'] = '✅';
            } else {
                $coverage['phone'] = '❌ (padrão)';
            }
        } else {
            $coverage['phone'] = '❌';
        }
        
        // ========== NOME E SOBRENOME ==========
        $firstName = $this->extractValue($data, ['first_name', 'firstName', 'customer.first_name']);
        $lastName = $this->extractValue($data, ['last_name', 'lastName', 'customer.last_name']);
        
        if (!$firstName) {
            $fullName = $this->extractValue($data, ['name', 'nome', 'customer.name']);
            if ($fullName && $fullName !== 'Cliente') {
                $fullName = trim($fullName);
                $fullName = preg_replace('/^(sr\.?|sra\.?|dr\.?|dra\.?)\s+/i', '', $fullName);
                $nameParts = array_values(array_filter(preg_split('/\s+/', $fullName)));
                
                if (count($nameParts) >= 1) $firstName = $nameParts[0];
                if (count($nameParts) >= 2) $lastName = end($nameParts);
            }
        }
        
        if ($firstName && strlen($firstName) >= 2) {
            $fnNormalized = $this->normalizeForHash($firstName);
            if ($fnNormalized) {
                $userData['fn'] = [hash('sha256', $fnNormalized)];
                $coverage['first_name'] = '✅ ' . $firstName;
            }
        } else {
            $coverage['first_name'] = '❌';
        }
        
        if ($lastName && strlen($lastName) >= 2) {
            $lnNormalized = $this->normalizeForHash($lastName);
            if ($lnNormalized) {
                $userData['ln'] = [hash('sha256', $lnNormalized)];
                $coverage['last_name'] = '✅ ' . $lastName;
            }
        } else {
            $coverage['last_name'] = '❌';
        }
        
        // ========== CPF (external_id) ==========
        $cpf = $this->extractValue($data, ['document', 'cpf', 'customer.document', 'customer.cpf']);
        if ($cpf) {
            $cpfClean = preg_replace('/\D/', '', $cpf);
            if ($cpfClean !== '00000000000' && strlen($cpfClean) === 11) {
                $userData['external_id'] = [hash('sha256', $cpfClean)];
                $coverage['cpf'] = '✅';
            } else {
                $coverage['cpf'] = '❌ (padrão)';
            }
        } else {
            $coverage['cpf'] = '❌';
        }
        
        // ========== DATA DE NASCIMENTO (db) - NOVO! ==========
        $dob = $this->extractValue($data, [
            'birth_date', 'nascimento', 'data_nascimento', 'birthdate',
            'customer.birth_date', 'customer.nascimento'
        ]);
        if ($dob) {
            // Formato esperado: YYYYMMDD
            $dobClean = preg_replace('/\D/', '', $dob);
            if (strlen($dobClean) === 8) {
                // Verifica se é YYYYMMDD ou DDMMYYYY
                $year = substr($dobClean, 0, 4);
                if ((int)$year < 1900 || (int)$year > 2100) {
                    // É DDMMYYYY, converte
                    $dobClean = substr($dobClean, 4, 4) . substr($dobClean, 2, 2) . substr($dobClean, 0, 2);
                }
                $userData['db'] = [hash('sha256', $dobClean)];
                $coverage['birth_date'] = '✅ ' . $dobClean;
            } else {
                $coverage['birth_date'] = '❌ (formato inválido: ' . $dob . ')';
            }
        } else {
            $coverage['birth_date'] = '❌';
        }
        
        // ========== GÊNERO (ge) - NOVO! ==========
        $gender = $this->extractValue($data, [
            'gender', 'genero', 'sexo',
            'customer.gender', 'customer.sexo', 'customer.genero'
        ]);
        if ($gender) {
            $g = strtolower(substr(trim($gender), 0, 1));
            if ($g === 'm' || $g === 'f') {
                $userData['ge'] = [hash('sha256', $g)];
                $coverage['gender'] = '✅ ' . $g;
            } else {
                $coverage['gender'] = '❌ (valor inválido: ' . $gender . ')';
            }
        } else {
            $coverage['gender'] = '❌';
        }
        
        // ========== CIDADE (ct) - NOVO! ==========
        $city = $this->extractValue($data, [
            'city', 'cidade',
            'customer.city', 'customer.cidade'
        ]);
        if ($city) {
            $cityNormalized = $this->normalizeForHash($city);
            // Remove espaços (Facebook requirement)
            $cityNormalized = str_replace(' ', '', $cityNormalized);
            if ($cityNormalized && strlen($cityNormalized) >= 2) {
                $userData['ct'] = [hash('sha256', $cityNormalized)];
                $coverage['city'] = '✅ ' . $cityNormalized;
            } else {
                $coverage['city'] = '❌ (normalização falhou)';
            }
        } else {
            $coverage['city'] = '❌';
        }
        
        // ========== ESTADO (st) - NOVO! ==========
        $state = $this->extractValue($data, [
            'state', 'estado', 'uf',
            'customer.state', 'customer.estado', 'customer.uf'
        ]);
        if ($state) {
            $stateNormalized = strtolower(trim($state));
            // Aceita apenas 2 caracteres
            if (strlen($stateNormalized) === 2) {
                $userData['st'] = [hash('sha256', $stateNormalized)];
                $coverage['state'] = '✅ ' . $stateNormalized;
            } else {
                $coverage['state'] = '❌ (não é sigla: ' . $state . ')';
            }
        } else {
            $coverage['state'] = '❌';
        }
        
        // ========== CEP (zp) - NOVO! ==========
        $zip = $this->extractValue($data, [
            'zip', 'cep', 'postal_code',
            'customer.zip', 'customer.cep'
        ]);
        if ($zip) {
            $zipClean = preg_replace('/\D/', '', $zip);
            // Facebook aceita 5 primeiros dígitos para BR
            if (strlen($zipClean) >= 5) {
                $zipFinal = substr($zipClean, 0, 5);
                $userData['zp'] = [hash('sha256', $zipFinal)];
                $coverage['zip'] = '✅ ' . $zipFinal;
            } else {
                $coverage['zip'] = '❌ (muito curto)';
            }
        } else {
            $coverage['zip'] = '❌';
        }
        
        // ========== PAÍS (sempre BR) ==========
        $userData['country'] = [hash('sha256', 'br')];
        $coverage['country'] = '✅ br';
        
        // ========== IP ==========
        $ip = $this->extractValue($data, ['ip', 'customer_ip', 'client_ip']);
        if (!$ip) {
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['HTTP_X_REAL_IP'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
        }
        if ($ip) {
            if (strpos($ip, ',') !== false) $ip = trim(explode(',', $ip)[0]);
            $userData['client_ip_address'] = $ip;
            $coverage['ip'] = '✅';
        } else {
            $coverage['ip'] = '❌';
        }
        
        // ========== USER AGENT ==========
        $ua = $data['user_agent'] ?? $_SERVER['HTTP_USER_AGENT'] ?? '';
        if ($ua) {
            $userData['client_user_agent'] = $ua;
            $coverage['user_agent'] = '✅';
        } else {
            $coverage['user_agent'] = '❌';
        }
        
        // ========== FBC ==========
        $fbc = $data['fbc'] ?? null;
        if ($fbc && strpos($fbc, 'fb.') === 0) {
            $userData['fbc'] = $fbc;
            $coverage['fbc'] = '✅';
        } else {
            $coverage['fbc'] = '❌' . ($fbc ? ' (formato inválido)' : '');
        }
        
        // ========== FBP ==========
        $fbp = $data['fbp'] ?? null;
        if ($fbp && strpos($fbp, 'fb.') === 0) {
            $userData['fbp'] = $fbp;
            $coverage['fbp'] = '✅';
        } else {
            $coverage['fbp'] = '❌' . ($fbp ? ' (formato inválido)' : '');
        }
        
        // Log detalhado de cobertura
        $this->log("📊 COBERTURA DE DADOS", $coverage);
        
        // Conta parâmetros
        $totalParams = 0;
        $hashedParams = ['em', 'ph', 'fn', 'ln', 'external_id', 'db', 'ge', 'ct', 'st', 'zp', 'country'];
        foreach ($hashedParams as $p) {
            if (isset($userData[$p])) $totalParams++;
        }
        if (isset($userData['fbc'])) $totalParams++;
        if (isset($userData['fbp'])) $totalParams++;
        
        $this->log("📈 Total de parâmetros enviados: $totalParams/13");
        
        return $userData;
    }
    
    /**
     * Extrai valor de múltiplas fontes
     */
    private function extractValue($data, $keys) {
        foreach ($keys as $key) {
            if (strpos($key, '.') !== false) {
                $parts = explode('.', $key);
                $value = $data;
                foreach ($parts as $part) {
                    if (is_array($value) && isset($value[$part])) {
                        $value = $value[$part];
                    } else {
                        $value = null;
                        break;
                    }
                }
                if ($value !== null && $value !== '') return $value;
            } else {
                if (isset($data[$key]) && $data[$key] !== null && $data[$key] !== '') {
                    return $data[$key];
                }
            }
        }
        return null;
    }
    
    /**
     * Normaliza para hash (lowercase, sem acentos)
     */
    private function normalizeForHash($str) {
        if (!$str) return null;
        $str = trim(strtolower($str));
        $str = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $str);
        $str = preg_replace('/[^a-z\s]/', '', $str);
        $str = preg_replace('/\s+/', ' ', $str);
        return trim($str) ?: null;
    }
    
    /**
     * ============================================
     * ENVIA PURCHASE
     * ============================================
     */
    public function sendPurchase($txId, $valorReais, $utmData = []) {
        $this->log("========== PURCHASE ==========", ['txId' => $txId, 'valor' => $valorReais]);
        
        if ($this->isPurchaseAlreadySent($txId)) {
            return ['success' => true, 'skipped' => true, 'reason' => 'already_sent', 'txId' => $txId];
        }
        
        $userData = $this->prepareUserData($utmData);
        $eventId = 'purchase_' . $txId . '_' . time();
        $sourceUrl = $utmData['source_url'] ?? $this->siteUrl;
        
        $eventData = [
            'event_name' => 'Purchase',
            'event_time' => time(),
            'event_id' => $eventId,
            'event_source_url' => $sourceUrl,
            'action_source' => 'website',
            'user_data' => $userData,
            'custom_data' => [
                'currency' => 'BRL',
                'value' => floatval($valorReais),
                'content_type' => 'product',
                'content_name' => $utmData['produto'] ?? (defined('PRODUTO_NOME') ? PRODUTO_NOME : 'Produto'),
                'content_ids' => [$utmData['produto_id'] ?? (defined('PRODUTO_ID') ? PRODUTO_ID : 'produto-001')],
                'num_items' => intval($utmData['cotas'] ?? 1),
                'order_id' => $txId
            ]
        ];
        
        $payload = ['data' => [$eventData]];
        if (!empty($this->testEventCode)) {
            $payload['test_event_code'] = $this->testEventCode;
        }
        
        $apiVersion = defined('FB_API_VERSION') ? FB_API_VERSION : 'v18.0';
        $url = "https://graph.facebook.com/{$apiVersion}/{$this->pixelId}/events?access_token={$this->accessToken}";
        
        $this->log("📤 Enviando Purchase", [
            'event_id' => $eventId,
            'value' => $valorReais,
            'content_id' => $eventData['custom_data']['content_ids'][0]
        ]);
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_SSL_VERIFYPEER => true
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        $responseData = json_decode($response, true);
        $success = $httpCode >= 200 && $httpCode < 300;
        
        if ($success) $this->markPurchaseSent($txId);
        
        $this->log("📥 Resposta Facebook", [
            'success' => $success ? '✅' : '❌',
            'http_code' => $httpCode,
            'events_received' => $responseData['events_received'] ?? 0,
            'messages' => $responseData['messages'] ?? [],
            'error' => $responseData['error']['message'] ?? $curlError ?: null
        ]);
        
        return [
            'success' => $success,
            'event_id' => $eventId,
            'http_code' => $httpCode,
            'events_received' => $responseData['events_received'] ?? 0,
            'response' => $responseData
        ];
    }
    
    /**
     * Eventos genéricos
     */
    public function pageView($data, $sourceUrl = '', $eventId = null) {
        return $this->sendEvent('PageView', $data, 0, $sourceUrl, $eventId);
    }
    
    public function viewContent($data, $value = 0, $sourceUrl = '', $eventId = null) {
        return $this->sendEvent('ViewContent', $data, $value, $sourceUrl, $eventId);
    }
    
    public function initiateCheckout($data, $value, $sourceUrl = '', $eventId = null) {
        return $this->sendEvent('InitiateCheckout', $data, $value, $sourceUrl, $eventId);
    }
    
    private function sendEvent($eventName, $data, $value, $sourceUrl, $eventId = null) {
        $userData = $this->prepareUserData($data);
        
        if (!$eventId) $eventId = strtolower($eventName) . '_' . bin2hex(random_bytes(8)) . '_' . time();
        
        $eventData = [
            'event_name' => $eventName,
            'event_time' => time(),
            'event_id' => $eventId,
            'event_source_url' => $sourceUrl ?: $this->siteUrl,
            'action_source' => 'website',
            'user_data' => $userData,
            'custom_data' => [
                'currency' => 'BRL',
                'value' => floatval($value),
                'content_type' => 'product',
                'content_name' => $data['content_name'] ?? (defined('PRODUTO_NOME') ? PRODUTO_NOME : 'Produto'),
                'content_ids' => [$data['content_id'] ?? (defined('PRODUTO_ID') ? PRODUTO_ID : 'produto-001')]
            ]
        ];
        
        $payload = ['data' => [$eventData]];
        if (!empty($this->testEventCode)) $payload['test_event_code'] = $this->testEventCode;
        
        $apiVersion = defined('FB_API_VERSION') ? FB_API_VERSION : 'v18.0';
        $url = "https://graph.facebook.com/{$apiVersion}/{$this->pixelId}/events?access_token={$this->accessToken}";
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        $this->log("$eventName enviado", ['event_id' => $eventId, 'http_code' => $httpCode]);
        
        return [
            'success' => $httpCode >= 200 && $httpCode < 300,
            'event_id' => $eventId,
            'http_code' => $httpCode,
            'response' => json_decode($response, true)
        ];
    }
    
    private function log($message, $data = []) {
        $logDir = __DIR__ . '/logs';
        if (!is_dir($logDir)) @mkdir($logDir, 0755, true);
        
        $timestamp = date('Y-m-d H:i:s');
        $logLine = "[$timestamp] $message";
        if (!empty($data)) {
            $logLine .= ' | ' . json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        }
        
        @file_put_contents($logDir . '/facebook_capi.log', $logLine . "\n", FILE_APPEND | LOCK_EX);
    }
}