<?php
/**
 * Sistema de Proteção v1.0
 *
 * Arquivo principal que combina todas as proteções.
 * Inclua este arquivo nas páginas que deseja proteger.
 *
 * USO BÁSICO:
 *   require_once __DIR__ . '/protetor/protetor.php';
 *   protetor_verificar();      // No início da página (antes do HTML)
 *   echo protetor_head();      // Dentro do <head>
 *   echo protetor_body();      // No final do <body>
 *
 * USO COM CACHE (para páginas internas):
 *   protetor_verificar_com_cache();
 */

// ============================================
// TIMEZONE: Horário de São Paulo
// ============================================
date_default_timezone_set('America/Sao_Paulo');

// Verificar se está instalado
$dbConfigPath = __DIR__ . '/db_config.php';
if (!file_exists($dbConfigPath)) {
    // Se não instalado, apenas carregar config padrão
    require_once __DIR__ . '/config.php';
} else {
    require_once $dbConfigPath;
}

// Carregar módulos
require_once __DIR__ . '/pclid.php';
require_once __DIR__ . '/filtro_referer.php';
require_once __DIR__ . '/filtro_device.php';
require_once __DIR__ . '/anti_clone.php';

// Iniciar sessão se necessário
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Lista de parâmetros UTM para preservar (pclid NÃO é salvo por segurança)
define('PROTETOR_UTM_PARAMS', [
    'utm_source', 'utm_medium', 'utm_campaign', 'utm_term', 'utm_content',
    'fbclid', 'gclid', 'ref', 'source'
    // pclid - NÃO incluir (token secreto, validado apenas na entrada)
]);

/**
 * Obtém o IP real do visitante (suporta proxy/Cloudflare)
 */
function protetor_get_ip() {
    // Cloudflare
    if (!empty($_SERVER['HTTP_CF_CONNECTING_IP'])) {
        return $_SERVER['HTTP_CF_CONNECTING_IP'];
    }
    // Proxy padrão
    if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
        return trim($ips[0]);
    }
    // Nginx proxy
    if (!empty($_SERVER['HTTP_X_REAL_IP'])) {
        return $_SERVER['HTTP_X_REAL_IP'];
    }
    // Direto
    return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
}

/**
 * Carrega a biblioteca GeoIP2 (lazy loading)
 */
function protetor_geoip_init() {
    static $loaded = false;
    if (!$loaded) {
        $pharPath = __DIR__ . '/geoip2.phar';
        if (file_exists($pharPath)) {
            require_once $pharPath;
            $loaded = true;
        }
    }
    return $loaded;
}

/**
 * Consulta uma API de GeoIP com tratamento de erro
 * Retorna o código do país (2 letras) ou null se falhar
 */
function protetor_geoip_api_call($url, $parser, $timeout = 3) {
    try {
        $context = stream_context_create([
            'http' => [
                'timeout' => $timeout,
                'ignore_errors' => true,
                'header' => "User-Agent: Mozilla/5.0\r\n"
            ]
        ]);

        $response = @file_get_contents($url, false, $context);

        if ($response === false || empty($response)) {
            return null;
        }

        return $parser($response);
    } catch (Exception $e) {
        return null;
    }
}

/**
 * Consulta múltiplas APIs de GeoIP com sistema de fallback
 * Cada API funciona como backup da anterior - nunca dá erro
 */
function protetor_get_country_from_apis($ip) {
    // Cache em sessão para evitar múltiplas consultas
    $cacheKey = 'geo_country_' . md5($ip);
    if (isset($_SESSION[$cacheKey])) {
        return $_SESSION[$cacheKey];
    }

    $resultados = [];
    $timeout = 3;

    // ========================================
    // APIS GRATUITAS E ILIMITADAS (prioridade)
    // ========================================

    // API 1: iplocation.net (GRATUITA, ILIMITADA, mais confiável)
    // Response codes: 200=OK, 400=Bad Request, 404=Not Found
    $result = protetor_geoip_api_call(
        "https://api.iplocation.net/?ip={$ip}",
        function($response) {
            $data = @json_decode($response, true);
            if ($data && isset($data['response_code']) && $data['response_code'] == 200) {
                $code = $data['country_code2'] ?? $data['country_code'] ?? null;
                if ($code && strlen($code) === 2) {
                    return strtoupper($code);
                }
            }
            return null;
        },
        $timeout
    );
    if ($result) $resultados['iplocation'] = $result;

    // API 2: ipwho.is (GRATUITA, ILIMITADA)
    $result = protetor_geoip_api_call(
        "https://ipwho.is/{$ip}",
        function($response) {
            $data = @json_decode($response, true);
            if ($data && isset($data['success']) && $data['success'] === true) {
                $code = $data['country_code'] ?? null;
                if ($code && strlen($code) === 2) {
                    return strtoupper($code);
                }
            }
            return null;
        },
        $timeout
    );
    if ($result) $resultados['ipwhois'] = $result;

    // ========================================
    // APIS COM LIMITE (fallback)
    // ========================================

    // API 3: ip-api.com (LIMITE: 45 req/min)
    $result = protetor_geoip_api_call(
        "http://ip-api.com/json/{$ip}?fields=status,countryCode",
        function($response) {
            $data = @json_decode($response, true);
            if ($data && isset($data['status']) && $data['status'] === 'success') {
                $code = $data['countryCode'] ?? null;
                if ($code && strlen($code) === 2) {
                    return strtoupper($code);
                }
            }
            return null;
        },
        $timeout
    );
    if ($result) $resultados['ipapi'] = $result;

    // API 4: ipapi.co (LIMITE: 1000/dia)
    $result = protetor_geoip_api_call(
        "https://ipapi.co/{$ip}/country/",
        function($response) {
            $response = trim($response);
            // Verifica se é um código de país válido (2 letras, sem erro)
            if (strlen($response) === 2 && ctype_alpha($response)) {
                return strtoupper($response);
            }
            return null;
        },
        $timeout
    );
    if ($result) $resultados['ipapico'] = $result;

    // ========================================
    // DECISÃO: Sistema de votação com fallback
    // ========================================

    // Se nenhuma API respondeu, retorna null (vai usar GeoLite2 local)
    if (empty($resultados)) {
        error_log("GeoIP: Nenhuma API respondeu para {$ip}");
        return null;
    }

    // Contar votos por país
    $votos = array_count_values($resultados);
    arsort($votos);

    // Pegar o país com mais votos
    $paisMaisVotado = array_key_first($votos);
    $qtdVotos = $votos[$paisMaisVotado];
    $totalApis = count($resultados);

    // Log para debug
    error_log("GeoIP para {$ip}: " . json_encode($resultados) . " => {$paisMaisVotado} ({$qtdVotos}/{$totalApis} votos)");

    // REGRA DE DECISÃO:
    // - Se maioria concorda, usa o resultado da maioria
    // - Se empate, prioriza iplocation.net (mais confiável para localização real)
    // - Se só 1 API respondeu, usa esse resultado

    if ($qtdVotos > 1 || $totalApis === 1) {
        // Maioria ou única resposta
        $_SESSION[$cacheKey] = $paisMaisVotado;
        return $paisMaisVotado;
    }

    // Empate: priorizar iplocation.net se disponível
    if (isset($resultados['iplocation'])) {
        $_SESSION[$cacheKey] = $resultados['iplocation'];
        return $resultados['iplocation'];
    }

    // Senão, usar o primeiro resultado disponível
    $_SESSION[$cacheKey] = $paisMaisVotado;
    return $paisMaisVotado;
}

/**
 * Obtém o país do IP (código ISO, ex: BR, US, PT)
 * Usa múltiplas APIs para garantir precisão e evitar falsos positivos
 */
function protetor_get_country($ip = null) {
    if ($ip === null) {
        $ip = protetor_get_ip();
    }

    // Para localhost, retorna BR para debug
    if ($ip === '::1' || $ip === '127.0.0.1') {
        return 'BR';
    }

    // IPs privados - permitir
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
        return 'BR'; // IP privado, assumir BR
    }

    // PRIMEIRO: Tentar APIs externas (mais confiáveis para localização real)
    $paisApis = protetor_get_country_from_apis($ip);
    if ($paisApis !== null) {
        return $paisApis;
    }

    // FALLBACK: Usar banco de dados local GeoLite2
    if (!protetor_geoip_init()) {
        return 'Unknown';
    }

    $dbPath = __DIR__ . '/GeoLite2-Country.mmdb';
    if (!file_exists($dbPath)) {
        return 'Unknown';
    }

    try {
        $reader = new \GeoIp2\Database\Reader($dbPath);
        $record = $reader->country($ip);
        return $record->country->isoCode ?? 'Unknown';
    } catch (Exception $e) {
        return 'Unknown';
    }
}

/**
 * Obtém o ISP/ASN do IP (para detectar datacenters)
 */
function protetor_get_isp($ip = null) {
    if (!protetor_geoip_init()) {
        return 'Unknown';
    }

    if ($ip === null) {
        $ip = protetor_get_ip();
    }

    // Para localhost, retorna teste
    if ($ip === '::1' || $ip === '127.0.0.1') {
        return 'Localhost';
    }

    $dbPath = __DIR__ . '/GeoLite2-ASN.mmdb';
    if (!file_exists($dbPath)) {
        return 'Unknown';
    }

    try {
        $reader = new \GeoIp2\Database\Reader($dbPath);
        $record = $reader->asn($ip);
        return $record->autonomousSystemOrganization ?? 'Unknown';
    } catch (Exception $e) {
        return 'Unknown';
    }
}

/**
 * Verifica se o país está na lista de países permitidos
 */
function protetor_pais_permitido($countryCode) {
    $paisesPermitidos = protetor_get_config('paises_permitidos', ['BR']);

    if (empty($paisesPermitidos)) {
        return true; // Se não configurado, permite todos
    }

    return in_array(strtoupper($countryCode), array_map('strtoupper', $paisesPermitidos));
}

/**
 * Verifica se o ISP é um datacenter conhecido (VPN/Proxy/Hosting)
 */
function protetor_is_datacenter($isp) {
    $datacenters = protetor_get_config('isps_bloqueados', [
        'Amazon', 'AWS', 'Google Cloud', 'Microsoft Azure', 'DigitalOcean',
        'Linode', 'Vultr', 'OVH', 'Hetzner', 'Contabo', 'Hostinger',
        'Cloudflare', 'Akamai', 'Fastly', 'NordVPN', 'ExpressVPN',
        'Surfshark', 'ProtonVPN', 'CyberGhost', 'Private Internet Access',
        'Mullvad', 'TunnelBear', 'Windscribe', 'HideMyAss', 'IPVanish',
        'VyprVPN', 'PureVPN', 'TorGuard', 'Data Center', 'Datacenter',
        'Hosting', 'Server', 'Cloud', 'VPS', 'Dedicated'
    ]);

    $ispLower = strtolower($isp);

    foreach ($datacenters as $dc) {
        if (stripos($ispLower, strtolower($dc)) !== false) {
            return true;
        }
    }

    return false;
}

/**
 * Salva UTMs na sessão (chamado automaticamente)
 */
function protetor_salvar_utms() {
    if (!isset($_SESSION['protetor_utms'])) {
        $_SESSION['protetor_utms'] = [];
    }

    // Capturar UTMs da URL e mesclar com existentes na sessão
    foreach (PROTETOR_UTM_PARAMS as $param) {
        if (isset($_GET[$param]) && !empty($_GET[$param])) {
            $_SESSION['protetor_utms'][$param] = $_GET[$param];
        }
    }

    // Salvar timestamp da primeira captura
    if (!isset($_SESSION['protetor_utms_timestamp']) && !empty($_SESSION['protetor_utms'])) {
        $_SESSION['protetor_utms_timestamp'] = time();
    }

    return $_SESSION['protetor_utms'];
}

/**
 * Recupera UTMs da sessão
 */
function protetor_get_utms() {
    protetor_salvar_utms(); // Garantir que sempre salvamos antes de recuperar

    return $_SESSION['protetor_utms'] ?? [];
}

/**
 * Retorna query string com UTMs para adicionar em links
 */
function protetor_utm_query() {
    $utms = protetor_get_utms();

    if (empty($utms)) {
        return '';
    }

    return http_build_query($utms);
}

/**
 * Adiciona UTMs a uma URL
 */
function protetor_url_com_utms($url) {
    $query = protetor_utm_query();

    if (empty($query)) {
        return $url;
    }

    $separator = (strpos($url, '?') === false) ? '?' : '&';
    return $url . $separator . $query;
}

// Executar salvamento de UTMs automaticamente
protetor_salvar_utms();

/**
 * Obtém conexão com banco de dados
 */
function protetor_get_db() {
    static $pdo = null;

    if ($pdo === null && defined('PROTETOR_DB_HOST')) {
        try {
            $pdo = new PDO(
                "mysql:host=" . PROTETOR_DB_HOST . ";dbname=" . PROTETOR_DB_NAME . ";charset=utf8mb4",
                PROTETOR_DB_USER,
                PROTETOR_DB_PASS,
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
        } catch (PDOException $e) {
            error_log("Protetor DB Error: " . $e->getMessage());
            return null;
        }
    }

    return $pdo;
}

/**
 * Obtém uma configuração do banco
 */
function protetor_get_config($chave, $default = null) {
    $pdo = protetor_get_db();

    if (!$pdo || !defined('PROTETOR_DB_PREFIX')) {
        // Fallback para constantes definidas
        $const = 'PROTETOR_' . strtoupper($chave);
        return defined($const) ? constant($const) : $default;
    }

    $prefix = PROTETOR_DB_PREFIX;

    try {
        $stmt = $pdo->prepare("SELECT valor, tipo FROM {$prefix}config WHERE chave = ?");
        $stmt->execute([$chave]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) {
            return $default;
        }

        switch ($row['tipo']) {
            case 'bool':
                return $row['valor'] === '1';
            case 'int':
                return (int) $row['valor'];
            case 'json':
                return json_decode($row['valor'], true);
            default:
                return $row['valor'];
        }
    } catch (PDOException $e) {
        return $default;
    }
}

/**
 * Registra um bloqueio no banco
 */
function protetor_registrar_bloqueio($motivo, $dados = []) {
    $pdo = protetor_get_db();

    if (!$pdo || !protetor_get_config('salvar_bloqueios', true)) {
        return;
    }

    $prefix = PROTETOR_DB_PREFIX;

    try {
        $stmt = $pdo->prepare("INSERT INTO {$prefix}bloqueios (ip, motivo, user_agent, referer, url_acessada, dados_request) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            protetor_get_ip(),
            $motivo,
            $_SERVER['HTTP_USER_AGENT'] ?? null,
            $_SERVER['HTTP_REFERER'] ?? null,
            $_SERVER['REQUEST_URI'] ?? null,
            json_encode($dados)
        ]);
    } catch (PDOException $e) {
        error_log("Protetor Log Error: " . $e->getMessage());
    }
}

/**
 * Registra um visitante válido
 */
function protetor_registrar_visitante() {
    $pdo = protetor_get_db();

    if (!$pdo || !protetor_get_config('salvar_visitantes', true)) {
        return;
    }

    $prefix = PROTETOR_DB_PREFIX;
    $sessionId = session_id();

    try {
        // Verificar se já existe
        $stmt = $pdo->prepare("SELECT id FROM {$prefix}visitantes WHERE session_id = ?");
        $stmt->execute([$sessionId]);

        if ($stmt->fetch()) {
            // Atualizar visita
            $stmt = $pdo->prepare("UPDATE {$prefix}visitantes SET ultima_visita = NOW(), paginas_visitadas = paginas_visitadas + 1 WHERE session_id = ?");
            $stmt->execute([$sessionId]);
        } else {
            // Novo visitante
            $device = device_detectar();
            $stmt = $pdo->prepare("INSERT INTO {$prefix}visitantes (session_id, ip, pclid, fbclid, utm_source, utm_medium, utm_campaign, referer, user_agent, dispositivo) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $sessionId,
                protetor_get_ip(),
                $_GET['pclid'] ?? null,
                $_GET['fbclid'] ?? null,
                $_GET['utm_source'] ?? null,
                $_GET['utm_medium'] ?? null,
                $_GET['utm_campaign'] ?? null,
                $_SERVER['HTTP_REFERER'] ?? null,
                $_SERVER['HTTP_USER_AGENT'] ?? null,
                $device
            ]);
        }
    } catch (PDOException $e) {
        error_log("Protetor Visitante Error: " . $e->getMessage());
    }
}

/**
 * Verifica se IP está na whitelist
 */
function protetor_ip_na_whitelist($ip) {
    $pdo = protetor_get_db();
    if (!$pdo) return false;

    $prefix = PROTETOR_DB_PREFIX;

    try {
        // Verificar IP exato
        $stmt = $pdo->prepare("SELECT id FROM {$prefix}ip_whitelist WHERE ip = ? AND tipo = 'single'");
        $stmt->execute([$ip]);
        if ($stmt->fetch()) return true;

        // Verificar ranges CIDR
        $stmt = $pdo->query("SELECT ip FROM {$prefix}ip_whitelist WHERE tipo = 'cidr'");
        while ($row = $stmt->fetch()) {
            if (protetor_ip_in_cidr($ip, $row['ip'])) {
                return true;
            }
        }
    } catch (PDOException $e) {
        error_log("Protetor Whitelist Error: " . $e->getMessage());
    }

    return false;
}

/**
 * Verifica se IP está na blacklist
 */
function protetor_ip_na_blacklist($ip) {
    $pdo = protetor_get_db();
    if (!$pdo) return false;

    $prefix = PROTETOR_DB_PREFIX;

    try {
        // Verificar IP exato
        $stmt = $pdo->prepare("SELECT id, motivo FROM {$prefix}ip_blacklist WHERE ip = ? AND tipo = 'single'");
        $stmt->execute([$ip]);
        if ($row = $stmt->fetch()) {
            return $row['motivo'] ?: 'ip_bloqueado';
        }

        // Verificar ranges CIDR
        $stmt = $pdo->query("SELECT ip, motivo FROM {$prefix}ip_blacklist WHERE tipo = 'cidr'");
        while ($row = $stmt->fetch()) {
            if (protetor_ip_in_cidr($ip, $row['ip'])) {
                return $row['motivo'] ?: 'ip_range_bloqueado';
            }
        }
    } catch (PDOException $e) {
        error_log("Protetor Blacklist Error: " . $e->getMessage());
    }

    return false;
}

/**
 * Verifica se um IP está dentro de um range CIDR
 */
function protetor_ip_in_cidr($ip, $cidr) {
    if (strpos($cidr, '/') === false) {
        return $ip === $cidr;
    }

    list($subnet, $bits) = explode('/', $cidr);
    $ip_long = ip2long($ip);
    $subnet_long = ip2long($subnet);

    if ($ip_long === false || $subnet_long === false) {
        return false;
    }

    $mask = -1 << (32 - (int)$bits);
    $subnet_long &= $mask;

    return ($ip_long & $mask) === $subnet_long;
}

/**
 * Verifica todas as proteções e bloqueia se necessário
 */
function protetor_verificar($redirecionar = true) {
    // Verificar se protetor está ativo
    if (!protetor_get_config('filtro_ativo', true)) {
        return ['permitido' => true, 'motivos' => ['protetor_desativado']];
    }

    $resultado = [
        'permitido' => true,
        'motivos' => [],
    ];

    $ip = protetor_get_ip();

    // 0. Verificar whitelist primeiro (sempre permite)
    if (protetor_get_config('filtro_ip_whitelist', true)) {
        if (protetor_ip_na_whitelist($ip)) {
            // IP na whitelist - permitir sem verificar mais nada
            protetor_registrar_visitante();
            $_SESSION['protetor_validado'] = true;
            $_SESSION['protetor_timestamp'] = time();
            return ['permitido' => true, 'motivos' => ['ip_whitelist']];
        }
    }

    // 0.5. Verificar blacklist (sempre bloqueia)
    if (protetor_get_config('filtro_ip_blacklist', true)) {
        $blacklist_motivo = protetor_ip_na_blacklist($ip);
        if ($blacklist_motivo) {
            $resultado['permitido'] = false;
            $resultado['motivos'][] = 'ip_blacklist: ' . $blacklist_motivo;
        }
    }

    // 0.6. Verificar país (GeoIP)
    if (protetor_get_config('filtro_geo', true) && $resultado['permitido']) {
        $country = protetor_get_country($ip);
        if ($country !== 'Unknown' && !protetor_pais_permitido($country)) {
            $resultado['permitido'] = false;
            $resultado['motivos'][] = 'pais_bloqueado: ' . $country;
        }
    }

    // 0.7. Verificar VPN/Datacenter (ISP)
    if (protetor_get_config('filtro_vpn', true) && $resultado['permitido']) {
        $isp = protetor_get_isp($ip);
        if ($isp !== 'Unknown' && protetor_is_datacenter($isp)) {
            $resultado['permitido'] = false;
            $resultado['motivos'][] = 'datacenter_vpn: ' . $isp;
        }
    }

    // 1. Verificar se é bot
    if (protetor_get_config('filtro_bot', true) && referer_detectar_bot()) {
        $resultado['permitido'] = false;
        $resultado['motivos'][] = 'bot_detectado';
    }

    // 2. Verificar pclid
    if (protetor_get_config('filtro_pclid', false) && $resultado['permitido']) {
        if (!pclid_verificar_request()) {
            $resultado['permitido'] = false;
            $resultado['motivos'][] = 'pclid_invalido';
        }
    }

    // 3. Verificar referer
    if (protetor_get_config('filtro_referer', true) && $resultado['permitido']) {
        $referer_check = referer_verificar_request();
        if (!$referer_check['permitido']) {
            $resultado['permitido'] = false;
            $resultado['motivos'][] = 'referer_' . $referer_check['motivo'];
        }
    }

    // 4. Verificar dispositivo
    if (protetor_get_config('filtro_device', true) && $resultado['permitido']) {
        $device = device_detectar();
        $apenas_mobile = protetor_get_config('apenas_mobile', true);
        $permitir_tablet = protetor_get_config('permitir_tablet', true);

        if ($apenas_mobile) {
            if ($device === 'desktop') {
                $resultado['permitido'] = false;
                $resultado['motivos'][] = 'desktop_bloqueado';
            } elseif ($device === 'tablet' && !$permitir_tablet) {
                $resultado['permitido'] = false;
                $resultado['motivos'][] = 'tablet_bloqueado';
            }
        }
    }

    // 5. Verificar UTMs obrigatórias
    if (protetor_get_config('filtro_utm', true) && $resultado['permitido']) {
        $utms_obrigatorias = protetor_get_config('utm_obrigatorias', ['utm_source', 'utm_medium', 'utm_campaign']);
        $permitir_fbclid = protetor_get_config('permitir_fbclid', true);

        $tem_fbclid = !empty($_GET['fbclid']);
        $tem_todas_utms = true;

        foreach ($utms_obrigatorias as $utm) {
            if (empty($_GET[$utm])) {
                $tem_todas_utms = false;
                break;
            }
        }

        if (!$tem_todas_utms && !($permitir_fbclid && $tem_fbclid)) {
            $resultado['permitido'] = false;
            $resultado['motivos'][] = 'utm_faltando';
        }
    }

    // Registrar resultado
    if ($resultado['permitido']) {
        protetor_registrar_visitante();
        $_SESSION['protetor_validado'] = true;
        $_SESSION['protetor_timestamp'] = time();
    } else {
        $motivo = implode(', ', $resultado['motivos']);
        protetor_registrar_bloqueio($motivo, [
            'get' => $_GET,
            'device' => device_detectar(),
        ]);

        // Log em arquivo
        if (protetor_get_config('log_arquivo', true)) {
            $log = date('Y-m-d H:i:s') . ' | BLOQUEADO | ' . $motivo . ' | IP: ' . protetor_get_ip() . PHP_EOL;
            @file_put_contents(__DIR__ . '/../protetor_bloqueios.log', $log, FILE_APPEND);
        }

        if ($redirecionar) {
            $redirect_url = protetor_get_config('redirect_url', '/noticia/pink/');
            // Garantir que é URL absoluta (começa com / ou http)
            if (!preg_match('#^(/|https?://)#', $redirect_url)) {
                $redirect_url = '/' . ltrim($redirect_url, './');
            }
            header('Location: ' . $redirect_url);
            exit;
        }
    }

    return $resultado;
}

/**
 * Verifica se o visitante já foi validado na sessão
 */
function protetor_ja_validado() {
    return isset($_SESSION['protetor_validado']) && $_SESSION['protetor_validado'] === true;
}

/**
 * Verifica com cache de sessão (para páginas internas)
 */
function protetor_verificar_com_cache($redirecionar = true) {
    $ip = protetor_get_ip();

    // SEMPRE verificar blacklist, mesmo com cache (prioridade máxima)
    if (protetor_get_config('filtro_ip_blacklist', true)) {
        $blacklist_motivo = protetor_ip_na_blacklist($ip);
        if ($blacklist_motivo) {
            // IP na blacklist - bloquear imediatamente
            protetor_registrar_bloqueio('ip_blacklist: ' . $blacklist_motivo, ['ip' => $ip]);

            if ($redirecionar) {
                $redirect_url = protetor_get_config('redirect_url', '/noticia/pink/');
                if (!preg_match('#^(/|https?://)#', $redirect_url)) {
                    $redirect_url = '/' . ltrim($redirect_url, './');
                }
                header('Location: ' . $redirect_url);
                exit;
            }
            return ['permitido' => false, 'motivos' => ['ip_blacklist: ' . $blacklist_motivo]];
        }
    }

    if (protetor_ja_validado()) {
        // Ainda assim registrar a página visitada
        protetor_registrar_visitante();
        return ['permitido' => true, 'motivos' => ['cache_sessao']];
    }
    return protetor_verificar($redirecionar);
}

/**
 * Gera código para o <head>
 */
function protetor_head() {
    $output = '';

    // === SCRIPTS DE TRACKING (Clarity, Facebook Pixel, etc.) ===
    $clarityId = protetor_get_config('clarity_id', '');
    $facebookPixelId = protetor_get_config('facebook_pixel_id', '');
    $scriptsHead = protetor_get_config('scripts_head', '');

    // Microsoft Clarity
    if (!empty($clarityId)) {
        $output .= "
<script type=\"text/javascript\">
    (function(c,l,a,r,i,t,y){
        c[a]=c[a]||function(){(c[a].q=c[a].q||[]).push(arguments)};
        t=l.createElement(r);t.async=1;t.src=\"https://www.clarity.ms/tag/\"+i;
        y=l.getElementsByTagName(r)[0];y.parentNode.insertBefore(t,y);
    })(window, document, \"clarity\", \"script\", \"{$clarityId}\");
</script>";
    }

    // Facebook Pixel
    if (!empty($facebookPixelId)) {
        $output .= "
<script>
!function(f,b,e,v,n,t,s)
{if(f.fbq)return;n=f.fbq=function(){n.callMethod?
n.callMethod.apply(n,arguments):n.queue.push(arguments)};
if(!f._fbq)f._fbq=n;n.push=n;n.loaded=!0;n.version='2.0';
n.queue=[];t=b.createElement(e);t.async=!0;
t.src=v;s=b.getElementsByTagName(e)[0];
s.parentNode.insertBefore(t,s)}(window, document,'script',
'https://connect.facebook.net/en_US/fbevents.js');
fbq('init', '{$facebookPixelId}');
fbq('track', 'PageView');
</script>
<noscript><img height=\"1\" width=\"1\" style=\"display:none\" src=\"https://www.facebook.com/tr?id={$facebookPixelId}&ev=PageView&noscript=1\"/></noscript>";
    }

    // Scripts customizados no head
    if (!empty($scriptsHead)) {
        $output .= "\n" . $scriptsHead . "\n";
    }

    // === CSS DE PROTEÇÃO ===
    if (protetor_get_config('anti_devtools', true) || protetor_get_config('anti_rightclick', true)) {
        $output .= anti_clone_meta_tags();
        $output .= anti_clone_css();
    }

    return $output;
}

/**
 * Gera código para o final do <body>
 */
function protetor_body() {
    $output = '';

    // === SCRIPT DE PRESERVAÇÃO DE UTMs ===
    $utms = protetor_get_utms();
    $utmsJson = json_encode($utms);
    $utmQuery = protetor_utm_query();

    $output .= "<script>
/* Protetor - UTM Preservation */
(function(){
    'use strict';

    // UTMs salvas da sessão
    window.PROTETOR_UTMS = {$utmsJson};
    window.PROTETOR_UTM_QUERY = " . json_encode($utmQuery) . ";

    // Função para adicionar UTMs a uma URL
    function addUtmsToUrl(url) {
        if (!window.PROTETOR_UTM_QUERY) return url;
        if (url.startsWith('#') || url.startsWith('javascript:') || url.startsWith('mailto:') || url.startsWith('tel:')) return url;
        if (url.includes('logout') || url.includes('sair')) return url;

        try {
            var urlObj = new URL(url, window.location.origin);

            // Não adicionar UTMs para domínios externos
            if (urlObj.hostname !== window.location.hostname) return url;

            // Adicionar cada UTM que não existe na URL
            Object.keys(window.PROTETOR_UTMS).forEach(function(key) {
                if (!urlObj.searchParams.has(key)) {
                    urlObj.searchParams.set(key, window.PROTETOR_UTMS[key]);
                }
            });

            return urlObj.toString();
        } catch(e) {
            // URL relativa simples
            var sep = url.indexOf('?') === -1 ? '?' : '&';
            return url + sep + window.PROTETOR_UTM_QUERY;
        }
    }

    // Interceptar cliques em links
    document.addEventListener('click', function(e) {
        var link = e.target.closest('a');
        if (link && link.href && !link.href.startsWith('#')) {
            var newHref = addUtmsToUrl(link.href);
            if (newHref !== link.href) {
                link.href = newHref;
            }
        }
    }, true);

    // Interceptar submissões de formulário
    document.addEventListener('submit', function(e) {
        var form = e.target;
        if (form.method.toLowerCase() === 'get' && form.action) {
            Object.keys(window.PROTETOR_UTMS).forEach(function(key) {
                if (!form.querySelector('input[name=\"' + key + '\"]')) {
                    var input = document.createElement('input');
                    input.type = 'hidden';
                    input.name = key;
                    input.value = window.PROTETOR_UTMS[key];
                    form.appendChild(input);
                }
            });
        }
    }, true);

    // Atualizar links existentes quando o DOM estiver pronto
    document.addEventListener('DOMContentLoaded', function() {
        document.querySelectorAll('a[href]').forEach(function(link) {
            if (!link.href.startsWith('#') && !link.href.includes('logout')) {
                link.href = addUtmsToUrl(link.href);
            }
        });
    });

    // Salvar no localStorage como backup (SEM pclid por segurança)
    if (window.PROTETOR_UTM_QUERY) {
        try {
            var utmsParaSalvar = {};
            Object.keys(window.PROTETOR_UTMS).forEach(function(k) {
                if (k !== 'pclid') { // Nunca salvar pclid no localStorage
                    utmsParaSalvar[k] = window.PROTETOR_UTMS[k];
                }
            });
            localStorage.setItem('protetor_utms', JSON.stringify(utmsParaSalvar));
            localStorage.setItem('protetor_utms_time', Date.now().toString());
        } catch(e) {}
    } else {
        // Tentar recuperar do localStorage se sessão vazia
        try {
            var saved = localStorage.getItem('protetor_utms');
            var savedTime = parseInt(localStorage.getItem('protetor_utms_time') || '0');
            var maxAge = 24 * 60 * 60 * 1000; // 24 horas

            if (saved && (Date.now() - savedTime) < maxAge) {
                window.PROTETOR_UTMS = JSON.parse(saved);
                var params = [];
                Object.keys(window.PROTETOR_UTMS).forEach(function(k) {
                    params.push(encodeURIComponent(k) + '=' + encodeURIComponent(window.PROTETOR_UTMS[k]));
                });
                window.PROTETOR_UTM_QUERY = params.join('&');
            }
        } catch(e) {}
    }

    // Expor função globalmente
    window.protetorAddUtms = addUtmsToUrl;

    // Remover pclid da URL após validação (segurança)
    // Impede que concorrentes copiem a URL com token
    if (window.location.search.includes('pclid=')) {
        try {
            var url = new URL(window.location.href);
            url.searchParams.delete('pclid');
            var newUrl = url.toString();
            // Usar replaceState para não adicionar ao histórico
            window.history.replaceState({}, document.title, newUrl);
        } catch(e) {
            // Fallback para navegadores antigos
            var cleanUrl = window.location.href.replace(/([?&])pclid=[^&]*(&|$)/, function(m, p1, p2) {
                return p2 ? p1 : '';
            });
            if (cleanUrl !== window.location.href) {
                window.history.replaceState({}, document.title, cleanUrl);
            }
        }
    }
})();
</script>";

    // === SCRIPTS DE PROTEÇÃO ANTI-CLONE ===
    $anti_devtools = protetor_get_config('anti_devtools', true);
    $anti_rightclick = protetor_get_config('anti_rightclick', true);
    $anti_shortcuts = protetor_get_config('anti_shortcuts', true);
    $anti_copy = protetor_get_config('anti_copy', true);
    $anti_select = protetor_get_config('anti_select', true);
    $anti_drag = protetor_get_config('anti_drag', true);

    if ($anti_devtools || $anti_rightclick || $anti_shortcuts || $anti_copy || $anti_select || $anti_drag) {
        $redirect_url = protetor_get_config('redirect_url', '/noticia/pink/');
        // Garantir que é URL absoluta (começa com / ou http)
        if (!preg_match('#^(/|https?://)#', $redirect_url)) {
            $redirect_url = '/' . ltrim($redirect_url, './');
        }
        $output .= "<script>(function(){'use strict';";
        $output .= "var safeUrl='" . addslashes($redirect_url) . "';";

        if ($anti_rightclick) {
            $output .= "document.addEventListener('contextmenu',function(e){e.preventDefault();return false;});";
        }

        if ($anti_shortcuts) {
            $output .= "document.addEventListener('keydown',function(e){if(e.keyCode===123||(e.ctrlKey&&e.shiftKey&&(e.keyCode===73||e.keyCode===74||e.keyCode===67))||(e.ctrlKey&&(e.keyCode===85||e.keyCode===83||e.keyCode===80))){e.preventDefault();window.location.href=safeUrl;return false;}});";
        }

        if ($anti_devtools) {
            $output .= "var dt=false,th=160,chk=0;setInterval(function(){var devOpen=window.outerWidth-window.innerWidth>th||window.outerHeight-window.innerHeight>th;if(devOpen){chk++;if(chk>2){window.location.href=safeUrl;}}else{chk=0;}},500);";
        }

        if ($anti_select) {
            $output .= "document.addEventListener('selectstart',function(e){if(e.target.tagName!=='INPUT'&&e.target.tagName!=='TEXTAREA'){e.preventDefault();return false;}});";
        }

        if ($anti_drag) {
            $output .= "document.addEventListener('dragstart',function(e){e.preventDefault();return false;});";
        }

        if ($anti_copy) {
            $output .= "document.addEventListener('copy',function(e){if(document.activeElement.tagName!=='INPUT'&&document.activeElement.tagName!=='TEXTAREA'){e.preventDefault();return false;}});";
        }

        $output .= "})();</script>";
    }

    return $output;
}

/**
 * Gera URL protegida com pclid
 */
function protetor_gerar_url($url, $utms = [], $expiry = null) {
    $url = pclid_url($url, $expiry);
    foreach ($utms as $key => $value) {
        $url .= '&' . urlencode($key) . '=' . urlencode($value);
    }
    return $url;
}
?>
