<?php
/**
 * Sistema de Token PCLID
 *
 * Gera tokens curtos e únicos para URLs de anúncios.
 * Tokens de ~10 caracteres, URL-safe.
 */

// Carregar config do banco se existir, senão usar padrão
$dbConfigPath = __DIR__ . '/db_config.php';
if (file_exists($dbConfigPath)) {
    require_once $dbConfigPath;
} else {
    require_once __DIR__ . '/config.php';
}

/**
 * Caracteres para encoding base62 (URL-safe)
 */
define('PCLID_CHARS', '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz');

/**
 * Converte número para base62
 */
function pclid_to_base62($num) {
    $chars = PCLID_CHARS;
    $base = strlen($chars);
    $result = '';

    while ($num > 0) {
        $result = $chars[$num % $base] . $result;
        $num = intdiv($num, $base);
    }

    return $result ?: '0';
}

/**
 * Converte base62 para número
 */
function pclid_from_base62($str) {
    $chars = PCLID_CHARS;
    $base = strlen($chars);
    $num = 0;

    for ($i = 0; $i < strlen($str); $i++) {
        $num = $num * $base + strpos($chars, $str[$i]);
    }

    return $num;
}

/**
 * Gera um token pclid curto (~10 caracteres)
 *
 * Formato: [hash de validação 6 chars][timestamp comprimido 4 chars]
 *
 * @param int $expiry Tempo de expiração em segundos (opcional)
 * @return string Token gerado (~10 caracteres)
 */
function pclid_gerar($expiry = null) {
    $expiry = $expiry ?? PROTETOR_TOKEN_EXPIRY;
    $expira_em = time() + $expiry;

    // Timestamp em base62 (comprimido)
    $timestamp_b62 = pclid_to_base62($expira_em);

    // Hash curto para validação (usando secret key)
    $secret = defined('PROTETOR_SECRET_KEY') ? PROTETOR_SECRET_KEY : 'default_secret';
    $hash_data = $expira_em . $secret . bin2hex(random_bytes(4));
    $hash = substr(hash('sha256', $hash_data), 0, 8);

    // Converter hash hex para base62 (mais curto)
    $hash_num = hexdec($hash);
    $hash_b62 = str_pad(pclid_to_base62($hash_num), 5, '0', STR_PAD_LEFT);

    // Token final: hash(5) + timestamp(5-6) = ~10-11 caracteres
    return $hash_b62 . $timestamp_b62;
}

/**
 * Valida um token pclid
 *
 * @param string $token Token a ser validado
 * @return bool|array False se inválido, array com dados se válido
 */
function pclid_validar($token) {
    if (empty($token) || strlen($token) < 8) {
        return false;
    }

    try {
        // Extrair partes do token
        $hash_b62 = substr($token, 0, 5);
        $timestamp_b62 = substr($token, 5);

        // Converter timestamp de volta
        $expira_em = pclid_from_base62($timestamp_b62);

        // Verificar se expirou
        if ($expira_em < time()) {
            return ['valido' => false, 'motivo' => 'expirado'];
        }

        // Verificar se é um timestamp válido (entre agora e 10 anos no futuro)
        $max_future = time() + (10 * 365 * 24 * 3600);
        if ($expira_em > $max_future || $expira_em < 1700000000) {
            return ['valido' => false, 'motivo' => 'formato_invalido'];
        }

        return [
            'valido' => true,
            'expira_em' => $expira_em,
            'tempo_restante' => $expira_em - time()
        ];
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Gera uma URL com pclid
 *
 * @param string $url URL base
 * @param int $expiry Tempo de expiração (opcional)
 * @return string URL com pclid
 */
function pclid_url($url, $expiry = null) {
    $token = pclid_gerar($expiry);
    $separator = (strpos($url, '?') === false) ? '?' : '&';
    return $url . $separator . 'pclid=' . $token;
}

/**
 * Verifica pclid da requisição atual
 * Compara com o token global salvo no banco
 *
 * @return bool True se válido
 */
function pclid_verificar_request() {
    $pclid = $_GET['pclid'] ?? '';

    if (empty($pclid)) {
        return false;
    }

    // Buscar token global do banco
    $tokenGlobal = pclid_get_token_global();

    if (empty($tokenGlobal)) {
        // Se não há token configurado, aceitar qualquer um válido
        $resultado = pclid_validar($pclid);
        return $resultado && (!isset($resultado['valido']) || $resultado['valido']);
    }

    // Comparar com token global (comparação exata)
    return $pclid === $tokenGlobal;
}

/**
 * Obtém o token global do banco de dados
 */
function pclid_get_token_global() {
    static $token = null;

    if ($token !== null) {
        return $token;
    }

    if (!defined('PROTETOR_DB_HOST') || !defined('PROTETOR_DB_PREFIX')) {
        return null;
    }

    try {
        $pdo = new PDO(
            "mysql:host=" . PROTETOR_DB_HOST . ";dbname=" . PROTETOR_DB_NAME . ";charset=utf8mb4",
            PROTETOR_DB_USER,
            PROTETOR_DB_PASS,
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );

        $prefix = PROTETOR_DB_PREFIX;
        $stmt = $pdo->prepare("SELECT valor FROM {$prefix}config WHERE chave = 'token_global'");
        $stmt->execute();
        $token = $stmt->fetchColumn() ?: null;

        return $token;
    } catch (PDOException $e) {
        return null;
    }
}

/**
 * Gera múltiplos tokens para uso em campanhas
 *
 * @param int $quantidade Quantidade de tokens
 * @param int $expiry Tempo de expiração
 * @return array Lista de tokens
 */
function pclid_gerar_lote($quantidade = 10, $expiry = null) {
    $tokens = [];
    for ($i = 0; $i < $quantidade; $i++) {
        $tokens[] = pclid_gerar($expiry);
    }
    return $tokens;
}
?>
