<?php
/**
 * Auto-Instalador do Sistema de Proteção
 *
 * Este arquivo configura o banco de dados e cria as tabelas necessárias.
 * Acesse via navegador para fazer a instalação inicial.
 */

session_start();

// Verificar se já está instalado
$configFile = __DIR__ . '/db_config.php';
$isInstalled = file_exists($configFile);

// Processar instalação
$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {

    if ($_POST['action'] === 'install') {
        $host = trim($_POST['db_host'] ?? 'localhost');
        $database = trim($_POST['db_name'] ?? '');
        $username = trim($_POST['db_user'] ?? 'root');
        $password = $_POST['db_pass'] ?? '';
        $prefix = trim($_POST['db_prefix'] ?? 'protetor_');

        // Credenciais do painel
        $adminUser = trim($_POST['admin_user'] ?? 'admin');
        $adminPass = $_POST['admin_pass'] ?? '';

        // URL de redirecionamento
        $redirectUrl = trim($_POST['redirect_url'] ?? '/noticia/');

        // Validar campos
        if (empty($database)) {
            $message = 'Nome do banco de dados é obrigatório!';
            $messageType = 'error';
        } elseif (empty($adminUser)) {
            $message = 'Usuário do painel é obrigatório!';
            $messageType = 'error';
        } elseif (empty($adminPass) || strlen($adminPass) < 6) {
            $message = 'Senha do painel deve ter pelo menos 6 caracteres!';
            $messageType = 'error';
        } elseif (empty($redirectUrl)) {
            $message = 'URL de redirecionamento é obrigatória!';
            $messageType = 'error';
        } else {
            try {
                // Testar conexão
                $pdo = new PDO(
                    "mysql:host=$host;dbname=$database;charset=utf8mb4",
                    $username,
                    $password,
                    [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
                );

                // Criar tabelas
                $tables = getTableSQL($prefix);
                foreach ($tables as $sql) {
                    $pdo->exec($sql);
                }

                // Salvar configuração
                $config = "<?php\n";
                $config .= "// Configuração do Sistema de Proteção - Gerado automaticamente\n";
                $config .= "// Data: " . date('Y-m-d H:i:s') . "\n\n";
                $config .= "// Banco de Dados\n";
                $config .= "define('PROTETOR_DB_HOST', " . var_export($host, true) . ");\n";
                $config .= "define('PROTETOR_DB_NAME', " . var_export($database, true) . ");\n";
                $config .= "define('PROTETOR_DB_USER', " . var_export($username, true) . ");\n";
                $config .= "define('PROTETOR_DB_PASS', " . var_export($password, true) . ");\n";
                $config .= "define('PROTETOR_DB_PREFIX', " . var_export($prefix, true) . ");\n\n";
                $config .= "// Credenciais do Painel Admin\n";
                $config .= "define('PROTETOR_ADMIN_USER', " . var_export($adminUser, true) . ");\n";
                $config .= "define('PROTETOR_ADMIN_PASS', " . var_export(password_hash($adminPass, PASSWORD_DEFAULT), true) . ");\n\n";
                $config .= "// Instalação\n";
                $config .= "define('PROTETOR_INSTALLED', true);\n";
                $config .= "?>\n";

                file_put_contents($configFile, $config);

                // Inserir configurações padrão (com URL de redirect)
                inserirConfigsPadrao($pdo, $prefix, $redirectUrl);

                // Auto-renomear o instalador por segurança
                $installFile = __FILE__;
                $backupName = __DIR__ . '/install_' . date('Ymd_His') . '_backup.php';
                @rename($installFile, $backupName);

                $message = 'Instalação concluída com sucesso! Redirecionando para o painel...';
                $messageType = 'success';
                $isInstalled = true;

                header('Refresh: 2; URL=admin/');

            } catch (PDOException $e) {
                $message = 'Erro de conexão: ' . $e->getMessage();
                $messageType = 'error';
            }
        }
    }

    if ($_POST['action'] === 'reinstall') {
        // Remover configuração para reinstalar
        if (file_exists($configFile)) {
            unlink($configFile);
        }
        $isInstalled = false;
        $message = 'Configuração removida. Preencha os dados para reinstalar.';
        $messageType = 'info';
    }
}

/**
 * Retorna SQL para criar as tabelas
 */
function getTableSQL($prefix) {
    return [
        // Tabela de configurações
        "CREATE TABLE IF NOT EXISTS {$prefix}config (
            id INT AUTO_INCREMENT PRIMARY KEY,
            chave VARCHAR(100) NOT NULL UNIQUE,
            valor TEXT,
            tipo ENUM('bool', 'string', 'int', 'json') DEFAULT 'string',
            descricao VARCHAR(255),
            grupo VARCHAR(50) DEFAULT 'geral',
            criado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
            atualizado_em DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            INDEX idx_chave (chave),
            INDEX idx_grupo (grupo)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        // Tabela de visitantes válidos
        "CREATE TABLE IF NOT EXISTS {$prefix}visitantes (
            id INT AUTO_INCREMENT PRIMARY KEY,
            session_id VARCHAR(128) NOT NULL,
            ip VARCHAR(45) NOT NULL,
            pclid VARCHAR(64),
            fbclid VARCHAR(255),
            utm_source VARCHAR(100),
            utm_medium VARCHAR(100),
            utm_campaign VARCHAR(255),
            referer TEXT,
            user_agent TEXT,
            dispositivo ENUM('mobile', 'desktop', 'tablet') DEFAULT 'desktop',
            pais VARCHAR(5),
            cidade VARCHAR(100),
            isp VARCHAR(255),
            validado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
            ultima_visita DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            paginas_visitadas INT DEFAULT 1,
            bloqueado TINYINT(1) DEFAULT 0,
            motivo_bloqueio VARCHAR(255),
            INDEX idx_session (session_id),
            INDEX idx_ip (ip),
            INDEX idx_pclid (pclid),
            INDEX idx_fbclid (fbclid),
            INDEX idx_validado (validado_em),
            INDEX idx_bloqueado (bloqueado)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        // Tabela de acessos bloqueados
        "CREATE TABLE IF NOT EXISTS {$prefix}bloqueios (
            id INT AUTO_INCREMENT PRIMARY KEY,
            ip VARCHAR(45) NOT NULL,
            motivo VARCHAR(255) NOT NULL,
            user_agent TEXT,
            referer TEXT,
            url_acessada TEXT,
            dados_request TEXT,
            isp VARCHAR(255),
            pais VARCHAR(5),
            dispositivo VARCHAR(20),
            criado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_ip (ip),
            INDEX idx_motivo (motivo),
            INDEX idx_criado (criado_em)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        // Tabela de IPs na whitelist
        "CREATE TABLE IF NOT EXISTS {$prefix}ip_whitelist (
            id INT AUTO_INCREMENT PRIMARY KEY,
            ip VARCHAR(45) NOT NULL,
            descricao VARCHAR(255),
            criado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_ip (ip)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        // Tabela de IPs na blacklist
        "CREATE TABLE IF NOT EXISTS {$prefix}ip_blacklist (
            id INT AUTO_INCREMENT PRIMARY KEY,
            ip VARCHAR(45) NOT NULL,
            tipo ENUM('single', 'cidr') DEFAULT 'single',
            motivo VARCHAR(255),
            criado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_ip (ip),
            INDEX idx_tipo (tipo)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        // Tabela de tokens pclid gerados
        "CREATE TABLE IF NOT EXISTS {$prefix}tokens (
            id INT AUTO_INCREMENT PRIMARY KEY,
            token VARCHAR(128) NOT NULL,
            campanha VARCHAR(255),
            usado TINYINT(1) DEFAULT 0,
            ip_uso VARCHAR(45),
            criado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
            usado_em DATETIME,
            expira_em DATETIME,
            UNIQUE KEY unique_token (token),
            INDEX idx_usado (usado),
            INDEX idx_expira (expira_em)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        // Tabela de estatísticas diárias
        "CREATE TABLE IF NOT EXISTS {$prefix}estatisticas (
            id INT AUTO_INCREMENT PRIMARY KEY,
            data DATE NOT NULL,
            total_visitantes INT DEFAULT 0,
            total_bloqueados INT DEFAULT 0,
            mobile INT DEFAULT 0,
            desktop INT DEFAULT 0,
            tablet INT DEFAULT 0,
            utm_sources JSON,
            motivos_bloqueio JSON,
            criado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_data (data),
            INDEX idx_data (data)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        // Tabela de cache de geolocalização
        "CREATE TABLE IF NOT EXISTS {$prefix}geo_cache (
            id INT AUTO_INCREMENT PRIMARY KEY,
            ip VARCHAR(45) NOT NULL,
            country_code VARCHAR(2) NOT NULL,
            source VARCHAR(20) NOT NULL COMMENT 'geolite2, dbip, geolite2+dbip, api',
            match_type VARCHAR(20) NOT NULL COMMENT 'match, priority, single, fallback',
            criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_ip (ip),
            INDEX idx_country (country_code)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci"
    ];
}

/**
 * Insere configurações padrão
 */
function inserirConfigsPadrao($pdo, $prefix, $redirectUrl = './noticia/') {
    $configs = [
        // Grupo: Geral
        ['filtro_ativo', '1', 'bool', 'Ativar sistema de proteção', 'geral'],
        ['redirect_url', $redirectUrl, 'string', 'URL de redirecionamento quando bloqueado', 'geral'],
        ['secret_key', bin2hex(random_bytes(16)), 'string', 'Chave secreta para tokens', 'geral'],
        ['token_expiry', '86400', 'int', 'Tempo de expiração do token (segundos)', 'geral'],

        // Grupo: Filtros
        ['filtro_pclid', '1', 'bool', 'Exigir token pclid', 'filtros'],
        ['filtro_referer', '1', 'bool', 'Filtrar por referer', 'filtros'],
        ['filtro_device', '1', 'bool', 'Filtrar por dispositivo', 'filtros'],
        ['filtro_utm', '1', 'bool', 'Exigir UTMs obrigatórias', 'filtros'],
        ['filtro_bot', '1', 'bool', 'Bloquear bots conhecidos', 'filtros'],
        ['filtro_ip_whitelist', '1', 'bool', 'Usar whitelist de IPs', 'filtros'],
        ['filtro_ip_blacklist', '1', 'bool', 'Usar blacklist de IPs', 'filtros'],

        // Grupo: Dispositivos
        ['apenas_mobile', '1', 'bool', 'Permitir apenas dispositivos móveis', 'dispositivos'],
        ['permitir_tablet', '1', 'bool', 'Permitir tablets', 'dispositivos'],

        // Grupo: UTMs
        ['utm_obrigatorias', '["utm_source","utm_medium","utm_campaign"]', 'json', 'UTMs obrigatórias', 'utms'],
        ['permitir_fbclid', '1', 'bool', 'Permitir acesso com fbclid mesmo sem UTMs', 'utms'],

        // Grupo: Referer
        ['referers_permitidos', '["facebook.com","fb.com","l.facebook.com","m.facebook.com","instagram.com","l.instagram.com","ig.me"]', 'json', 'Referers permitidos', 'referer'],
        ['referers_bloqueados', '["facebook.com/ads/library","www.facebook.com/ads/library"]', 'json', 'Referers bloqueados (biblioteca de anúncios)', 'referer'],
        ['bloquear_sem_referer', '0', 'bool', 'Bloquear acessos sem referer', 'referer'],

        // Grupo: Anti-Clone
        ['anti_devtools', '1', 'bool', 'Detectar DevTools aberto', 'anticlone'],
        ['anti_rightclick', '1', 'bool', 'Desabilitar clique direito', 'anticlone'],
        ['anti_shortcuts', '1', 'bool', 'Desabilitar atalhos (F12, Ctrl+U, etc)', 'anticlone'],
        ['anti_copy', '1', 'bool', 'Desabilitar copiar texto', 'anticlone'],
        ['anti_select', '1', 'bool', 'Desabilitar seleção de texto', 'anticlone'],
        ['anti_drag', '1', 'bool', 'Desabilitar arrastar elementos', 'anticlone'],

        // Grupo: Bots
        ['bloquear_facebook_crawler', '1', 'bool', 'Bloquear Facebook Crawlers', 'bots'],
        ['bloquear_scrapers', '1', 'bool', 'Bloquear Scrapers conhecidos', 'bots'],
        ['bloquear_datacenters', '1', 'bool', 'Bloquear IPs de Datacenters', 'bots'],
        ['bloquear_vpn_comercial', '1', 'bool', 'Bloquear VPNs comerciais', 'bots'],
        ['bots_bloqueados', '["bot","crawler","spider","scraper","curl","wget","python","httpclient","java","libwww","facebook","Facebot","facebookexternalhit"]', 'json', 'User agents de bots para bloquear', 'bots'],

        // Grupo: Geolocalização
        ['filtro_geo', '1', 'bool', 'Ativar filtro por país', 'geo'],
        ['filtro_vpn', '0', 'bool', 'Bloquear VPN/Proxy/Datacenter', 'geo'],
        ['paises_permitidos', '["BR"]', 'json', 'Países permitidos (códigos ISO: BR, US, PT...)', 'geo'],
        ['isps_bloqueados', '["amazon","aws","google","google cloud","azure","digitalocean","vultr","linode","ovh","hetzner","contabo","nordvpn","expressvpn","surfshark","protonvpn","cyberghost"]', 'json', 'ISPs/Datacenters bloqueados', 'geo'],
        ['bloquear_europa', '0', 'bool', 'Bloquear datacenters europeus', 'geo'],

        // Grupo: Tracking
        ['salvar_visitantes', '1', 'bool', 'Salvar visitantes válidos no banco', 'tracking'],
        ['salvar_bloqueios', '1', 'bool', 'Salvar tentativas bloqueadas', 'tracking'],
        ['log_arquivo', '1', 'bool', 'Salvar logs em arquivo', 'tracking'],

        // Grupo: Scripts
        ['clarity_id', '', 'string', 'ID do Microsoft Clarity (ex: abc123xyz)', 'scripts'],
        ['facebook_pixel_id', '', 'string', 'ID do Facebook Pixel (ex: 123456789012345)', 'scripts'],
        ['scripts_head', '', 'string', 'Scripts customizados para inserir no HEAD', 'scripts'],
    ];

    $stmt = $pdo->prepare("INSERT IGNORE INTO {$prefix}config (chave, valor, tipo, descricao, grupo) VALUES (?, ?, ?, ?, ?)");

    foreach ($configs as $config) {
        $stmt->execute($config);
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Instalador - Sistema de Proteção</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Inter', sans-serif; }
    </style>
</head>
<body class="bg-gray-100 min-h-screen flex items-center justify-center p-4">
    <div class="bg-white rounded-2xl shadow-xl max-w-lg w-full p-8">
        <div class="text-center mb-8">
            <div class="w-16 h-16 bg-green-500 rounded-full flex items-center justify-center mx-auto mb-4">
                <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z"/>
                </svg>
            </div>
            <h1 class="text-2xl font-bold text-gray-800">Sistema de Proteção</h1>
            <p class="text-gray-500 mt-2">Instalador Automático</p>
        </div>

        <?php if ($message): ?>
        <div class="mb-6 p-4 rounded-lg <?= $messageType === 'error' ? 'bg-red-100 text-red-700' : ($messageType === 'success' ? 'bg-green-100 text-green-700' : 'bg-blue-100 text-blue-700') ?>">
            <?= htmlspecialchars($message) ?>
        </div>
        <?php endif; ?>

        <?php if ($isInstalled): ?>
        <div class="text-center">
            <div class="bg-green-100 text-green-700 p-4 rounded-lg mb-6">
                <svg class="w-12 h-12 mx-auto mb-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                </svg>
                <p class="font-semibold">Sistema já instalado!</p>
            </div>
            <div class="space-y-3">
                <a href="admin/" class="block w-full bg-green-500 text-white py-3 px-4 rounded-lg font-semibold hover:bg-green-600 transition text-center">
                    Acessar Painel
                </a>
                <form method="POST" onsubmit="return confirm('Tem certeza? Isso vai remover a configuração atual.');">
                    <input type="hidden" name="action" value="reinstall">
                    <button type="submit" class="w-full bg-gray-200 text-gray-700 py-3 px-4 rounded-lg font-semibold hover:bg-gray-300 transition">
                        Reinstalar
                    </button>
                </form>
            </div>
        </div>
        <?php else: ?>
        <form method="POST" class="space-y-4">
            <input type="hidden" name="action" value="install">

            <div class="bg-gray-50 p-4 rounded-lg">
                <h3 class="font-semibold text-gray-700 mb-3">Banco de Dados</h3>
                <div class="space-y-3">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Host</label>
                        <input type="text" name="db_host" value="localhost" required
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-transparent">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Nome do Banco</label>
                        <input type="text" name="db_name" placeholder="meu_banco" required
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-transparent">
                    </div>
                    <div class="grid grid-cols-2 gap-3">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Usuário</label>
                            <input type="text" name="db_user" value="root" required
                                class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-transparent">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Senha</label>
                            <input type="password" name="db_pass"
                                class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-transparent">
                        </div>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Prefixo das Tabelas</label>
                        <input type="text" name="db_prefix" value="protetor_" required
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-transparent">
                    </div>
                </div>
            </div>

            <div class="bg-gray-50 p-4 rounded-lg">
                <h3 class="font-semibold text-gray-700 mb-3">Credenciais do Painel</h3>
                <div class="space-y-3">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Usuário Admin</label>
                        <input type="text" name="admin_user" value="admin" required
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-transparent">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Senha Admin</label>
                        <input type="password" name="admin_pass" required minlength="6"
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-transparent">
                        <p class="text-xs text-gray-500 mt-1">Mínimo 6 caracteres</p>
                    </div>
                </div>
            </div>

            <div class="bg-gray-50 p-4 rounded-lg">
                <h3 class="font-semibold text-gray-700 mb-3">Configuração de Bloqueio</h3>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">URL de Redirecionamento</label>
                    <input type="text" name="redirect_url" value="./noticia/" required
                        class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-transparent">
                    <p class="text-xs text-gray-500 mt-1">Para onde enviar visitantes bloqueados (pode ser uma URL completa)</p>
                </div>
            </div>

            <button type="submit" class="w-full bg-green-500 text-white py-3 px-4 rounded-lg font-semibold hover:bg-green-600 transition mt-6">
                Instalar Sistema
            </button>
        </form>
        <?php endif; ?>

        <div class="mt-8 pt-6 border-t border-gray-200 text-center text-sm text-gray-500">
            <p>Sistema de Proteção v1.0</p>
        </div>
    </div>
</body>
</html>
