<?php
/**
 * Filtro de Referer
 *
 * Verifica se o visitante veio de uma fonte permitida (anúncio do Facebook/Instagram)
 * e bloqueia acessos da biblioteca de anúncios ou diretos.
 */

require_once __DIR__ . '/config.php';

/**
 * Obtém o referer da requisição
 *
 * @return string|null Referer ou null se não disponível
 */
function referer_obter() {
    return $_SERVER['HTTP_REFERER'] ?? null;
}

/**
 * Extrai o domínio do referer
 *
 * @param string $referer URL do referer
 * @return string|null Domínio ou null
 */
function referer_extrair_dominio($referer) {
    if (empty($referer)) {
        return null;
    }

    $parsed = parse_url($referer);
    return $parsed['host'] ?? null;
}

/**
 * Verifica se o referer é da biblioteca de anúncios
 *
 * @param string $referer URL do referer
 * @return bool True se for da biblioteca (deve ser bloqueado)
 */
function referer_eh_biblioteca_anuncios($referer) {
    global $PROTETOR_REFERERS_BLOQUEADOS;

    if (empty($referer)) {
        return false;
    }

    $referer_lower = strtolower($referer);

    foreach ($PROTETOR_REFERERS_BLOQUEADOS as $bloqueado) {
        if (strpos($referer_lower, strtolower($bloqueado)) !== false) {
            return true;
        }
    }

    return false;
}

/**
 * Verifica se o referer é de uma fonte permitida
 *
 * @param string $referer URL do referer
 * @return bool True se permitido
 */
function referer_eh_permitido($referer) {
    global $PROTETOR_REFERERS_PERMITIDOS;

    if (empty($referer)) {
        return false;
    }

    $dominio = referer_extrair_dominio($referer);

    if (empty($dominio)) {
        return false;
    }

    $dominio_lower = strtolower($dominio);

    foreach ($PROTETOR_REFERERS_PERMITIDOS as $permitido) {
        // Verificar se o domínio termina com o permitido
        if ($dominio_lower === strtolower($permitido) ||
            substr($dominio_lower, -strlen($permitido) - 1) === '.' . strtolower($permitido)) {
            return true;
        }
    }

    return false;
}

/**
 * Verifica o referer da requisição atual
 *
 * @return array Resultado da verificação
 */
function referer_verificar_request() {
    if (!PROTETOR_REFERER_ATIVO) {
        return ['permitido' => true, 'motivo' => 'filtro_desativado'];
    }

    $referer = referer_obter();

    // Se não tem referer, pode ser acesso direto ou link compartilhado
    if (empty($referer)) {
        // Verificar se tem fbclid (veio do Facebook mesmo sem referer)
        if (!empty($_GET['fbclid'])) {
            return ['permitido' => true, 'motivo' => 'fbclid_presente'];
        }
        return ['permitido' => false, 'motivo' => 'sem_referer'];
    }

    // Verificar se é da biblioteca de anúncios (bloqueado)
    if (referer_eh_biblioteca_anuncios($referer)) {
        return ['permitido' => false, 'motivo' => 'biblioteca_anuncios'];
    }

    // Verificar se é de fonte permitida
    if (referer_eh_permitido($referer)) {
        return ['permitido' => true, 'motivo' => 'referer_permitido'];
    }

    // Verificar se tem fbclid mesmo com referer diferente
    if (!empty($_GET['fbclid'])) {
        return ['permitido' => true, 'motivo' => 'fbclid_presente'];
    }

    return ['permitido' => false, 'motivo' => 'referer_nao_permitido', 'referer' => $referer];
}

/**
 * Detecta se o acesso parece ser de um bot/crawler
 *
 * @return bool True se parecer ser bot
 */
function referer_detectar_bot() {
    global $PROTETOR_BOTS_BLOQUEADOS;

    $user_agent = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');

    foreach ($PROTETOR_BOTS_BLOQUEADOS as $bot) {
        if (strpos($user_agent, strtolower($bot)) !== false) {
            return true;
        }
    }

    return false;
}
?>
