<?php
/**
 * Filtro de Dispositivo
 *
 * Detecta o tipo de dispositivo e bloqueia acessos de desktop
 * (clonadores geralmente usam desktop para copiar sites)
 */

require_once __DIR__ . '/config.php';

/**
 * Detecta o tipo de dispositivo baseado no User Agent
 *
 * @return string 'mobile', 'tablet' ou 'desktop'
 */
function device_detectar() {
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';

    if (empty($user_agent)) {
        return 'desktop'; // Sem user agent = suspeito
    }

    $user_agent = strtolower($user_agent);

    // Padrões de tablets
    $tablets = [
        'ipad',
        'android(?!.*mobile)',
        'tablet',
        'kindle',
        'silk',
        'playbook',
    ];

    // Padrões de mobile
    $mobiles = [
        'mobile',
        'android.*mobile',
        'iphone',
        'ipod',
        'blackberry',
        'windows phone',
        'opera mini',
        'opera mobi',
        'iemobile',
        'webos',
        'palm',
        'symbian',
        'nokia',
        'samsung.*mobile',
        'lg.*mobile',
        'mot.*mobile',
        'htc.*mobile',
    ];

    // Verificar tablet primeiro
    foreach ($tablets as $pattern) {
        if (preg_match('/' . $pattern . '/i', $user_agent)) {
            // Verificar se não é mobile (alguns tablets têm 'mobile' no UA)
            if (strpos($user_agent, 'mobile') === false || strpos($user_agent, 'ipad') !== false) {
                return 'tablet';
            }
        }
    }

    // Verificar mobile
    foreach ($mobiles as $pattern) {
        if (preg_match('/' . $pattern . '/i', $user_agent)) {
            return 'mobile';
        }
    }

    return 'desktop';
}

/**
 * Verifica se o dispositivo é mobile
 *
 * @return bool True se for mobile
 */
function device_eh_mobile() {
    return device_detectar() === 'mobile';
}

/**
 * Verifica se o dispositivo é tablet
 *
 * @return bool True se for tablet
 */
function device_eh_tablet() {
    return device_detectar() === 'tablet';
}

/**
 * Verifica se o dispositivo é desktop
 *
 * @return bool True se for desktop
 */
function device_eh_desktop() {
    return device_detectar() === 'desktop';
}

/**
 * Verifica se o dispositivo é móvel (mobile ou tablet)
 *
 * @return bool True se for mobile ou tablet
 */
function device_eh_portatil() {
    $device = device_detectar();
    return $device === 'mobile' || $device === 'tablet';
}

/**
 * Verifica o dispositivo da requisição atual
 *
 * @return array Resultado da verificação
 */
function device_verificar_request() {
    if (!PROTETOR_DEVICE_ATIVO) {
        return ['permitido' => true, 'motivo' => 'filtro_desativado', 'dispositivo' => device_detectar()];
    }

    $dispositivo = device_detectar();

    if (PROTETOR_APENAS_MOBILE) {
        // Permitir mobile e tablet, bloquear desktop
        if ($dispositivo === 'desktop') {
            return [
                'permitido' => false,
                'motivo' => 'desktop_bloqueado',
                'dispositivo' => $dispositivo
            ];
        }
    }

    return [
        'permitido' => true,
        'motivo' => 'dispositivo_permitido',
        'dispositivo' => $dispositivo
    ];
}

/**
 * Obtém informações detalhadas do dispositivo
 *
 * @return array Informações do dispositivo
 */
function device_info() {
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';

    return [
        'tipo' => device_detectar(),
        'user_agent' => $user_agent,
        'eh_mobile' => device_eh_mobile(),
        'eh_tablet' => device_eh_tablet(),
        'eh_desktop' => device_eh_desktop(),
        'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
        'accept_language' => $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '',
    ];
}
?>
