<?php
/**
 * Funções de banco de dados para o sistema de proteção
 */

require_once __DIR__ . '/../includes/config.php';

/**
 * Cria a tabela de visitantes válidos se não existir
 */
function protetor_criar_tabela() {
    global $pdo;

    $sql = "CREATE TABLE IF NOT EXISTS visitantes_validos (
        id INT AUTO_INCREMENT PRIMARY KEY,
        session_id VARCHAR(128) NOT NULL,
        ip VARCHAR(45) NOT NULL,
        pclid VARCHAR(64),
        fbclid VARCHAR(255),
        utm_source VARCHAR(100),
        utm_medium VARCHAR(100),
        utm_campaign VARCHAR(255),
        referer TEXT,
        user_agent TEXT,
        dispositivo ENUM('mobile', 'desktop', 'tablet') DEFAULT 'desktop',
        validado_em DATETIME DEFAULT CURRENT_TIMESTAMP,
        ultima_visita DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        paginas_visitadas INT DEFAULT 1,
        bloqueado TINYINT(1) DEFAULT 0,
        motivo_bloqueio VARCHAR(255),
        INDEX idx_session (session_id),
        INDEX idx_ip (ip),
        INDEX idx_pclid (pclid),
        INDEX idx_fbclid (fbclid),
        INDEX idx_validado (validado_em)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    try {
        $pdo->exec($sql);
        return true;
    } catch (PDOException $e) {
        error_log("Erro ao criar tabela visitantes_validos: " . $e->getMessage());
        return false;
    }
}

/**
 * Registra um visitante válido no banco
 */
function protetor_registrar_visitante($dados) {
    global $pdo;

    try {
        // Verificar se já existe
        $stmt = $pdo->prepare("SELECT id, paginas_visitadas FROM visitantes_validos WHERE session_id = ?");
        $stmt->execute([$dados['session_id']]);
        $existente = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($existente) {
            // Atualizar visita existente
            $stmt = $pdo->prepare("UPDATE visitantes_validos SET
                ultima_visita = NOW(),
                paginas_visitadas = paginas_visitadas + 1
                WHERE id = ?");
            $stmt->execute([$existente['id']]);
            return $existente['id'];
        }

        // Inserir novo visitante
        $stmt = $pdo->prepare("INSERT INTO visitantes_validos
            (session_id, ip, pclid, fbclid, utm_source, utm_medium, utm_campaign, referer, user_agent, dispositivo)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");

        $stmt->execute([
            $dados['session_id'],
            $dados['ip'],
            $dados['pclid'] ?? null,
            $dados['fbclid'] ?? null,
            $dados['utm_source'] ?? null,
            $dados['utm_medium'] ?? null,
            $dados['utm_campaign'] ?? null,
            $dados['referer'] ?? null,
            $dados['user_agent'] ?? null,
            $dados['dispositivo'] ?? 'desktop'
        ]);

        return $pdo->lastInsertId();
    } catch (PDOException $e) {
        error_log("Erro ao registrar visitante: " . $e->getMessage());
        return false;
    }
}

/**
 * Verifica se o visitante está na lista de válidos
 */
function protetor_visitante_valido($session_id) {
    global $pdo;

    try {
        $stmt = $pdo->prepare("SELECT id, bloqueado FROM visitantes_validos WHERE session_id = ? AND bloqueado = 0");
        $stmt->execute([$session_id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result ? true : false;
    } catch (PDOException $e) {
        error_log("Erro ao verificar visitante: " . $e->getMessage());
        return false;
    }
}

/**
 * Bloqueia um visitante
 */
function protetor_bloquear_visitante($session_id, $motivo = '') {
    global $pdo;

    try {
        $stmt = $pdo->prepare("UPDATE visitantes_validos SET bloqueado = 1, motivo_bloqueio = ? WHERE session_id = ?");
        $stmt->execute([$motivo, $session_id]);
        return true;
    } catch (PDOException $e) {
        error_log("Erro ao bloquear visitante: " . $e->getMessage());
        return false;
    }
}

/**
 * Obtém estatísticas de visitantes
 */
function protetor_estatisticas() {
    global $pdo;

    try {
        $stats = [];

        // Total de visitantes válidos
        $stmt = $pdo->query("SELECT COUNT(*) FROM visitantes_validos WHERE bloqueado = 0");
        $stats['total_validos'] = $stmt->fetchColumn();

        // Total bloqueados
        $stmt = $pdo->query("SELECT COUNT(*) FROM visitantes_validos WHERE bloqueado = 1");
        $stats['total_bloqueados'] = $stmt->fetchColumn();

        // Visitantes hoje
        $stmt = $pdo->query("SELECT COUNT(*) FROM visitantes_validos WHERE DATE(validado_em) = CURDATE()");
        $stats['hoje'] = $stmt->fetchColumn();

        // Por dispositivo
        $stmt = $pdo->query("SELECT dispositivo, COUNT(*) as total FROM visitantes_validos GROUP BY dispositivo");
        $stats['por_dispositivo'] = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        // Por UTM source
        $stmt = $pdo->query("SELECT utm_source, COUNT(*) as total FROM visitantes_validos WHERE utm_source IS NOT NULL GROUP BY utm_source ORDER BY total DESC LIMIT 10");
        $stats['por_utm_source'] = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        return $stats;
    } catch (PDOException $e) {
        error_log("Erro ao obter estatísticas: " . $e->getMessage());
        return [];
    }
}

// Criar tabela automaticamente ao incluir este arquivo
protetor_criar_tabela();
?>
