<?php
/**
 * Proteções Anti-Clone (JavaScript)
 *
 * Gera código JavaScript que dificulta a clonagem do site:
 * - Desabilita clique direito
 * - Desabilita teclas de atalho (F12, Ctrl+U, Ctrl+Shift+I, etc)
 * - Detecta DevTools aberto
 * - Ofusca seletores CSS
 * - Adiciona watermark invisível
 */

require_once __DIR__ . '/config.php';

/**
 * Gera o código JavaScript de proteção
 *
 * @return string Código JavaScript
 */
function anti_clone_gerar_js() {
    if (!PROTETOR_ANTI_CLONE_ATIVO) {
        return '';
    }

    $js = <<<'JAVASCRIPT'
(function() {
    'use strict';

    // ========================================
    // DESABILITAR CLIQUE DIREITO
    // ========================================
    document.addEventListener('contextmenu', function(e) {
        e.preventDefault();
        return false;
    });

    // ========================================
    // DESABILITAR TECLAS DE ATALHO
    // ========================================
    document.addEventListener('keydown', function(e) {
        // F12
        if (e.keyCode === 123) {
            e.preventDefault();
            return false;
        }

        // Ctrl+Shift+I (DevTools)
        if (e.ctrlKey && e.shiftKey && e.keyCode === 73) {
            e.preventDefault();
            return false;
        }

        // Ctrl+Shift+J (Console)
        if (e.ctrlKey && e.shiftKey && e.keyCode === 74) {
            e.preventDefault();
            return false;
        }

        // Ctrl+Shift+C (Inspect Element)
        if (e.ctrlKey && e.shiftKey && e.keyCode === 67) {
            e.preventDefault();
            return false;
        }

        // Ctrl+U (View Source)
        if (e.ctrlKey && e.keyCode === 85) {
            e.preventDefault();
            return false;
        }

        // Ctrl+S (Save Page)
        if (e.ctrlKey && e.keyCode === 83) {
            e.preventDefault();
            return false;
        }

        // Ctrl+P (Print)
        if (e.ctrlKey && e.keyCode === 80) {
            e.preventDefault();
            return false;
        }
    });

    // ========================================
    // DETECTAR DEVTOOLS
    // ========================================
    var devToolsOpen = false;
    var threshold = 160;

    var checkDevTools = function() {
        var widthThreshold = window.outerWidth - window.innerWidth > threshold;
        var heightThreshold = window.outerHeight - window.innerHeight > threshold;

        if (widthThreshold || heightThreshold) {
            if (!devToolsOpen) {
                devToolsOpen = true;
                onDevToolsOpen();
            }
        } else {
            devToolsOpen = false;
        }
    };

    var onDevToolsOpen = function() {
        // Redirecionar ou mostrar aviso
        console.clear();
        console.log('%cAcesso não autorizado detectado!', 'color: red; font-size: 24px; font-weight: bold;');
    };

    setInterval(checkDevTools, 1000);

    // Detectar via console.log
    var element = new Image();
    Object.defineProperty(element, 'id', {
        get: function() {
            onDevToolsOpen();
        }
    });

    // ========================================
    // DESABILITAR SELEÇÃO DE TEXTO
    // ========================================
    document.addEventListener('selectstart', function(e) {
        // Permitir seleção em inputs e textareas
        if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA') {
            return true;
        }
        e.preventDefault();
        return false;
    });

    // ========================================
    // DESABILITAR ARRASTAR ELEMENTOS
    // ========================================
    document.addEventListener('dragstart', function(e) {
        e.preventDefault();
        return false;
    });

    // ========================================
    // DESABILITAR COPIAR
    // ========================================
    document.addEventListener('copy', function(e) {
        // Permitir copiar em inputs
        if (document.activeElement.tagName === 'INPUT' || document.activeElement.tagName === 'TEXTAREA') {
            return true;
        }
        e.preventDefault();
        return false;
    });

    // ========================================
    // CSS ANTI-SELEÇÃO
    // ========================================
    var style = document.createElement('style');
    style.textContent = `
        * {
            -webkit-user-select: none !important;
            -moz-user-select: none !important;
            -ms-user-select: none !important;
            user-select: none !important;
        }
        input, textarea {
            -webkit-user-select: text !important;
            -moz-user-select: text !important;
            -ms-user-select: text !important;
            user-select: text !important;
        }
    `;
    document.head.appendChild(style);

    // ========================================
    // WATERMARK INVISÍVEL
    // ========================================
    var watermark = document.createElement('div');
    watermark.style.cssText = 'position:fixed;bottom:-9999px;right:-9999px;opacity:0.001;pointer-events:none;z-index:-1;';
    watermark.innerHTML = '<!-- Protected - ' + new Date().toISOString() + ' -->';
    document.body.appendChild(watermark);

})();
JAVASCRIPT;

    return $js;
}

/**
 * Gera tag script com proteções
 *
 * @return string Tag <script> completa
 */
function anti_clone_script_tag() {
    if (!PROTETOR_ANTI_CLONE_ATIVO) {
        return '';
    }

    $js = anti_clone_gerar_js();
    return '<script>' . $js . '</script>';
}

/**
 * Gera CSS adicional de proteção
 *
 * @return string CSS de proteção
 */
function anti_clone_css() {
    if (!PROTETOR_ANTI_CLONE_ATIVO) {
        return '';
    }

    return <<<'CSS'
<style>
    /* Proteção anti-seleção */
    body {
        -webkit-touch-callout: none;
    }

    /* Esconder scrollbar no inspect */
    ::-webkit-scrollbar {
        width: 8px;
        height: 8px;
    }

    /* Desabilitar highlight no mobile */
    * {
        -webkit-tap-highlight-color: transparent;
    }

    /* Proteção de imagens */
    img {
        pointer-events: none;
    }
</style>
CSS;
}

/**
 * Gera meta tags de proteção
 *
 * @return string Meta tags
 */
function anti_clone_meta_tags() {
    return <<<'META'
<meta name="robots" content="noarchive, noimageindex">
<meta http-equiv="X-UA-Compatible" content="IE=edge">
META;
}
?>
