<?php
require_once 'auth.php';
requireAuth();

$pdo = getDB();
$prefix = PROTETOR_DB_PREFIX;

$message = '';
$messageType = '';

// Processar ações
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    // Adicionar à whitelist
    if ($action === 'add_whitelist') {
        $ip = trim($_POST['ip'] ?? '');
        $descricao = trim($_POST['descricao'] ?? '');

        if (filter_var($ip, FILTER_VALIDATE_IP)) {
            try {
                $stmt = $pdo->prepare("INSERT INTO {$prefix}ip_whitelist (ip, descricao) VALUES (?, ?)");
                $stmt->execute([$ip, $descricao]);
                $message = "IP $ip adicionado à whitelist!";
                $messageType = 'success';
            } catch (PDOException $e) {
                if ($e->getCode() == 23000) {
                    $message = "IP $ip já está na whitelist!";
                } else {
                    $message = "Erro ao adicionar IP: " . $e->getMessage();
                }
                $messageType = 'error';
            }
        } else {
            $message = "IP inválido!";
            $messageType = 'error';
        }
    }

    // Adicionar à blacklist
    if ($action === 'add_blacklist') {
        $ip = trim($_POST['ip'] ?? '');
        $motivo = trim($_POST['motivo'] ?? '');
        $tipo = strpos($ip, '/') !== false ? 'cidr' : 'single';

        // Validar IP ou CIDR
        $isValid = false;
        if ($tipo === 'cidr') {
            $parts = explode('/', $ip);
            $isValid = filter_var($parts[0], FILTER_VALIDATE_IP) && is_numeric($parts[1]) && $parts[1] >= 0 && $parts[1] <= 32;
        } else {
            $isValid = filter_var($ip, FILTER_VALIDATE_IP);
        }

        if ($isValid) {
            try {
                $stmt = $pdo->prepare("INSERT INTO {$prefix}ip_blacklist (ip, tipo, motivo) VALUES (?, ?, ?)");
                $stmt->execute([$ip, $tipo, $motivo]);
                $message = "IP $ip adicionado à blacklist!";
                $messageType = 'success';
            } catch (PDOException $e) {
                if ($e->getCode() == 23000) {
                    $message = "IP $ip já está na blacklist!";
                } else {
                    $message = "Erro ao adicionar IP: " . $e->getMessage();
                }
                $messageType = 'error';
            }
        } else {
            $message = "IP ou CIDR inválido!";
            $messageType = 'error';
        }
    }

    // Remover da whitelist
    if ($action === 'remove_whitelist') {
        $id = (int)($_POST['id'] ?? 0);
        $stmt = $pdo->prepare("DELETE FROM {$prefix}ip_whitelist WHERE id = ?");
        $stmt->execute([$id]);
        $message = "IP removido da whitelist!";
        $messageType = 'success';
    }

    // Remover da blacklist
    if ($action === 'remove_blacklist') {
        $id = (int)($_POST['id'] ?? 0);
        $stmt = $pdo->prepare("DELETE FROM {$prefix}ip_blacklist WHERE id = ?");
        $stmt->execute([$id]);
        $message = "IP removido da blacklist!";
        $messageType = 'success';
    }
}

// Buscar IPs
$whitelist = $pdo->query("SELECT * FROM {$prefix}ip_whitelist ORDER BY criado_em DESC")->fetchAll(PDO::FETCH_ASSOC);
$blacklist = $pdo->query("SELECT * FROM {$prefix}ip_blacklist ORDER BY criado_em DESC")->fetchAll(PDO::FETCH_ASSOC);

// Obter IP do visitante atual
$currentIp = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['HTTP_X_REAL_IP'] ?? $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
if (strpos($currentIp, ',') !== false) {
    $currentIp = trim(explode(',', $currentIp)[0]);
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gerenciar IPs - Sistema de Proteção</title>
    <link rel="icon" type="image/svg+xml" href="/assets/favicon-protetor.svg">
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>body { font-family: 'Inter', sans-serif; }</style>
</head>
<body class="bg-gray-100 min-h-screen">
    <header class="bg-white shadow-sm">
        <div class="max-w-7xl mx-auto px-4 py-4 flex items-center justify-between">
            <div class="flex items-center gap-3">
                <div class="w-10 h-10 bg-green-500 rounded-lg flex items-center justify-center">
                    <i class="fas fa-shield-alt text-white"></i>
                </div>
                <div>
                    <h1 class="font-bold text-gray-800">Sistema de Proteção</h1>
                    <p class="text-xs text-gray-500">Painel Administrativo</p>
                </div>
            </div>
            <nav class="flex items-center gap-4">
                <a href="index.php" class="text-gray-600 hover:text-green-500">Dashboard</a>
                <a href="settings.php" class="text-gray-600 hover:text-green-500">Configurações</a>
                <a href="visitors.php" class="text-gray-600 hover:text-green-500">Visitantes</a>
                <a href="blocks.php" class="text-gray-600 hover:text-green-500">Bloqueios</a>
                <a href="tokens.php" class="text-gray-600 hover:text-green-500">Tokens</a>
                <a href="ips.php" class="text-green-500 font-medium">IPs</a>
                <a href="logout.php" class="text-gray-400 hover:text-red-500"><i class="fas fa-sign-out-alt"></i></a>
            </nav>
        </div>
    </header>

    <main class="max-w-7xl mx-auto px-4 py-8">
        <?php if ($message): ?>
        <div class="mb-4 p-4 rounded-lg <?= $messageType === 'error' ? 'bg-red-100 text-red-700' : 'bg-green-100 text-green-700' ?>">
            <i class="fas fa-<?= $messageType === 'error' ? 'exclamation-circle' : 'check-circle' ?> mr-2"></i>
            <?= htmlspecialchars($message) ?>
        </div>
        <?php endif; ?>

        <div class="mb-6 p-4 bg-blue-50 rounded-lg">
            <i class="fas fa-info-circle text-blue-500 mr-2"></i>
            <span class="text-blue-700">Seu IP atual: <code class="bg-blue-100 px-2 py-1 rounded"><?= htmlspecialchars($currentIp) ?></code></span>
        </div>

        <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
            <!-- Whitelist -->
            <div class="bg-white rounded-xl shadow-sm">
                <div class="px-6 py-4 border-b border-gray-100 flex items-center justify-between">
                    <div>
                        <h2 class="font-bold text-gray-800">
                            <i class="fas fa-check-circle text-green-500 mr-2"></i>Whitelist
                        </h2>
                        <p class="text-sm text-gray-500">IPs sempre permitidos</p>
                    </div>
                    <span class="px-3 py-1 bg-green-100 text-green-700 rounded-full text-sm">
                        <?= count($whitelist) ?> IPs
                    </span>
                </div>

                <!-- Formulário Whitelist -->
                <div class="p-4 bg-gray-50 border-b border-gray-100">
                    <form method="POST" class="flex gap-2">
                        <input type="hidden" name="action" value="add_whitelist">
                        <input type="text" name="ip" placeholder="Ex: 192.168.1.100" required
                            class="flex-1 px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-green-500">
                        <input type="text" name="descricao" placeholder="Descrição (opcional)"
                            class="flex-1 px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-green-500">
                        <button type="submit" class="bg-green-500 text-white px-4 py-2 rounded-lg hover:bg-green-600 transition">
                            <i class="fas fa-plus"></i>
                        </button>
                    </form>
                </div>

                <!-- Lista Whitelist -->
                <div class="max-h-96 overflow-y-auto">
                    <?php if (empty($whitelist)): ?>
                    <div class="p-6 text-center text-gray-400">
                        <i class="fas fa-inbox text-3xl mb-2"></i>
                        <p>Nenhum IP na whitelist</p>
                    </div>
                    <?php else: ?>
                    <table class="w-full text-sm">
                        <thead class="bg-gray-50 sticky top-0">
                            <tr class="text-left text-gray-500">
                                <th class="px-4 py-2">IP</th>
                                <th class="px-4 py-2">Descrição</th>
                                <th class="px-4 py-2">Data</th>
                                <th class="px-4 py-2"></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($whitelist as $item): ?>
                            <tr class="border-b border-gray-50 hover:bg-gray-50">
                                <td class="px-4 py-2">
                                    <code class="bg-green-50 text-green-700 px-2 py-1 rounded text-xs"><?= htmlspecialchars($item['ip']) ?></code>
                                </td>
                                <td class="px-4 py-2 text-gray-600"><?= htmlspecialchars($item['descricao'] ?: '-') ?></td>
                                <td class="px-4 py-2 text-gray-400 text-xs"><?= date('d/m/Y', strtotime($item['criado_em'])) ?></td>
                                <td class="px-4 py-2">
                                    <form method="POST" class="inline" onsubmit="return confirm('Remover este IP?');">
                                        <input type="hidden" name="action" value="remove_whitelist">
                                        <input type="hidden" name="id" value="<?= $item['id'] ?>">
                                        <button type="submit" class="text-red-400 hover:text-red-600">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </form>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Blacklist -->
            <div class="bg-white rounded-xl shadow-sm">
                <div class="px-6 py-4 border-b border-gray-100 flex items-center justify-between">
                    <div>
                        <h2 class="font-bold text-gray-800">
                            <i class="fas fa-ban text-red-500 mr-2"></i>Blacklist
                        </h2>
                        <p class="text-sm text-gray-500">IPs sempre bloqueados</p>
                    </div>
                    <span class="px-3 py-1 bg-red-100 text-red-700 rounded-full text-sm">
                        <?= count($blacklist) ?> IPs
                    </span>
                </div>

                <!-- Formulário Blacklist -->
                <div class="p-4 bg-gray-50 border-b border-gray-100">
                    <form method="POST" class="flex gap-2">
                        <input type="hidden" name="action" value="add_blacklist">
                        <input type="text" name="ip" placeholder="IP ou CIDR (ex: 192.168.1.0/24)" required
                            class="flex-1 px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-red-500">
                        <input type="text" name="motivo" placeholder="Motivo (opcional)"
                            class="flex-1 px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-red-500">
                        <button type="submit" class="bg-red-500 text-white px-4 py-2 rounded-lg hover:bg-red-600 transition">
                            <i class="fas fa-plus"></i>
                        </button>
                    </form>
                </div>

                <!-- Lista Blacklist -->
                <div class="max-h-96 overflow-y-auto">
                    <?php if (empty($blacklist)): ?>
                    <div class="p-6 text-center text-gray-400">
                        <i class="fas fa-inbox text-3xl mb-2"></i>
                        <p>Nenhum IP na blacklist</p>
                    </div>
                    <?php else: ?>
                    <table class="w-full text-sm">
                        <thead class="bg-gray-50 sticky top-0">
                            <tr class="text-left text-gray-500">
                                <th class="px-4 py-2">IP</th>
                                <th class="px-4 py-2">Tipo</th>
                                <th class="px-4 py-2">Motivo</th>
                                <th class="px-4 py-2"></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($blacklist as $item): ?>
                            <tr class="border-b border-gray-50 hover:bg-gray-50">
                                <td class="px-4 py-2">
                                    <code class="bg-red-50 text-red-700 px-2 py-1 rounded text-xs"><?= htmlspecialchars($item['ip']) ?></code>
                                </td>
                                <td class="px-4 py-2">
                                    <span class="px-2 py-1 rounded text-xs <?= $item['tipo'] === 'cidr' ? 'bg-purple-100 text-purple-700' : 'bg-gray-100 text-gray-600' ?>">
                                        <?= $item['tipo'] ?>
                                    </span>
                                </td>
                                <td class="px-4 py-2 text-gray-600 text-xs"><?= htmlspecialchars($item['motivo'] ?: '-') ?></td>
                                <td class="px-4 py-2">
                                    <form method="POST" class="inline" onsubmit="return confirm('Remover este IP?');">
                                        <input type="hidden" name="action" value="remove_blacklist">
                                        <input type="hidden" name="id" value="<?= $item['id'] ?>">
                                        <button type="submit" class="text-red-400 hover:text-red-600">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </form>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Dicas -->
        <div class="mt-8 bg-white rounded-xl shadow-sm p-6">
            <h3 class="font-bold text-gray-800 mb-4">
                <i class="fas fa-lightbulb text-yellow-500 mr-2"></i>Dicas de Uso
            </h3>
            <div class="grid md:grid-cols-2 gap-6 text-sm text-gray-600">
                <div>
                    <h4 class="font-medium text-gray-700 mb-2">Whitelist</h4>
                    <ul class="space-y-1">
                        <li><i class="fas fa-check text-green-500 mr-1"></i> Use para seu IP ou de pessoas autorizadas</li>
                        <li><i class="fas fa-check text-green-500 mr-1"></i> IPs na whitelist NUNCA são bloqueados</li>
                        <li><i class="fas fa-check text-green-500 mr-1"></i> Adicione seu IP atual para testar</li>
                    </ul>
                </div>
                <div>
                    <h4 class="font-medium text-gray-700 mb-2">Blacklist</h4>
                    <ul class="space-y-1">
                        <li><i class="fas fa-ban text-red-500 mr-1"></i> Bloqueia IPs específicos ou ranges (CIDR)</li>
                        <li><i class="fas fa-ban text-red-500 mr-1"></i> Use CIDR para bloquear subnets inteiras</li>
                        <li><i class="fas fa-ban text-red-500 mr-1"></i> Ex: 192.168.1.0/24 bloqueia 256 IPs</li>
                    </ul>
                </div>
            </div>
        </div>
    </main>
</body>
</html>
