<?php
/**
 * Autenticação do Painel Admin - Protetor
 *
 * Usa credenciais próprias definidas no instalador
 */

// Timezone: Horário de São Paulo
date_default_timezone_set('America/Sao_Paulo');

session_start();

// Carregar configuração do banco do protetor
$dbConfig = __DIR__ . '/../db_config.php';
if (!file_exists($dbConfig)) {
    header('Location: ../install.php');
    exit;
}
require_once $dbConfig;

/**
 * Verifica se está autenticado
 */
function isAuthenticated() {
    return isset($_SESSION['protetor_admin_auth']) && $_SESSION['protetor_admin_auth'] === true;
}

/**
 * Requer autenticação
 */
function requireAuth() {
    if (!isAuthenticated()) {
        header('Location: login.php');
        exit;
    }
}

/**
 * Faz login usando credenciais do protetor
 */
function login($usuario, $senha) {
    if (!defined('PROTETOR_ADMIN_USER') || !defined('PROTETOR_ADMIN_PASS')) {
        return false;
    }

    if ($usuario === PROTETOR_ADMIN_USER && password_verify($senha, PROTETOR_ADMIN_PASS)) {
        $_SESSION['protetor_admin_auth'] = true;
        $_SESSION['protetor_admin_user'] = $usuario;
        $_SESSION['protetor_admin_login_time'] = time();
        return true;
    }

    return false;
}

/**
 * Faz logout
 */
function logout() {
    unset($_SESSION['protetor_admin_auth']);
    unset($_SESSION['protetor_admin_user']);
    unset($_SESSION['protetor_admin_login_time']);
}

/**
 * Obtém dados do admin logado
 */
function getAdminLogado() {
    if (!isAuthenticated()) {
        return null;
    }

    return [
        'usuario' => $_SESSION['protetor_admin_user'] ?? 'Admin'
    ];
}

/**
 * Conecta ao banco de dados do protetor
 */
function getDB() {
    static $pdo = null;

    if ($pdo === null && defined('PROTETOR_DB_HOST')) {
        try {
            $pdo = new PDO(
                "mysql:host=" . PROTETOR_DB_HOST . ";dbname=" . PROTETOR_DB_NAME . ";charset=utf8mb4",
                PROTETOR_DB_USER,
                PROTETOR_DB_PASS,
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
        } catch (PDOException $e) {
            die("Erro de conexão: " . $e->getMessage());
        }
    }

    return $pdo;
}

/**
 * Obtém uma configuração do banco
 */
function getConfig($chave, $default = null) {
    $pdo = getDB();

    if (!$pdo || !defined('PROTETOR_DB_PREFIX')) {
        return $default;
    }

    $prefix = PROTETOR_DB_PREFIX;

    try {
        $stmt = $pdo->prepare("SELECT valor, tipo FROM {$prefix}config WHERE chave = ?");
        $stmt->execute([$chave]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) {
            return $default;
        }

        switch ($row['tipo']) {
            case 'bool':
                return $row['valor'] === '1';
            case 'int':
                return (int) $row['valor'];
            case 'json':
                return json_decode($row['valor'], true);
            default:
                return $row['valor'];
        }
    } catch (PDOException $e) {
        return $default;
    }
}

/**
 * Define uma configuração no banco
 */
function setConfig($chave, $valor) {
    $pdo = getDB();

    if (!$pdo || !defined('PROTETOR_DB_PREFIX')) {
        return false;
    }

    $prefix = PROTETOR_DB_PREFIX;

    // Converter valor para string
    if (is_bool($valor)) {
        $valor = $valor ? '1' : '0';
    } elseif (is_array($valor)) {
        $valor = json_encode($valor);
    }

    try {
        $stmt = $pdo->prepare("UPDATE {$prefix}config SET valor = ? WHERE chave = ?");
        return $stmt->execute([$valor, $chave]);
    } catch (PDOException $e) {
        return false;
    }
}

/**
 * Obtém todas as configurações de um grupo
 */
function getConfigsByGroup($grupo) {
    $pdo = getDB();

    if (!$pdo || !defined('PROTETOR_DB_PREFIX')) {
        return [];
    }

    $prefix = PROTETOR_DB_PREFIX;

    try {
        $stmt = $pdo->prepare("SELECT chave, valor, tipo, descricao FROM {$prefix}config WHERE grupo = ? ORDER BY id");
        $stmt->execute([$grupo]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $configs = [];
        foreach ($rows as $row) {
            $valor = $row['valor'];
            switch ($row['tipo']) {
                case 'bool':
                    $valor = $row['valor'] === '1';
                    break;
                case 'int':
                    $valor = (int) $row['valor'];
                    break;
                case 'json':
                    $valor = json_decode($row['valor'], true);
                    break;
            }
            $configs[$row['chave']] = [
                'valor' => $valor,
                'tipo' => $row['tipo'],
                'descricao' => $row['descricao']
            ];
        }

        return $configs;
    } catch (PDOException $e) {
        return [];
    }
}

/**
 * Obtém todos os grupos de configuração
 */
function getConfigGroups() {
    $pdo = getDB();

    if (!$pdo || !defined('PROTETOR_DB_PREFIX')) {
        return [];
    }

    $prefix = PROTETOR_DB_PREFIX;

    try {
        $stmt = $pdo->query("SELECT DISTINCT grupo FROM {$prefix}config ORDER BY grupo");
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    } catch (PDOException $e) {
        return [];
    }
}
?>
