<?php
/**
 * ============================================
 * TRACKING.PHP - QUALIDADE MÁXIMA
 * ============================================
 * 
 * Recebe eventos do frontend e envia para Facebook CAPI
 * IMPORTANTE: Purchase é BLOQUEADO - só aceita via webhook
 */

date_default_timezone_set('America/Sao_Paulo');

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(200); exit; }

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/facebook_capi.php';

// ============================================
// LOG
// ============================================
function logTracking($message, $data = []) {
    $logDir = __DIR__ . '/logs';
    if (!is_dir($logDir)) @mkdir($logDir, 0755, true);
    
    $timestamp = date('Y-m-d H:i:s');
    $logLine = "[$timestamp] $message";
    if (!empty($data)) $logLine .= ' | ' . json_encode($data, JSON_UNESCAPED_UNICODE);
    @file_put_contents($logDir . '/tracking.log', $logLine . "\n", FILE_APPEND);
}

// ============================================
// ANTI-DUPLICAÇÃO
// ============================================
function isEventDuplicate($eventId) {
    if (empty($eventId)) return false;
    
    $cacheDir = __DIR__ . '/cache';
    if (!is_dir($cacheDir)) @mkdir($cacheDir, 0755, true);
    
    $cacheFile = $cacheDir . '/events_' . date('Y-m-d-H') . '.json';
    $cache = file_exists($cacheFile) ? json_decode(file_get_contents($cacheFile), true) ?: [] : [];
    
    if (isset($cache[$eventId])) return true;
    
    $cache[$eventId] = time();
    
    // Limpa eventos antigos
    $now = time();
    foreach ($cache as $id => $time) {
        if ($now - $time > 600) unset($cache[$id]);
    }
    
    @file_put_contents($cacheFile, json_encode($cache));
    return false;
}

// ============================================
// RECEBE DADOS
// ============================================
$input = json_decode(file_get_contents('php://input'), true);

if (!$input) {
    http_response_code(400);
    echo json_encode(['error' => 'Dados inválidos']);
    exit;
}

// Aceita 'event' OU 'event_name'
$event = $input['event'] ?? $input['event_name'] ?? null;
$eventId = $input['event_id'] ?? null;

if (empty($event)) {
    http_response_code(400);
    echo json_encode(['error' => 'Evento não especificado']);
    exit;
}

// ============================================
// BLOQUEIO: PURCHASE NÃO PERMITIDO
// ============================================
if ($event === 'Purchase') {
    logTracking("⛔ PURCHASE BLOQUEADO", ['event_id' => $eventId]);
    http_response_code(403);
    echo json_encode([
        'success' => false,
        'error' => 'Purchase deve ser enviado pelo webhook, não pelo frontend'
    ]);
    exit;
}

// Anti-duplicação
if (isEventDuplicate($eventId)) {
    logTracking("⚠️ DUPLICADO: $event", ['event_id' => $eventId]);
    echo json_encode(['success' => true, 'skipped' => true, 'reason' => 'duplicate']);
    exit;
}

// ============================================
// EXTRAI DADOS
// ============================================
$sourceUrl = $input['source_url'] ?? '';
$value = floatval($input['value'] ?? 0);
if ($value <= 0) $value = VALOR_PADRAO_REAIS;

$numItems = intval($input['num_items'] ?? 1);

// Cookies do Facebook - captura de múltiplas fontes
$fbc = $input['fbc'] ?? $_COOKIE['_fbc'] ?? null;
$fbp = $input['fbp'] ?? $_COOKIE['_fbp'] ?? null;
$fbclid = $input['fbclid'] ?? null;

// Valida fbc - aceita formatos fb.X.TIMESTAMP.CLICKID
if (!empty($fbc)) {
    if (strpos($fbc, 'fb.') !== 0) {
        logTracking("⚠️ FBC inválido ignorado: $fbc");
        $fbc = null;
    }
}

// Se não tem fbc mas tem fbclid, gera
if (empty($fbc) && !empty($fbclid)) {
    $fbc = 'fb.1.' . time() . '.' . $fbclid;
    logTracking("✓ FBC gerado do fbclid: $fbc");
}

// Gera fbp se não existir ou inválido
if (empty($fbp) || strpos($fbp, 'fb.') !== 0) {
    $fbp = 'fb.1.' . round(microtime(true) * 1000) . '.' . mt_rand(1000000000, 9999999999);
    logTracking("✓ FBP gerado: $fbp");
}

// IP do cliente
$clientIp = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['HTTP_X_REAL_IP'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
if (strpos($clientIp, ',') !== false) $clientIp = trim(explode(',', $clientIp)[0]);

$userData = [
    'email' => $input['email'] ?? null,
    'phone' => $input['phone'] ?? null,
    'name' => $input['name'] ?? null,
    'document' => $input['document'] ?? null,
    'fbc' => $fbc,
    'fbp' => $fbp,
    'fbclid' => $fbclid,
    'user_agent' => $input['user_agent'] ?? $_SERVER['HTTP_USER_AGENT'] ?? '',
    'ip' => $clientIp,
    'content_name' => $input['content_name'] ?? PRODUTO_NOME,
    'content_id' => $input['content_id'] ?? (defined('PRODUTO_ID') ? PRODUTO_ID : 'produto-001'),
    // UTMs
    'utm_source' => $input['utm_source'] ?? null,
    'utm_medium' => $input['utm_medium'] ?? null,
    'utm_campaign' => $input['utm_campaign'] ?? null,
    'utm_content' => $input['utm_content'] ?? null,
    'utm_term' => $input['utm_term'] ?? null
];

logTracking("📥 Recebido: $event", [
    'event_id' => $eventId,
    'value' => $value,
    'has_fbc' => !empty($fbc),
    'has_fbp' => !empty($fbp),
    'has_email' => !empty($input['email'])
]);

// ============================================
// ENVIA PARA FACEBOOK CAPI
// ============================================
$fb = new FacebookCAPI();
$result = ['success' => false];

switch ($event) {
    case 'PageView':
        $result = $fb->pageView($userData, $sourceUrl, $eventId);
        break;
    case 'ViewContent':
        $result = $fb->viewContent($userData, $value, $sourceUrl, $eventId);
        break;
    case 'AddToCart':
        $result = $fb->addToCart($userData, $value, $numItems, $sourceUrl, $eventId);
        break;
    case 'InitiateCheckout':
        $result = $fb->initiateCheckout($userData, $value, $numItems, $sourceUrl, $eventId);
        break;
    case 'AddPaymentInfo':
        $result = $fb->addPaymentInfo($userData, $value, $sourceUrl, $eventId);
        break;
    default:
        $result = ['success' => false, 'error' => 'Evento não suportado'];
}

logTracking("📤 $event enviado", ['success' => $result['success'] ?? false, 'event_id' => $eventId]);

echo json_encode($result);