<?php
/**
 * ============================================
 * STATUS API - VERIFICAÇÃO DE PAGAMENTO
 * ============================================
 * Verifica: Comprovantes + Gateway
 * Suporta: Umbrella, TechByNet, AllowPay, Genesys
 */

require_once __DIR__ . '/config.php';

setCorsHeaders();

// ============================================
// OBTER ID DA TRANSAÇÃO
// ============================================
$id = $_GET['id'] ?? $_GET['txid'] ?? $_POST['id'] ?? null;

// Tenta JSON body também
if (!$id) {
    $rawInput = file_get_contents('php://input');
    if (!empty($rawInput)) {
        $json = json_decode($rawInput, true);
        $id = $json['id'] ?? $json['txid'] ?? $json['pix_id'] ?? null;
    }
}

$checkAprovado = isset($_GET['check_aprovado']);

if (!$id) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'ID não informado']);
    exit;
}

$id = trim($id);

logCheckout('status.log', 'Verificação de status', [
    'id' => $id,
    'method' => $_SERVER['REQUEST_METHOD'],
    'check_aprovado' => $checkAprovado
]);

// ============================================
// FUNÇÃO: VERIFICAR COMPROVANTES
// ============================================
function verificarComprovantes($id) {
    // Caminhos possíveis para os arquivos de comprovantes
    $caminhosPossiveis = [
        __DIR__ . '/../comprovantes/',  // /comprovantes/ (raiz)
        __DIR__ . '/comprovantes/'       // /api/comprovantes/
    ];
    
    foreach ($caminhosPossiveis as $comprovantesDir) {
        // ========== VERIFICA APROVADOS ==========
        $aprovadosFile = $comprovantesDir . 'aprovados.json';
        if (file_exists($aprovadosFile)) {
            $aprovados = json_decode(file_get_contents($aprovadosFile), true) ?: [];
            foreach ($aprovados as $item) {
                if (($item['pix_id'] ?? '') === $id || ($item['txid'] ?? '') === $id || ($item['id'] ?? '') === $id) {
                    logCheckout('status.log', 'Comprovante APROVADO encontrado', [
                        'id' => $id,
                        'arquivo' => $aprovadosFile
                    ]);
                    return [
                        'found' => true,
                        'isPaid' => true,
                        'status' => 'paid',
                        'method' => 'manual'
                    ];
                }
            }
        }
        
        // ========== VERIFICA REJEITADOS ==========
        $rejeitadosFile = $comprovantesDir . 'rejeitados.json';
        if (file_exists($rejeitadosFile)) {
            $rejeitados = json_decode(file_get_contents($rejeitadosFile), true) ?: [];
            foreach ($rejeitados as $item) {
                if (($item['pix_id'] ?? '') === $id || ($item['txid'] ?? '') === $id || ($item['id'] ?? '') === $id) {
                    logCheckout('status.log', 'Comprovante REJEITADO encontrado', [
                        'id' => $id,
                        'motivo' => $item['motivo'] ?? ''
                    ]);
                    return [
                        'found' => true,
                        'isPaid' => false,
                        'rejeitado' => true,
                        'status' => 'rejected',
                        'motivo' => $item['motivo'] ?? 'Comprovante rejeitado',
                        'method' => 'manual'
                    ];
                }
            }
        }
        
        // ========== VERIFICA STATUS JSON INDIVIDUAL ==========
        $statusFile = $comprovantesDir . $id . '_status.json';
        if (file_exists($statusFile)) {
            $statusData = json_decode(file_get_contents($statusFile), true);
            if ($statusData) {
                if (($statusData['status'] ?? '') === 'approved') {
                    return [
                        'found' => true,
                        'isPaid' => true,
                        'status' => 'paid',
                        'method' => $statusData['auto_aprovado'] ? 'auto_ocr' : 'manual'
                    ];
                } elseif (($statusData['status'] ?? '') === 'rejected') {
                    return [
                        'found' => true,
                        'isPaid' => false,
                        'rejeitado' => true,
                        'status' => 'rejected',
                        'motivo' => $statusData['motivo'] ?? 'Comprovante rejeitado',
                        'method' => 'manual'
                    ];
                } elseif (($statusData['status'] ?? '') === 'pending') {
                    return [
                        'found' => true,
                        'isPaid' => false,
                        'status' => 'pending_review',
                        'comprovante_pendente' => true,
                        'method' => 'manual'
                    ];
                }
            }
        }
    }
    
    return ['found' => false];
}

// ============================================
// VERIFICAÇÃO SOMENTE DE APROVAÇÃO MANUAL
// ============================================
if ($checkAprovado) {
    $resultado = verificarComprovantes($id);
    
    if ($resultado['found']) {
        echo json_encode([
            'success' => true,
            'isPaid' => $resultado['isPaid'] ?? false,
            'aprovado' => $resultado['isPaid'] ?? false,
            'rejeitado' => $resultado['rejeitado'] ?? false,
            'status' => $resultado['status'] ?? 'pending',
            'motivo' => $resultado['motivo'] ?? null,
            'method' => $resultado['method'] ?? 'manual'
        ]);
    } else {
        echo json_encode([
            'success' => true,
            'isPaid' => false,
            'aprovado' => false,
            'status' => 'pending',
            'method' => 'manual'
        ]);
    }
    exit;
}

// ============================================
// VERIFICAÇÃO COMPLETA (COMPROVANTES + GATEWAY)
// ============================================
try {
    // 1. PRIMEIRO: Verifica comprovantes aprovados/rejeitados
    $comprovanteResult = verificarComprovantes($id);
    
    if ($comprovanteResult['found'] && ($comprovanteResult['isPaid'] ?? false)) {
        echo json_encode([
            'success' => true,
            'id' => $id,
            'isPaid' => true,
            'status' => 'paid',
            'method' => $comprovanteResult['method'] ?? 'manual'
        ]);
        exit;
    }
    
    if ($comprovanteResult['found'] && ($comprovanteResult['rejeitado'] ?? false)) {
        echo json_encode([
            'success' => true,
            'id' => $id,
            'isPaid' => false,
            'rejeitado' => true,
            'status' => 'rejected',
            'motivo' => $comprovanteResult['motivo'] ?? 'Comprovante rejeitado',
            'method' => 'manual'
        ]);
        exit;
    }
    
    // 2. SEGUNDO: Verifica via Gateway
    $gateway = getGatewayConfig();
    $gatewayName = getGatewayName();
    
    if (!$gateway) {
        throw new Exception("Gateway não configurado");
    }
    
    // ============================================
    // MONTA URL CONFORME O GATEWAY
    // ============================================
    $endpoint = $gateway['endpoints']['pix_status'] ?? '';
    $baseUrl = rtrim($gateway['api_url'], '/');
    
    if ($gatewayName === 'umbrella') {
        // Umbrella: GET /api/user/transactions/{id}
        $url = $baseUrl . '/api/user/transactions/' . urlencode($id);
    } elseif ($gatewayName === 'techbynet') {
        // TechByNet: GET {base_url}/{id}
        $url = $baseUrl . '/' . urlencode($id);
    } elseif ($gatewayName === 'allowpay') {
        // AllowPay: GET {base_url}/{id}
        $url = $baseUrl . '/' . urlencode($id);
    } elseif ($gatewayName === 'genesys') {
        // Genesys: GET /v1/transactions/{id}
        $url = $baseUrl . '/v1/transactions/' . urlencode($id);
    } else {
        // Padrão: GET {base_url}/{endpoint}/{id}
        $url = $baseUrl . rtrim($endpoint, '/') . '/' . urlencode($id);
    }
    
    // ============================================
    // MONTA HEADERS DE AUTENTICAÇÃO
    // ============================================
    $authType = $gateway['auth_type'] ?? 'apikey';
    $headers = [
        'Content-Type: application/json',
        'Accept: application/json',
        'User-Agent: ' . ($gateway['user_agent'] ?: 'SmartCheckout/2.0')
    ];
    
    if ($authType === 'basic') {
        // Basic Auth: base64(api_key:company_id)
        $credentials = $gateway['api_key'] . ':' . ($gateway['company_id'] ?? '');
        $headers[] = 'Authorization: Basic ' . base64_encode($credentials);
    } elseif ($authType === 'header') {
        // Header customizado (Genesys usa api-secret)
        $authHeader = $gateway['auth_header'] ?? 'api-secret';
        $headers[] = $authHeader . ': ' . $gateway['api_key'];
    } else {
        // API Key padrão (x-api-key)
        $authHeader = $gateway['auth_header'] ?? 'x-api-key';
        $authPrefix = $gateway['auth_prefix'] ?? '';
        $headers[] = $authHeader . ': ' . $authPrefix . $gateway['api_key'];
    }
    
    logCheckout('status.log', 'Consultando gateway', [
        'gateway' => $gatewayName,
        'url' => $url,
        'id' => $id
    ]);
    
    // ============================================
    // FAZ REQUISIÇÃO CURL
    // ============================================
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 15,
        CURLOPT_SSL_VERIFYPEER => true
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    logCheckout('status.log', 'Resposta do gateway', [
        'http_code' => $httpCode,
        'curl_error' => $curlError ?: 'nenhum',
        'response_preview' => substr($response, 0, 500)
    ]);
    
    if ($curlError) {
        throw new Exception("Erro de conexão: $curlError");
    }
    
    $data = json_decode($response, true);
    
    if (!$data && $httpCode !== 200) {
        throw new Exception("Resposta inválida do gateway (HTTP $httpCode)");
    }
    
    // ============================================
    // EXTRAI STATUS DA RESPOSTA
    // ============================================
    $status = null;
    
    // Tenta usando o path configurado no gateway
    $statusField = $gateway['status_field'] ?? 'data.status';
    $status = extractByPath($data, $statusField);
    
    // Fallbacks comuns se não encontrou
    if (!$status) {
        $statusPaths = [
            'data.data[0].status',  // Umbrella
            'data.data.status',     // TechByNet
            'data.status',
            'status',               // AllowPay, Genesys
            'transaction.status'
        ];
        
        foreach ($statusPaths as $path) {
            $status = extractByPath($data, $path);
            if ($status) {
                logCheckout('status.log', "Status encontrado em: $path", ['status' => $status]);
                break;
            }
        }
    }
    
    // ============================================
    // PROCESSA RESULTADO
    // ============================================
    if ($status) {
        $isPaid = isPaidStatus($status);
        $mappedStatus = mapStatus($status);
        
        // Se gateway diz pendente, verifica novamente comprovantes
        if (!$isPaid) {
            $comprovanteCheck = verificarComprovantes($id);
            if ($comprovanteCheck['found'] && ($comprovanteCheck['isPaid'] ?? false)) {
                logCheckout('status.log', 'Gateway pendente, mas comprovante aprovado', ['id' => $id]);
                echo json_encode([
                    'success' => true,
                    'gateway' => $gateway['name'],
                    'id' => $id,
                    'status' => 'paid',
                    'apiStatus' => $status,
                    'isPaid' => true,
                    'method' => 'manual_approved'
                ]);
                exit;
            }
        }
        
        logCheckout('status.log', 'Status final', [
            'api_status' => $status,
            'mapped_status' => $mappedStatus,
            'is_paid' => $isPaid
        ]);
        
        echo json_encode([
            'success' => true,
            'gateway' => $gateway['name'],
            'id' => $id,
            'status' => $mappedStatus,
            'apiStatus' => $status,
            'isPaid' => $isPaid,
            'method' => 'gateway',
            'comprovante_pendente' => $comprovanteResult['comprovante_pendente'] ?? false
        ]);
        exit;
    }
    
    // ============================================
    // FALLBACK: Se não conseguiu status do gateway
    // ============================================
    logCheckout('status.log', 'Não foi possível extrair status do gateway', [
        'http_code' => $httpCode,
        'response_keys' => is_array($data) ? array_keys($data) : 'não é array'
    ]);
    
    // Verifica comprovantes como fallback final
    $comprovanteCheck = verificarComprovantes($id);
    if ($comprovanteCheck['found'] && ($comprovanteCheck['isPaid'] ?? false)) {
        echo json_encode([
            'success' => true,
            'id' => $id,
            'isPaid' => true,
            'status' => 'paid',
            'method' => 'manual_fallback'
        ]);
        exit;
    }
    
    // Retorna pendente
    echo json_encode([
        'success' => true,
        'id' => $id,
        'isPaid' => false,
        'status' => 'pending',
        'method' => 'unknown',
        'comprovante_pendente' => $comprovanteResult['comprovante_pendente'] ?? false
    ]);
    
} catch (Exception $e) {
    logCheckout('status.log', 'ERRO: ' . $e->getMessage());
    
    // Em caso de erro, tenta verificação de comprovantes
    $comprovanteCheck = verificarComprovantes($id);
    if ($comprovanteCheck['found'] && ($comprovanteCheck['isPaid'] ?? false)) {
        echo json_encode([
            'success' => true,
            'id' => $id,
            'isPaid' => true,
            'status' => 'paid',
            'method' => 'manual_fallback'
        ]);
        exit;
    }
    
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'id' => $id,
        'isPaid' => false,
        'status' => 'error'
    ]);
}