/**
 * ============================================
 * SMART TRACKING v7 - FBC/FBP MELHORADO
 * ============================================
 * 
 * MELHORIAS:
 * - Captura fbc/fbp de múltiplas fontes
 * - Salva em cookie E localStorage
 * - Gera automaticamente se não existir
 * - Endpoint configurável
 * 
 * INSTALAÇÃO:
 * <script src="/seguro/consulta/tarifa/api/smart-tracking.js"></script>
 */

(function() {
    'use strict';
    
    // ============================================
    // CONFIGURAÇÃO - AJUSTE O ENDPOINT AQUI
    // ============================================
    var CONFIG = {
        pixelId: '2357252418054828',
        trackingEndpoint: '/seguro/consulta/tarifa/api/tracking.php',
        defaultValue: 59.90,
        defaultCurrency: 'BRL',
        defaultContentName: 'Tarifa BrasiCred',
        defaultContentId: 'tarifa-001',
        debug: false,
        // Purchase é enviado APENAS pelo webhook
        blockPurchaseFromFrontend: true
    };
    
    var sentEventIds = {};
    
    // ============================================
    // UTILITÁRIOS
    // ============================================
    function log(msg, data) {
        if (CONFIG.debug) console.log('[SmartTracking]', msg, data || '');
    }
    
    function warn(msg, data) {
        console.warn('[SmartTracking]', msg, data || '');
    }
    
    /**
     * Obtém cookie por nome
     */
    function getCookie(name) {
        try {
            var value = '; ' + document.cookie;
            var parts = value.split('; ' + name + '=');
            if (parts.length === 2) {
                return parts.pop().split(';').shift();
            }
        } catch(e) {}
        return null;
    }
    
    /**
     * Define cookie com duração de 90 dias
     */
    function setCookie(name, value, days) {
        try {
            days = days || 90;
            var expires = new Date(Date.now() + days * 864e5).toUTCString();
            document.cookie = name + '=' + value + '; expires=' + expires + '; path=/; SameSite=Lax';
        } catch(e) {}
    }
    
    /**
     * Obtém do localStorage
     */
    function getLocal(name) {
        try {
            return localStorage.getItem(name);
        } catch(e) {
            return null;
        }
    }
    
    /**
     * Salva no localStorage
     */
    function setLocal(name, value) {
        try {
            localStorage.setItem(name, value);
        } catch(e) {}
    }
    
    /**
     * Obtém parâmetro da URL
     */
    function getUrlParam(param) {
        try {
            var urlParams = new URLSearchParams(window.location.search);
            return urlParams.get(param);
        } catch(e) {
            // Fallback para browsers antigos
            var regex = new RegExp('[?&]' + param + '=([^&#]*)');
            var results = regex.exec(window.location.search);
            return results ? decodeURIComponent(results[1]) : null;
        }
    }
    
    // ============================================
    // FBC - Facebook Click ID
    // ============================================
    function getFbc() {
        var fbc = null;
        
        // 1. Tenta cookie _fbc
        fbc = getCookie('_fbc');
        if (fbc && fbc.indexOf('fb.') === 0) {
            log('FBC do cookie:', fbc);
            return fbc;
        }
        
        // 2. Tenta localStorage
        fbc = getLocal('_fbc');
        if (fbc && fbc.indexOf('fb.') === 0) {
            log('FBC do localStorage:', fbc);
            // Salva no cookie também
            setCookie('_fbc', fbc);
            return fbc;
        }
        
        // 3. Gera a partir do fbclid na URL
        var fbclid = getUrlParam('fbclid');
        if (fbclid) {
            fbc = 'fb.1.' + Date.now() + '.' + fbclid;
            log('FBC gerado do fbclid:', fbc);
            // Salva em ambos
            setCookie('_fbc', fbc);
            setLocal('_fbc', fbc);
            return fbc;
        }
        
        // 4. Tenta fbclid do localStorage (pode ter sido salvo antes)
        fbclid = getLocal('fbclid');
        if (fbclid) {
            fbc = 'fb.1.' + Date.now() + '.' + fbclid;
            setCookie('_fbc', fbc);
            setLocal('_fbc', fbc);
            return fbc;
        }
        
        log('FBC não disponível');
        return null;
    }
    
    // ============================================
    // FBP - Facebook Browser ID
    // ============================================
    function getFbp() {
        var fbp = null;
        
        // 1. Tenta cookie _fbp (definido pelo Pixel)
        fbp = getCookie('_fbp');
        if (fbp && fbp.indexOf('fb.') === 0) {
            log('FBP do cookie:', fbp);
            // Salva no localStorage também
            setLocal('_fbp', fbp);
            return fbp;
        }
        
        // 2. Tenta localStorage
        fbp = getLocal('_fbp');
        if (fbp && fbp.indexOf('fb.') === 0) {
            log('FBP do localStorage:', fbp);
            // Salva no cookie também
            setCookie('_fbp', fbp);
            return fbp;
        }
        
        // 3. Gera novo FBP
        var randomId = Math.floor(Math.random() * 10000000000);
        fbp = 'fb.1.' + Date.now() + '.' + randomId;
        log('FBP gerado:', fbp);
        
        // Salva em ambos
        setCookie('_fbp', fbp);
        setLocal('_fbp', fbp);
        
        return fbp;
    }
    
    // ============================================
    // SALVA FBCLID PARA USO FUTURO
    // ============================================
    function saveFbclid() {
        var fbclid = getUrlParam('fbclid');
        if (fbclid) {
            setLocal('fbclid', fbclid);
            log('FBCLID salvo:', fbclid);
        }
    }
    
    // ============================================
    // COLETA DADOS DO USUÁRIO
    // ============================================
    function getUserData() {
        var fbc = getFbc();
        var fbp = getFbp();
        
        return {
            fbc: fbc,
            fbp: fbp,
            fbclid: getUrlParam('fbclid') || getLocal('fbclid'),
            user_agent: navigator.userAgent,
            source_url: window.location.href,
            // UTMs
            utm_source: getUrlParam('utm_source'),
            utm_medium: getUrlParam('utm_medium'),
            utm_campaign: getUrlParam('utm_campaign'),
            utm_content: getUrlParam('utm_content'),
            utm_term: getUrlParam('utm_term'),
            sck: getUrlParam('sck'),
            src: getUrlParam('src')
        };
    }
    
    // ============================================
    // GERA EVENT ID ÚNICO
    // ============================================
    function generateEventId(eventName) {
        var random = Math.random().toString(36).substring(2, 10);
        return eventName.toLowerCase() + '_' + Date.now() + '_' + random;
    }
    
    // ============================================
    // VERIFICA DUPLICAÇÃO
    // ============================================
    function isDuplicate(eventId) {
        if (sentEventIds[eventId]) {
            return true;
        }
        sentEventIds[eventId] = Date.now();
        
        // Limpa eventos antigos (mais de 1 hora)
        var oneHourAgo = Date.now() - 3600000;
        for (var id in sentEventIds) {
            if (sentEventIds[id] < oneHourAgo) {
                delete sentEventIds[id];
            }
        }
        
        return false;
    }
    
    // ============================================
    // ENVIA PARA SERVIDOR (CAPI)
    // ============================================
    function sendToServer(eventName, eventData, eventId) {
        // Anti-duplicação
        if (isDuplicate(eventId)) {
            log('Evento duplicado ignorado:', eventId);
            return;
        }
        
        var userData = getUserData();
        
        var payload = {
            event_name: eventName,
            event_id: eventId,
            event_time: Math.floor(Date.now() / 1000),
            source_url: window.location.href,
            // FACEBOOK - ESSENCIAL PARA MATCH RATE
            fbc: userData.fbc,
            fbp: userData.fbp,
            fbclid: userData.fbclid,
            user_agent: userData.user_agent,
            // UTMs
            utm_source: userData.utm_source,
            utm_medium: userData.utm_medium,
            utm_campaign: userData.utm_campaign,
            utm_content: userData.utm_content,
            utm_term: userData.utm_term,
            sck: userData.sck,
            src: userData.src,
            // Dados do evento
            value: eventData.value || CONFIG.defaultValue,
            currency: eventData.currency || CONFIG.defaultCurrency,
            content_name: eventData.content_name || CONFIG.defaultContentName,
            content_ids: eventData.content_ids || [CONFIG.defaultContentId],
            content_type: eventData.content_type || 'product',
            num_items: eventData.num_items || 1
        };
        
        log('Enviando para servidor:', payload);
        
        // Usa sendBeacon se disponível (mais confiável)
        if (navigator.sendBeacon) {
            try {
                var blob = new Blob([JSON.stringify(payload)], { type: 'application/json' });
                navigator.sendBeacon(CONFIG.trackingEndpoint, blob);
                log('Enviado via sendBeacon');
            } catch(e) {
                sendViaFetch(payload);
            }
        } else {
            sendViaFetch(payload);
        }
    }
    
    function sendViaFetch(payload) {
        try {
            fetch(CONFIG.trackingEndpoint, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload),
                keepalive: true
            }).then(function(response) {
                log('Resposta do servidor:', response.status);
            }).catch(function(err) {
                warn('Erro ao enviar:', err);
            });
        } catch(e) {
            warn('Fetch não suportado');
        }
    }
    
    // ============================================
    // INTERCEPTA FBQ
    // ============================================
    function interceptFbq() {
        if (typeof window.fbq === 'undefined') {
            log('fbq não encontrado, tentando novamente...');
            setTimeout(interceptFbq, 500);
            return;
        }
        
        var originalFbq = window.fbq;
        
        window.fbq = function() {
            var args = Array.prototype.slice.call(arguments);
            
            // Chama fbq original primeiro
            originalFbq.apply(this, args);
            
            // Intercepta eventos 'track'
            if (args[0] === 'track' && args[1]) {
                var eventName = args[1];
                var eventData = args[2] || {};
                
                // BLOQUEIA Purchase no frontend
                if (eventName === 'Purchase' && CONFIG.blockPurchaseFromFrontend) {
                    warn('Purchase BLOQUEADO no frontend (enviado apenas pelo webhook)');
                    return;
                }
                
                // Gera event_id
                var eventId = generateEventId(eventName);
                
                // Eventos para enviar ao servidor
                var eventsToTrack = ['PageView', 'ViewContent', 'AddToCart', 'InitiateCheckout', 'AddPaymentInfo'];
                
                if (eventsToTrack.indexOf(eventName) !== -1) {
                    log('Interceptado:', eventName);
                    sendToServer(eventName, eventData, eventId);
                }
            }
        };
        
        // Copia propriedades
        for (var prop in originalFbq) {
            if (originalFbq.hasOwnProperty(prop)) {
                window.fbq[prop] = originalFbq[prop];
            }
        }
        
        log('fbq interceptado com sucesso');
    }
    
    // ============================================
    // INICIALIZAÇÃO
    // ============================================
    function init() {
        log('Inicializando SmartTracking v7...');
        
        // Salva fbclid se existir na URL
        saveFbclid();
        
        // Gera/captura fbc e fbp imediatamente
        var fbc = getFbc();
        var fbp = getFbp();
        
        log('FBC capturado:', fbc);
        log('FBP capturado:', fbp);
        
        // Intercepta fbq
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', function() {
                setTimeout(interceptFbq, 100);
            });
        } else {
            setTimeout(interceptFbq, 100);
        }
    }
    
    // ============================================
    // API PÚBLICA
    // ============================================
    window.SmartTracking = {
        getFbc: getFbc,
        getFbp: getFbp,
        getUserData: getUserData,
        sendEvent: sendToServer,
        getConfig: function() { return CONFIG; }
    };
    
    // Inicia
    init();
    
})();