<?php
/**
 * ============================================
 * FACEBOOK CAPI - QUALIDADE MÁXIMA DE EVENTOS
 * ============================================
 * Versão: 3.0
 * 
 * MELHORIAS:
 * - Captura nome/sobrenome de múltiplas fontes
 * - Melhor normalização de dados antes do hash
 * - Logs detalhados de cobertura
 * - Anti-duplicação por txId
 */

class FacebookCAPI {
    
    private $pixelId;
    private $accessToken;
    private $testEventCode;
    private $siteUrl;
    
    public function __construct() {
        $this->pixelId = defined('FB_PIXEL_ID') ? FB_PIXEL_ID : '2357252418054828';
        $this->accessToken = defined('FB_ACCESS_TOKEN') ? FB_ACCESS_TOKEN : 'EAAMchqNcUmMBQiBm6tuKZBZAkDYffLazN5u3wApky1afxti4SXtLDs5eW6vN6Pe6Jvss0KZAGvXiZCHeSc4Q2ZAqGm8aBwNbMVn7YWbcYuOOAHjvwe95DnKMq8lWawz2WwuOJcQ5Rrs259LdD6EZCZBKpNcyZBIDgqZCtT2R8Cd8ggpmdryqZBnFTGVpxSw2GmjQZDZD';
        $this->testEventCode = defined('FB_TEST_EVENT_CODE') ? FB_TEST_EVENT_CODE : '';
        $this->siteUrl = defined('SITE_URL') ? SITE_URL : $this->detectSiteUrl();
    }
    
    private function detectSiteUrl() {
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        return $protocol . '://' . $host;
    }
    
    /**
     * Anti-duplicação: verifica se já enviou Purchase para este txId
     */
    private function isPurchaseAlreadySent($txId) {
        if (empty($txId)) return false;
        
        $cacheDir = __DIR__ . '/cache';
        if (!is_dir($cacheDir)) @mkdir($cacheDir, 0755, true);
        
        $cacheFile = $cacheDir . '/purchase_sent.json';
        $cache = file_exists($cacheFile) ? json_decode(file_get_contents($cacheFile), true) : [];
        
        if (isset($cache[$txId])) {
            $this->log("⚠️ Purchase já enviado para txId: $txId");
            return true;
        }
        return false;
    }
    
    private function markPurchaseSent($txId) {
        if (empty($txId)) return;
        
        $cacheDir = __DIR__ . '/cache';
        $cacheFile = $cacheDir . '/purchase_sent.json';
        
        $cache = file_exists($cacheFile) ? json_decode(file_get_contents($cacheFile), true) : [];
        $cache[$txId] = date('Y-m-d H:i:s');
        
        // Limpa cache antigo (24h)
        $oneDayAgo = date('Y-m-d H:i:s', strtotime('-24 hours'));
        foreach ($cache as $id => $date) {
            if ($date < $oneDayAgo) unset($cache[$id]);
        }
        
        @file_put_contents($cacheFile, json_encode($cache, JSON_PRETTY_PRINT));
    }
    
    /**
     * ============================================
     * PREPARAÇÃO DE USER_DATA - QUALIDADE MÁXIMA
     * ============================================
     * Captura dados de múltiplas fontes para maximizar cobertura
     */
    private function prepareUserData($data) {
        $userData = [];
        $coverage = []; // Para log de cobertura
        
        // ========== EMAIL ==========
        $email = $this->extractValue($data, [
            'email',
            'customer.email',
            'cliente.email',
            'user.email'
        ]);
        
        if ($email && filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $emailNormalized = strtolower(trim($email));
            $userData['em'] = [hash('sha256', $emailNormalized)];
            $coverage['email'] = true;
        }
        
        // ========== TELEFONE ==========
        $phone = $this->extractValue($data, [
            'phone',
            'telefone',
            'celular',
            'customer.phone',
            'customer.telefone',
            'customer.cellphone',
            'cliente.telefone',
            'cliente.celular'
        ]);
        
        if ($phone) {
            $phoneClean = preg_replace('/\D/', '', $phone);
            // Adiciona código do país se necessário
            if (strlen($phoneClean) <= 11 && !str_starts_with($phoneClean, '55')) {
                $phoneClean = '55' . $phoneClean;
            }
            if (strlen($phoneClean) >= 12) {
                $userData['ph'] = [hash('sha256', $phoneClean)];
                $coverage['phone'] = true;
            }
        }
        
        // ========== NOME E SOBRENOME ==========
        // Tenta várias fontes para maximizar cobertura
        $firstName = null;
        $lastName = null;
        
        // Fonte 1: Campos separados
        $firstName = $this->extractValue($data, [
            'first_name',
            'firstName',
            'primeiro_nome',
            'customer.first_name',
            'customer.firstName'
        ]);
        
        $lastName = $this->extractValue($data, [
            'last_name',
            'lastName',
            'sobrenome',
            'ultimo_nome',
            'customer.last_name',
            'customer.lastName'
        ]);
        
        // Fonte 2: Nome completo (divide em primeiro e último)
        if (!$firstName) {
            $fullName = $this->extractValue($data, [
                'name',
                'nome',
                'customer.name',
                'customer.nome',
                'cliente.nome',
                'payer.name'
            ]);
            
            if ($fullName) {
                $fullName = trim($fullName);
                // Remove títulos comuns
                $fullName = preg_replace('/^(sr\.?|sra\.?|dr\.?|dra\.?)\s+/i', '', $fullName);
                
                // Divide por espaços
                $nameParts = preg_split('/\s+/', $fullName);
                $nameParts = array_filter($nameParts); // Remove vazios
                $nameParts = array_values($nameParts); // Reindexa
                
                if (count($nameParts) >= 1) {
                    $firstName = $nameParts[0];
                }
                
                if (count($nameParts) >= 2) {
                    // Pega o ÚLTIMO nome (não o segundo)
                    $lastName = end($nameParts);
                }
            }
        }
        
        // Hash do primeiro nome
        if ($firstName && strlen($firstName) >= 2) {
            // Normaliza: lowercase, remove acentos
            $fnNormalized = $this->normalizeForHash($firstName);
            if ($fnNormalized) {
                $userData['fn'] = [hash('sha256', $fnNormalized)];
                $coverage['first_name'] = $firstName;
            }
        }
        
        // Hash do sobrenome
        if ($lastName && strlen($lastName) >= 2) {
            $lnNormalized = $this->normalizeForHash($lastName);
            if ($lnNormalized) {
                $userData['ln'] = [hash('sha256', $lnNormalized)];
                $coverage['last_name'] = $lastName;
            }
        }
        
        // ========== CPF (external_id) ==========
        $cpf = $this->extractValue($data, [
            'document',
            'cpf',
            'customer.document',
            'customer.cpf',
            'customer.tax_id',
            'cliente.cpf',
            'payer.document'
        ]);
        
        if ($cpf) {
            $cpfClean = preg_replace('/\D/', '', $cpf);
            if (strlen($cpfClean) === 11) {
                $userData['external_id'] = [hash('sha256', $cpfClean)];
                $coverage['cpf'] = true;
            }
        }
        
        // ========== DATA DE NASCIMENTO ==========
        $dob = $this->extractValue($data, [
            'birth_date',
            'birthdate',
            'data_nascimento',
            'nascimento',
            'customer.birth_date',
            'customer.birthdate'
        ]);
        
        if ($dob) {
            $dobClean = preg_replace('/\D/', '', $dob);
            if (strlen($dobClean) === 8) {
                // Detecta formato
                $year = substr($dobClean, 0, 4);
                if ((int)$year < 1900 || (int)$year > 2100) {
                    // É DDMMYYYY, converte para YYYYMMDD
                    $dobClean = substr($dobClean, 4, 4) . substr($dobClean, 2, 2) . substr($dobClean, 0, 2);
                }
                $userData['db'] = [hash('sha256', $dobClean)];
                $coverage['birth_date'] = true;
            }
        }
        
        // ========== GÊNERO ==========
        $gender = $this->extractValue($data, [
            'gender',
            'genero',
            'sexo',
            'customer.gender'
        ]);
        
        if ($gender) {
            $g = strtolower(substr(trim($gender), 0, 1));
            if ($g === 'm' || $g === 'f') {
                $userData['ge'] = [hash('sha256', $g)];
                $coverage['gender'] = true;
            }
        }
        
        // ========== CIDADE ==========
        $city = $this->extractValue($data, [
            'city',
            'cidade',
            'customer.city',
            'address.city'
        ]);
        
        if ($city) {
            $cityNormalized = $this->normalizeForHash($city);
            if ($cityNormalized) {
                $userData['ct'] = [hash('sha256', $cityNormalized)];
                $coverage['city'] = true;
            }
        }
        
        // ========== ESTADO ==========
        $state = $this->extractValue($data, [
            'state',
            'estado',
            'uf',
            'customer.state',
            'address.state'
        ]);
        
        if ($state) {
            $stateNormalized = strtolower(trim($state));
            if (strlen($stateNormalized) === 2) {
                $userData['st'] = [hash('sha256', $stateNormalized)];
                $coverage['state'] = true;
            }
        }
        
        // ========== CEP ==========
        $zip = $this->extractValue($data, [
            'zip',
            'cep',
            'postal_code',
            'customer.zip',
            'address.zip'
        ]);
        
        if ($zip) {
            $zipClean = preg_replace('/\D/', '', $zip);
            if (strlen($zipClean) >= 5) {
                // Facebook espera 5 dígitos
                $userData['zp'] = [hash('sha256', substr($zipClean, 0, 5))];
                $coverage['zip'] = true;
            }
        }
        
        // ========== PAÍS (sempre BR) ==========
        $userData['country'] = [hash('sha256', 'br')];
        
        // ========== IP ==========
        $ip = $this->extractValue($data, [
            'ip',
            'customer_ip',
            'client_ip'
        ]);
        
        if (!$ip) {
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['HTTP_X_REAL_IP'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
        }
        
        if ($ip) {
            if (strpos($ip, ',') !== false) {
                $ip = trim(explode(',', $ip)[0]);
            }
            $userData['client_ip_address'] = $ip;
            $coverage['ip'] = true;
        }
        
        // ========== USER AGENT ==========
        $ua = $data['user_agent'] ?? $_SERVER['HTTP_USER_AGENT'] ?? '';
        if ($ua) {
            $userData['client_user_agent'] = $ua;
        }
        
        // ========== FBC (Facebook Click ID) ==========
        $fbc = $data['fbc'] ?? null;
        if ($fbc) {
            $fbcParts = explode('.', $fbc);
            if (count($fbcParts) >= 4) {
                $userData['fbc'] = $fbc;
                $coverage['fbc'] = true;
            }
        }
        
        // ========== FBP (Facebook Browser ID) ==========
        $fbp = $data['fbp'] ?? null;
        if ($fbp) {
            $userData['fbp'] = $fbp;
            $coverage['fbp'] = true;
        }
        
        // Log de cobertura para debug
        $this->log("📊 Cobertura de dados", $coverage);
        
        return $userData;
    }
    
    /**
     * Extrai valor de múltiplas fontes possíveis
     */
    private function extractValue($data, $keys) {
        foreach ($keys as $key) {
            // Suporta notação de ponto: "customer.email"
            if (strpos($key, '.') !== false) {
                $parts = explode('.', $key);
                $value = $data;
                foreach ($parts as $part) {
                    if (is_array($value) && isset($value[$part])) {
                        $value = $value[$part];
                    } else {
                        $value = null;
                        break;
                    }
                }
                if ($value !== null && $value !== '') {
                    return $value;
                }
            } else {
                if (isset($data[$key]) && $data[$key] !== null && $data[$key] !== '') {
                    return $data[$key];
                }
            }
        }
        return null;
    }
    
    /**
     * Normaliza string para hash (Facebook requirements)
     * - Lowercase
     * - Remove acentos
     * - Remove espaços extras
     */
    private function normalizeForHash($str) {
        if (!$str) return null;
        
        $str = trim($str);
        $str = strtolower($str);
        
        // Remove acentos
        $str = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $str);
        
        // Remove caracteres não alfabéticos (exceto espaço)
        $str = preg_replace('/[^a-z\s]/', '', $str);
        
        // Remove espaços extras
        $str = preg_replace('/\s+/', '', $str);
        
        return $str ?: null;
    }
    
    /**
     * ============================================
     * ENVIA PURCHASE
     * ============================================
     */
    public function sendPurchase($txId, $valorReais, $utmData = []) {
        $this->log("=== PURCHASE ===", ['txId' => $txId, 'valor' => $valorReais]);
        
        // Anti-duplicação
        if ($this->isPurchaseAlreadySent($txId)) {
            return [
                'success' => true,
                'skipped' => true,
                'reason' => 'already_sent',
                'txId' => $txId
            ];
        }
        
        // Prepara user_data
        $userData = $this->prepareUserData($utmData);
        
        // Event ID único
        $eventId = 'purchase_' . $txId . '_' . time();
        
        // URL de origem
        $sourceUrl = $utmData['source_url'] ?? $this->siteUrl;
        
        // Dados do evento
        $eventData = [
            'event_name' => 'Purchase',
            'event_time' => time(),
            'event_id' => $eventId,
            'event_source_url' => $sourceUrl,
            'action_source' => 'website',
            'user_data' => $userData,
            'custom_data' => [
                'currency' => 'BRL',
                'value' => floatval($valorReais),
                'content_type' => 'product',
                'content_name' => $utmData['produto'] ?? $utmData['content_name'] ?? (defined('PRODUTO_NOME') ? PRODUTO_NOME : 'Produto'),
                'content_ids' => [$utmData['produto_id'] ?? $utmData['content_id'] ?? (defined('PRODUTO_ID') ? PRODUTO_ID : 'produto-001')],
                'num_items' => intval($utmData['cotas'] ?? $utmData['num_items'] ?? 1),
                'order_id' => $txId
            ]
        ];
        
        // Payload
        $payload = ['data' => [$eventData]];
        
        if (!empty($this->testEventCode)) {
            $payload['test_event_code'] = $this->testEventCode;
        }
        
        // URL da API
        $apiVersion = defined('FB_API_VERSION') ? FB_API_VERSION : 'v18.0';
        $url = "https://graph.facebook.com/{$apiVersion}/{$this->pixelId}/events?access_token={$this->accessToken}";
        
        $this->log("📤 Enviando Purchase", [
            'event_id' => $eventId,
            'value' => $valorReais,
            'has_fn' => isset($userData['fn']),
            'has_ln' => isset($userData['ln']),
            'has_em' => isset($userData['em']),
            'has_ph' => isset($userData['ph']),
            'has_cpf' => isset($userData['external_id']),
            'has_fbc' => isset($userData['fbc']),
            'has_fbp' => isset($userData['fbp'])
        ]);
        
        // Envia
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_SSL_VERIFYPEER => true
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        $responseData = json_decode($response, true);
        $success = $httpCode >= 200 && $httpCode < 300;
        
        if ($success) {
            $this->markPurchaseSent($txId);
        }
        
        $this->log("📥 Resposta Facebook", [
            'success' => $success,
            'http_code' => $httpCode,
            'events_received' => $responseData['events_received'] ?? 0,
            'messages' => $responseData['messages'] ?? [],
            'error' => $curlError ?: null
        ]);
        
        return [
            'success' => $success,
            'event_id' => $eventId,
            'http_code' => $httpCode,
            'events_received' => $responseData['events_received'] ?? 0,
            'response' => $responseData
        ];
    }
    
    /**
     * Evento genérico
     */
    private function sendGenericEvent($eventName, $data, $value, $numItems, $sourceUrl, $eventId = null) {
        $userData = $this->prepareUserData($data);
        
        if (empty($eventId)) {
            $eventId = strtolower($eventName) . '_' . bin2hex(random_bytes(8)) . '_' . time();
        }
        
        $eventData = [
            'event_name' => $eventName,
            'event_time' => time(),
            'event_id' => $eventId,
            'event_source_url' => $sourceUrl ?: $this->siteUrl,
            'action_source' => 'website',
            'user_data' => $userData,
            'custom_data' => [
                'currency' => 'BRL',
                'value' => floatval($value),
                'content_type' => 'product',
                'content_name' => $data['content_name'] ?? (defined('PRODUTO_NOME') ? PRODUTO_NOME : 'Produto'),
                'content_ids' => [$data['content_id'] ?? (defined('PRODUTO_ID') ? PRODUTO_ID : 'produto-001')],
                'num_items' => intval($numItems)
            ]
        ];
        
        $payload = ['data' => [$eventData]];
        
        if (!empty($this->testEventCode)) {
            $payload['test_event_code'] = $this->testEventCode;
        }
        
        $apiVersion = defined('FB_API_VERSION') ? FB_API_VERSION : 'v18.0';
        $url = "https://graph.facebook.com/{$apiVersion}/{$this->pixelId}/events?access_token={$this->accessToken}";
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        $this->log("$eventName enviado", [
            'event_id' => $eventId,
            'http_code' => $httpCode,
            'value' => $value
        ]);
        
        return [
            'success' => $httpCode >= 200 && $httpCode < 300,
            'event_id' => $eventId,
            'http_code' => $httpCode,
            'response' => json_decode($response, true)
        ];
    }
    
    // Métodos de conveniência
    public function pageView($data, $sourceUrl = '', $eventId = null) {
        return $this->sendGenericEvent('PageView', $data, 0, 1, $sourceUrl, $eventId);
    }
    
    public function viewContent($data, $value = 0, $sourceUrl = '', $eventId = null) {
        return $this->sendGenericEvent('ViewContent', $data, $value, 1, $sourceUrl, $eventId);
    }
    
    public function addToCart($data, $value, $numItems = 1, $sourceUrl = '', $eventId = null) {
        return $this->sendGenericEvent('AddToCart', $data, $value, $numItems, $sourceUrl, $eventId);
    }
    
    public function initiateCheckout($data, $value, $numItems = 1, $sourceUrl = '', $eventId = null) {
        return $this->sendGenericEvent('InitiateCheckout', $data, $value, $numItems, $sourceUrl, $eventId);
    }
    
    public function addPaymentInfo($data, $value, $sourceUrl = '', $eventId = null) {
        return $this->sendGenericEvent('AddPaymentInfo', $data, $value, 1, $sourceUrl, $eventId);
    }
    
    public function purchase($data, $value, $orderId, $numItems = 1, $sourceUrl = '', $eventId = null) {
        $utmData = array_merge($data, [
            'produto' => $data['content_name'] ?? null,
            'produto_id' => $data['content_id'] ?? null,
            'cotas' => $numItems,
            'source_url' => $sourceUrl
        ]);
        return $this->sendPurchase($orderId, $value, $utmData);
    }
    
    public function customEvent($eventName, $data, $customData = [], $sourceUrl = '', $eventId = null) {
        return $this->sendGenericEvent($eventName, $data, $customData['value'] ?? 0, 1, $sourceUrl, $eventId);
    }
    
    /**
     * Log
     */
    private function log($message, $data = []) {
        $logDir = __DIR__ . '/logs';
        if (!is_dir($logDir)) @mkdir($logDir, 0755, true);
        
        $timestamp = date('Y-m-d H:i:s');
        $logLine = "[$timestamp] $message";
        if (!empty($data)) {
            $logLine .= ' | ' . json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        }
        
        @file_put_contents($logDir . '/facebook_capi.log', $logLine . "\n", FILE_APPEND | LOCK_EX);
    }
}